%* lsaf_getprocessflowsbyquery                                                                     *;
%*                                                                                                 *;
%* Gets a list of Process Flows based on query criteria.                                           *;
%*                                                                                                 *;
%* macrotype LSAF                                                                                  *;
%* since 2.5                                                                                       *;
%* exposure external                                                                               *;
%*                                                                                                 *;
%* History                                                                                         *;
%* 2021-03-22  initial coding                                                                      *;
%* 2021-08-12  Remove header documentation                                                         *;
%* 2021-10-13  US11739                                                                             *;
%* 2022-01-07  DE17257 - update length of process flow id                                          *;
%*                                                                                                 *;

%macro lsaf_getprocessflowsbyquery(
  lsaf_status=, 
  lsaf_name=, 
  lsaf_contextpath=, 
  lsaf_processdefinition=, 
  lsaf_owner=, 
  lsaf_lastmodifiedby=, 
  lsaf_fromlastmodifieddate=, 
  lsaf_tolastmodifieddate=, 
  lsaf_creationmethod=, 
  sas_dsname=work.lsafgetprocessflowsbyquery
  ) /des='Get a list of all the process flows in the system as defined by user input.';

  %if "&sas_dsname" EQ "" %then %let sas_dsname=work.lsafgetprocessflowsbyquery;

  %****************************************;
  %* Initialize macro variables           *;
  %****************************************;
  %lsaf_getProductName();

  %global _lsafMsg_
          _lsafRC_
          _lsafPfQueryLimitReached_
  ;

  %local _lsafRowCount_
         logMsgPrefix
         logMsgPrimary
         macroName
  ;

  %let _lsafRC_ =%str(-999);
  %let _lsafMsg_=%str(The SAS Macro did not execute properly.  Unknown err%str(or).);
  %let _lsafRowCount_=0;
  %let macroName=%sysfunc(lowcase(&SYSMACRONAME));
  
  %let logMsgPrimary=%str(ERR%str(OR): &_LSAF_PRODUCT_NAME_ Macro: );
  %let logMsgPrefix=%str(NOTE: &_LSAF_PRODUCT_NAME_ Macro: );
 
  %**********************************************;
  %* Start the datastep to call the java code   *;
  %**********************************************;
  data &sas_dsname (label="Created by &SYSMACRONAME"
                    keep= processFlowId processFlowName processFlowStatus creationMethod owner processDefinitionId
                          lastModifiedBy lastModifiedDate lastModifiedDateSASFormat path
    );

    attrib returnCode           length=8
           message              length=$200
           logMsgPrefix         length=$70
           logMsgPrimary        length=$70
           logMsg               length=$300
           rowCount             length=8
           isLimitReached       length=3

           processFlowId             length=$64    label="Process Flow ID"
           processFlowName           length=$256   label="Process Flow Name"
           processFlowStatus         length=$32    label="Process Flow Status"
           creationMethod            length=$32    label="The Creation Method of the Process Flow"
           owner                     length=$32    label="The Owner of the Process Flow"
           processDefinitionId       length=$32    label="The ID of the Process Definition used to create the Process Flow"
           lastModifiedBy            length=$64    label="The User ID that last Modified the Process Flow"
           lastModifiedDate          length=$64    label="Date Last Modified"
           lastModifiedDateSASFormat length=$64    label="Date Last Modified (SAS Format)"
           path                      length=$2048  label="The Path of the Context in which the Process Flow is Located"
    ;

    rowCount=0;
    isLimitReached=0;
      
    %***********************************;
    %* Declare the java objects        *;
    %***********************************;
    declare javaobj srv("com/sas/lsaf/macro/workflow/processflow/ProcessFlowService");
    declare javaobj results("com/sas/lsaf/macro/workflow/processflow/SasProcessFlowSearchResult");

    %***********************************;
    %* Attempt the operation           *;
    %***********************************;
    srv.callStaticBooleanMethod("search", "%bquote(&lsaf_status)", "%bquote(&lsaf_name)", "%bquote(&lsaf_contextpath)",  
      "%bquote(&lsaf_processdefinition)", "%bquote(&lsaf_owner)", "%bquote(&lsaf_lastmodifiedby)", "%bquote(&lsaf_fromlastmodifieddate)", 
      "%bquote(&lsaf_tolastmodifieddate)", "%bquote(&lsaf_creationmethod)", results, isLimitReached);      

    %***********************************;
    %* Retrieve the results            *;
    %***********************************;
    results.callIntMethod("getReturnCode", returnCode);
    results.callStringMethod("getReturnMessage", message);

    %***********************************;
    %* Process the results             *;
    %***********************************;
    logMsgPrefix="NOTE: &_LSAF_PRODUCT_NAME_ Macro: * ";
    if (returnCode ne 0) then
    do;
      logMsgPrimary="ERR%str(OR): &_LSAF_PRODUCT_NAME_ Macro:";
      logMsg=" Cannot retrieve list of queried process flows.";
    end;
    else
    do;
      logMsgPrimary=logMsgPrefix;
      logMsg=" List of all the queried process flows was returned.";
      results.callIntMethod("getSize", rowCount);
      do row=0 to rowCount - 1;
        results.callStringMethod("getId", row, processFlowId);
        results.callStringMethod("getName", row, processFlowName);
        results.callStringMethod("getStatus", row, processFlowStatus);
        results.callStringMethod("getOwner", row, owner);
        results.callStringMethod("getProcessDefinitionKey", row, processDefinitionId);
        results.callStringMethod("getLastModifiedBy", row, lastModifiedBy);
        results.callStringMethod("getLastModified", row, lastModifiedDate);
        results.callStringMethod("getLastModifiedSasFormat", row, lastModifiedDateSASFormat);
        results.callStringMethod("getContextPath", row, path);
        results.callStringMethod("getCreationMethod", row, creationMethod);
        output;
      end;
    end;

    %***********************************;
    %* Print messages in log           *;
    %***********************************;
    put;
    put logMsgPrimary " " logMsg;
    put logMsgPrefix " " %sysfunc(lowcase("&SYSMACRONAME"));
    if (rowCount=0) 
      then put logMsgPrefix " No queried process flows were found.";
      else put logMsgPrefix " " rowCount " records written to data set &sas_dsname";
    put logMsgPrefix " _lsafMsg_= " message ;
    put logMsgPrefix " _lsafRC_= " returnCode ;
    put logMsgPrefix " Process Flow Limit Reached= " isLimitReached;
    put;

    %***********************************;
    %* Set the macro variables         *;
    %***********************************;
    call symputx("_lsafRC_", returnCode);
    call symputx("_lsafMsg_", message);
    call symputx("_lsafRowCount_", rowCount);
    call symputx("_lsafPfQueryLimitReached_", isLimitReached);
  run;

  %***********************************;
  %* Sort the data set               *;
  %***********************************;
  %if (&_lsafRowCount_ > 0) %then
  %do;
    %let notesOption=%sysfunc(getoption(notes));
    options nonotes;
    proc sort data=&sas_dsname sortseq=linguistic(strength=primary);
      by processFlowName;
    run;
    options &notesOption;
  %end;

%mend lsaf_getprocessflowsbyquery;
