%* lsaf_getprocessflowproperties                                                                   *;
%*                                                                                                 *;
%* Gets the properties for the specified process flow. The properties are stored in a SAS data set.*;
%*                                                                                                 *;
%* macrotype LSAF                                                                                  *;
%* since 1.5                                                                                       *;
%* exposure external                                                                               *;
%*                                                                                                 *;
%* History                                                                                         *;
%* 2016-12-05  original creation.                                                                  *;
%* 2019-02-12  update to 2.1                                                                       *;
%* 2019-11-01  update to handle a blank SAS output file.                                           *;
%* 2021-08-12  Remove header documentation                                                         *;
%*                                                                                                 *;

%macro lsaf_getprocessflowproperties(
  lsaf_path=,
  lsaf_processflow=,
  sas_dsname=work.lsafGetProcessFlowProperties
  ) /des='Get a list of process flow properties of a specific process flow in a SAS Life Science Analytics Framework context.';
  
  %if "&sas_dsname" EQ "" %then %let sas_dsname=work.lsafGetProcessFlowProperties;

  %****************************************;
  %* Initialize macro variables           *;
  %****************************************;
  %lsaf_getProductName();

  %global _lsafMsg_
          _lsafRC_
  ;

  %let _lsafRC_ =%str(-999);
  %let _lsafMsg_=%str(The SAS Macro did not execute properly.  Unknown err%str(or).);


  %**********************************************;
  %* Start the datastep to call the java code   *;
  %**********************************************;
  data &sas_dsname (label="Created by &SYSMACRONAME"
                    keep= processFlowPath processFlowName
                          name displayName value type isReadonly isExtended);

    attrib returnCode      length=8
           message         length=$200
           logMsgPrefix    length=$70
           logMsgPrimary   length=$70
           logMsg          length=$300
           rowCount        length=8

           processFlowPath length=$2048  label="Context Path"
           processFlowName length=$256   label="Process Flow Name"
           name            length=$256   label="Property Name"
           displayName     length=$256   label="Property Display Name"
           value           length=$2048  label="Property Value"
           type            length=$32    label="Property Type"
           isReadonly      length=8      label="Property value is not modifiable."
           isExtended      length=8      label="Property is an extended attribute"
    ;

    rowCount=0;

    %***********************************;
    %* Declare the java objects        *;
    %***********************************;
    declare javaobj srv("com/sas/lsaf/macro/workflow/processflow/ProcessFlowService") ;
    declare javaobj results("com/sas/lsaf/macro/workflow/processflow/SasProcessFlowPropertiesResult");

    %***********************************;
    %* Attempt the operation           *;
    %***********************************;
    srv.callStaticVoidMethod("getProperties", "%bquote(&lsaf_path)", "%bquote(&lsaf_processflow)", results) ;

    %***********************************;
    %* Retrieve the results            *;
    %***********************************;
    results.callIntMethod("getReturnCode", returnCode);
    results.callStringMethod("getReturnMessage", message);

    %***********************************;
    %* Process the results             *;
    %***********************************;
    logMsgPrefix="NOTE: &_LSAF_PRODUCT_NAME_ Macro: * ";
    if (returnCode ne 0) then
    do;
      logMsgPrimary="ERR%str(OR): &_LSAF_PRODUCT_NAME_ Macro:";
      logMsg=" Cannot retrieve process flow properties for &lsaf_processflow in &lsaf_path" ;
    end;
    else
    do;
      logMsgPrimary=logMsgPrefix;
      logMsg=" Process flow properties were returned for &lsaf_processflow in &lsaf_path" ;
      results.callIntMethod("getSize", rowCount);
      do row=0 to rowCount - 1;
        results.callStringMethod("getProcessFlowContext", processFlowPath);
        results.callStringMethod("getProcessFlowName", processFlowName);
        results.callStringMethod("getPropertyName", row, name);
        results.callStringMethod("getPropertyValue", row, value);
        results.callStringMethod("getPropertyDisplayName", row, displayName);
        results.callStringMethod("getPropertyType", row, type);
        results.callBooleanMethod("getPropertyIsReadonly", row, isReadonly);
        results.callBooleanMethod("getPropertyIsExtended", row, isExtended);
        output;
      end;
    end;

    %***********************************;
    %* Print messages in log           *;
    %***********************************;
    put;
    put logMsgPrimary " " logMsg;
    put logMsgPrefix " " %sysfunc(lowcase("&SYSMACRONAME"));
    if (rowCount=0) 
      then put logMsgPrefix " No process flow properties were found.";
      else put logMsgPrefix " " rowCount " records written to data set &sas_dsname";
    put logMsgPrefix " _lsafMsg_= " message ;
    put logMsgPrefix " _lsafRC_= " returnCode ;
    put;

    %***********************************;
    %* Set the macro variables         *;
    %***********************************;
    call symputx("_lsafRC_", returnCode);
    call symputx("_lsafMsg_", message);

  run;

%mend lsaf_getprocessflowproperties;
