%* lsaf_getprocessdefsbytype                                                                       *;
%*                                                                                                 *;
%* Gets the metadata for all of the process flow definitions that are available at the specified   *;
%* context type. The metadata is stored in a SAS data set.                                         *;
%*                                                                                                 *;
%* macrotype LSAF                                                                                  *;
%* since 1.5                                                                                       *;
%* exposure external                                                                               *;
%*                                                                                                 *;
%* History                                                                                         *;
%* 2016-08-23  original creation.                                                                  *;
%* 2019-02-12  update to 2.1                                                                       *;
%* 2019-11-01  update to handle a blank SAS output file.                                           *;
%* 2021-06-07  update of contexts variable attributes.                                             *;
%* 2021-08-10  Remove header documentation                                                         *;
%* 2021-10-13  US11739                                                                             *;
%*                                                                                                 *;

%macro lsaf_getprocessdefsbytype(
  lsaf_typeid=,
  sas_dsname=work.lsafGetProcessDefsByType
  ) /des='Get a list of process flow definitions deployed at the specified context type.';
  
  %if "&sas_dsname" EQ "" %then %let sas_dsname=work.lsafGetProcessDefsByType;

  %****************************************;
  %* Initialize macro variables           *;
  %****************************************;
  %lsaf_getProductName();

  %global _lsafMsg_
          _lsafRC_
  ;

  %let _lsafRC_ =%str(-999);
  %let _lsafMsg_=%str(The SAS Macro did not execute properly.  Unknown err%str(or).);

  %local _lsafRowCount_;

  %let _lsafRowCount_=0;

  %**********************************************;
  %* Start the datastep to call the java code   *;
  %**********************************************;
  data &sas_dsname (label="Created by &SYSMACRONAME"
                    keep=contexts defId name description deployedVersion sourceFilePath sourceFileVersion status comment)
  ;
    attrib returnCode          length=8
           message             length=$200
           logMsgPrefix        length=$70
           logMsgPrimary       length=$70
           logMsg              length=$300
           rowCount            length=8

           defID               length=$256   label="Process Flow Definition ID"
           name                length=$256   label="Process Flow Definition Name"
           description         length=$1024  label="Process Flow Definition Description"
           deployedVersion     length=$32    label="Process Flow Definition Version"
           sourceFilePath      length=$2048  label="Source file path"
           sourceFileVersion   length=$32    label="Source file version number"
           contexts            length=$2048  label="Contexts where definition is deployed"
           status	       length=$64    label="Status"
           comment             length=$1024  label="Comment"
    ;

    rowCount=0;

    %***********************************;
    %* Declare the java objects        *;
    %***********************************;
    declare javaobj srv("com/sas/lsaf/macro/workflow/definition/ProcessDefinitionService") ;
    declare javaobj results("com/sas/lsaf/macro/workflow/definition/SasProcessDefinitionInfosResult");

    %***********************************;
    %* Attempt the operation           *;
    %***********************************;
    srv.callStaticVoidMethod("getProcessDefinitionsByType", "%bquote(&lsaf_typeid)", results) ;

    %***********************************;
    %* Retrieve the results            *;
    %***********************************;
    results.callIntMethod("getReturnCode", returnCode);
    results.callStringMethod("getReturnMessage", message);

    %***********************************;
    %* Process the results             *;
    %***********************************;
    logMsgPrefix="NOTE: &_LSAF_PRODUCT_NAME_ Macro: * ";
    if (returnCode ne 0) then
    do;
      logMsgPrimary="ERR%str(OR): &_LSAF_PRODUCT_NAME_ Macro:";
      logMsg=" Cannot retrieve process flow definitions for type &lsaf_typeid" ;
    end;
    else
    do;
      logMsgPrimary=logMsgPrefix;
      logMsg=" Process flow definitions were returned for &lsaf_typeid" ;
      results.callIntMethod("getSize", rowCount);
      do row=0 to rowCount - 1;
        results.callStringMethod("getId", row, defID);
        results.callStringMethod("getName", row, name);
        results.callStringMethod("getDescription", row, description);
        results.callStringMethod("getDeployedVersion", row, deployedVersion);
        results.callStringMethod("getDefinitionFilePath", row, sourceFilePath);
        results.callStringMethod("getDefinitionFileVersion", row, sourceFileVersion);
        results.callStringMethod("getContextTypes", row, contexts);
        results.callStringMethod("getStatus", row, status);
        results.callStringMethod("getComment", row, comment);
        output;
      end;
    end;

    %***********************************;
    %* Print messages in log           *;
    %***********************************;
    put;
    put logMsgPrimary " " logMsg;
    put logMsgPrefix " " %sysfunc(lowcase("&SYSMACRONAME"));
    if (rowCount=0) 
      then put logMsgPrefix " No process flow definitions were found.";
      else put logMsgPrefix " " rowCount " records written to data set &sas_dsname";
    put logMsgPrefix " _lsafMsg_= " message ;
    put logMsgPrefix " _lsafRC_= " returnCode ;
    put;

    %***********************************;
    %* Set the macro variables         *;
    %***********************************;
    call symputx("_lsafRC_", returnCode);
    call symputx("_lsafMsg_", message);
    call symputx("_lsafRowCount_", rowCount);

  run;

  %***********************************;
  %* Sort by name                    *;
  %***********************************;
  %if (&_lsafRowCount_ > 0) %then
  %do;
    %let notesOption=%sysfunc(getoption(notes));
    options nonotes;
    proc sort data=&sas_dsname sortseq=linguistic(strength=primary);
      by defID;
    run;
    options &notesOption;
  %end;

%mend lsaf_getprocessdefsbytype;
