%* lsaf_getpotentialmembers                                                                      *;
%*                                                                                               *;
%* Gets the metadata for all of the users or groups that are available to be added as members    *;
%* to the specified context. The metadata is stored in a SAS data set.                           *;
%*                                                                                               *;
%* macrotype LSAF                                                                                *;
%* since 2.3                                                                                     *;
%* exposure external                                                                             *;
%*                                                                                               *;
%* History                                                                                       *;
%* 2020-02-10  initial coding                                                                    *;
%* 2020-12-04  updates for service refactor                                                      *;
%* 2021-08-09  Remove header documentation                                                       *;
%* 2021-10-13  US11739                                                                           *;
%* 2021-10-19  principal refactor                                                                *;
%*                                                                                               *;

%macro lsaf_getpotentialmembers(
  lsaf_path=,
  sas_dsname=work.lsafGetPotentialMembers
  ) /des='Get a list of the potential members in an SAS Life Science Analytics Framework context.';
  
  %if "&sas_dsname" EQ "" %then %let sas_dsname=work.lsafGetPotentialMembers;

  %****************************************;
  %* Initialize macro variables           *;
  %****************************************;
  %lsaf_getProductName();

  %global _lsafMsg_
          _lsafRC_
  ;

  %let _lsafRC_ =%str(-999);
  %let _lsafMsg_=%str(The SAS Macro did not execute properly.  Unknown err%str(or).);

  %local _lsafRowCount_;

  %let _lsafRowCount_=0;

  %**********************************************;
  %*  Start the datastep to call the java code  *;
  %**********************************************;
  data &sas_dsname (
          label="Created by &SYSMACRONAME"
          keep=contextPath type principalId description grpSrcCtxt
          );
    attrib returnCode      length=8;
    attrib message         length=$200;
    attrib logMsgPrefix    length=$70;
    attrib logMsgPrimary   length=$70;
    attrib logMsg          length=$300;
    attrib rowCount        length=8;

    attrib contextPath     length=$2048     label="Context Path" ;
    attrib type            length=$32       label="Principal Type";
    attrib principalId     length=$256      label="Principal Name";
    attrib description     length=$128      label="Principal Description";
    attrib grpSrcCtxt      length=$2048     label="Principal Group Context Source Path";

    rowCount=0;

    %***********************************;
    %* Declare the java objects        *;
    %***********************************;
    declare javaobj srv("com/sas/lsaf/macro/security/membership/MembershipService");
    declare javaobj results("com/sas/lsaf/macro/security/SasPrincipalsResult");

    %***********************************;
    %* Attempt the operation           *;
    %***********************************;
    srv.callStaticVoidMethod("getPotentialMembers", "%bquote(&lsaf_path)", results);

    %***********************************;
    %* Retrieve the results            *;
    %***********************************;
    results.callIntMethod("getReturnCode", returnCode);
    results.callStringMethod("getReturnMessage", message);

    %***********************************;
    %* Process the results             *;
    %***********************************;
    logMsgPrefix="NOTE: &_LSAF_PRODUCT_NAME_ Macro: * ";
    if (returnCode ne 0) then
    do;
      logMsgPrimary="ERR%str(OR): &_LSAF_PRODUCT_NAME_ Macro:";
      logMsg=" Cannot retrieve potential members for &lsaf_path" ;
    end;
    else
    do;
      logMsgPrimary=logMsgPrefix;
      logMsg=" Potential members were returned for &lsaf_path" ;
      results.callIntMethod("getSize", rowCount);
      do row=0 to rowCount - 1;
        contextPath="&lsaf_path";
        results.callStringMethod("getType", row, type);
        results.callStringMethod("getName", row, principalId);
        results.callStringMethod("getDescription", row, description);
        results.callStringMethod("getContextPath", row, grpSrcCtxt);
        output;
      end;
    end;

    %***********************************;
    %* Print messages in log           *;
    %***********************************;
    put;
    put logMsgPrimary " " logMsg;
    put logMsgPrefix " " %sysfunc(lowcase("&SYSMACRONAME"));
    if (rowCount=0) 
      then put logMsgPrefix " No members were found.";
      else put logMsgPrefix " " rowCount " records written to data set &sas_dsname";
    put logMsgPrefix " _lsafMsg_= " message ;
    put logMsgPrefix " _lsafRC_= " returnCode ;
    put;

    %***********************************;
    %* Set the macro variables         *;
    %***********************************;
    call symputx("_lsafRC_", returnCode);
    call symputx("_lsafMsg_", message);
    call symputx("_lsafRowCount_", rowCount);

  run;

  %***********************************;
  %* Sort the data set               *;
  %***********************************;
  %if (&_lsafRowCount_ > 0) %then
  %do;
   %let notesOption=%sysfunc(getoption(notes));
    options nonotes;
    proc sort data=&sas_dsname sortseq=linguistic(strength=primary);
      by type principalId;
    run;
    options &notesOption;
  %end;

%mend lsaf_getpotentialmembers;
