%* lsaf_getpfusertaskproperties                                                                     *;
%*                                                                                                  *;
%* Get a list of the process flow user task properties.                                             *;
%*                                                                                                  *;
%* macrotype LSAF                                                                                   *;
%* since 2.5                                                                                        *;
%* exposure external                                                                                *;
%*                                                                                                  *;
%* History                                                                                          *;
%* 2021-02-23  original creation.                                                                   *;
%* 2021-08-12  Remove header documentation                                                          *;
%* 2023-03-09  DE17654 - SAS Notes option updates                                                   *;
%*                                                                                                  *;

%macro lsaf_getpfusertaskproperties(
  lsaf_taskid=,
  sas_dsname=work.lsafgetpfusertaskproperties
  ) /des='Get a list of the process flow user task properties in a SAS Life Science Analytics Framework context.';
  
  %if "&sas_dsname" EQ "" %then %let sas_dsname=work.lsafgetpfusertaskproperties;

  %****************************************;
  %* Initialize macro variables           *;
  %****************************************;
  %lsaf_getProductName();

  %global _lsafMsg_
          _lsafRC_
  ;

  %let _lsafRC_ =%str(-999);
  %let _lsafMsg_=%str(The SAS Macro did not execute properly.  Unknown err%str(or).);

  %**********************************************;
  %* Start the datastep to call the java code   *;
  %**********************************************;
  data &sas_dsname (label="Created by &SYSMACRONAME"
                    keep=taskId propertyName propertyDisplayName propertyType propertyValue isReadOnly isExtended isRequired);

    attrib returnCode       length=8
           message          length=$200
           logMsgPrefix     length=$70
           logMsgPrimary    length=$70
           logMsg           length=$300
           rowCount         length=8

           taskId              length=$64   label="Task Identifier"
           propertyName        length=$256  label="Property Name"
           propertyDisplayName length=$256  label="Property Display Name"
           propertyType        length=$64   label="Property Type"
           propertyValue       length=$64   label="Propert Value"
           isReadOnly          length=8     label="Property value is not modifiable"
           isExtended          length=8     label="Property is an extended attribute"
           isRequired          length=8     label="Property value is required" 
    ;

    rowCount=0;

    %***********************************;
    %* Declare the java objects        *;
    %***********************************;
    declare javaobj srv("com/sas/lsaf/macro/workflow/task/ProcessFlowTaskService") ;
    declare javaobj results("com/sas/lsaf/macro/common/result/SasObjectPropertiesResult");

    %***********************************;
    %* Attempt the operation           *;
    %***********************************;
    srv.callStaticVoidMethod("getUserTaskProperties", "%bquote(&lsaf_taskid)", results) ;

    %***********************************;
    %* Retrieve the results            *;
    %***********************************;
    results.callIntMethod("getReturnCode", returnCode);
    results.callStringMethod("getReturnMessage", message);

    %***********************************;
    %* Process the results             *;
    %***********************************;
    logMsgPrefix="NOTE: &_LSAF_PRODUCT_NAME_ Macro: * ";
    if (returnCode ne 0) then
    do;
      logMsgPrimary="ERR%str(OR): &_LSAF_PRODUCT_NAME_ Macro:";
      logMsg=" Cannot retrieve user task properties for task &lsaf_taskid.." ;
    end;
    else
    do;
      logMsgPrimary=logMsgPrefix;
      logMsg=" User task properties were returned for task &lsaf_taskid.." ;
      results.callIntMethod("getSize", rowCount);
      taskId="&lsaf_taskid";
      do row=0 to rowCount - 1;
        results.callStringMethod("getPropertyName", row, propertyName);
        results.callStringMethod("getPropertyDisplayName", row, propertyDisplayName);
        results.callStringMethod("getPropertyType", row, propertyType);
        results.callStringMethod("getPropertyValue", row, propertyValue);    
        results.callBooleanMethod("getPropertyIsReadonly", row, isReadOnly);
        results.callBooleanMethod("getPropertyIsRequired", row, isRequired);
        results.callBooleanMethod("getPropertyIsExtended", row, isExtended);
        output;
      end;
    end;

    %***********************************;
    %* Print messages in log           *;
    %***********************************;
    put;
    put logMsgPrimary " " logMsg;
    put logMsgPrefix " " %sysfunc(lowcase("&SYSMACRONAME"));
    if (rowCount=0) 
      then put logMsgPrefix " No user task properties were found.";
      else put logMsgPrefix " " rowCount " records written to data set &sas_dsname";
    put logMsgPrefix " _lsafMsg_= " message ;
    put logMsgPrefix " _lsafRC_= " returnCode ;
    put;

    %***********************************;
    %* Set the macro variables         *;
    %***********************************;
    call symputx("_lsafRC_", returnCode);
    call symputx("_lsafMsg_", message);

  run;

%mend lsaf_getpfusertaskproperties;
