%* lsaf_getpfusertaskcandidates                                                                     *;
%*                                                                                                  *;
%* Get a list of the process flow user task candidates.                                             *;
%*                                                                                                  *;
%* macrotype LSAF                                                                                   *;
%* since 2.5                                                                                        *;
%* exposure external                                                                                *;
%*                                                                                                  *;
%* History                                                                                          *;
%* 2021-02-23  original creation.                                                                   *;
%* 2021-08-12  Remove header documentation                                                          *;
%* 2021-10-19  principal refactor                                                                   *;
%* 2023-03-09  DE17654 - SAS Notes option updates                                                   *;
%*                                                                                                  *;

%macro lsaf_getpfusertaskcandidates(
  lsaf_taskid=,
  sas_dsname=work.lsafgetpfusertaskcandidates
  ) /des='Get a list of the process flow user task candidates in a SAS Life Science Analytics Framework context.';
  
  %if "&sas_dsname" EQ "" %then %let sas_dsname=work.lsafgetpfusertaskcandidates;

  %****************************************;
  %* Initialize macro variables           *;
  %****************************************;
  %lsaf_getProductName();

  %global _lsafMsg_
          _lsafRC_
  ;

  %let _lsafRC_ =%str(-999);
  %let _lsafMsg_=%str(The SAS Macro did not execute properly.  Unknown err%str(or).);

  %**********************************************;
  %* Start the datastep to call the java code   *;
  %**********************************************;
  data &sas_dsname (label="Created by &SYSMACRONAME"
                    keep=taskId candidateType candidateId description grpSrcCtxt);

    attrib returnCode       length=8
           message          length=$200
           logMsgPrefix     length=$70
           logMsgPrimary    length=$70
           logMsg           length=$300
           rowCount         length=8

           taskId           length=$64   label="Task Identifier"
           candidateType    length=$32   label="Candidate Type"
           candidateId      length=$256  label="Candidate Name"
           description      length=$128  label="Candidate Description"
           grpSrcCtxt       length=$2048 label="Candidate Group Context Source Path"

    ;

    rowCount=0;
    
    %***********************************;
    %* Declare the java objects        *;
    %***********************************;
    declare javaobj srv("com/sas/lsaf/macro/workflow/task/ProcessFlowTaskService") ;
    declare javaobj results("com/sas/lsaf/macro/security/SasPrincipalsResult");

    %***********************************;
    %* Attempt the operation           *;
    %***********************************;
    srv.callStaticVoidMethod("getUserTaskCandidates", "%bquote(&lsaf_taskid)", results) ;

    %***********************************;
    %* Retrieve the results            *;
    %***********************************;
    results.callIntMethod("getReturnCode", returnCode);
    results.callStringMethod("getReturnMessage", message);

    %***********************************;
    %* Process the results             *;
    %***********************************;
    logMsgPrefix="NOTE: &_LSAF_PRODUCT_NAME_ Macro: * ";
    if (returnCode ne 0) then
    do;
      logMsgPrimary="ERR%str(OR): &_LSAF_PRODUCT_NAME_ Macro:";
      logMsg=" Cannot retrieve user task candidates for task &lsaf_taskid.." ;
    end;
    else
    do;
      logMsgPrimary=logMsgPrefix;
      logMsg=" User task candidates ran successfully for task &lsaf_taskid.." ;
      results.callIntMethod("getSize", rowCount);
      taskId="&lsaf_taskid";
      do row=0 to rowCount - 1;
        results.callStringMethod("getType", row, candidateType);
        results.callStringMethod("getName", row, candidateId);
        results.callStringMethod("getDescription", row, description);
        results.callStringMethod("getContextPath", row, grpSrcCtxt);
        output;
      end;
    end;

    %***********************************;
    %* Print messages in log           *;
    %***********************************;
    put;
    put logMsgPrimary " " logMsg;
    put logMsgPrefix " " %sysfunc(lowcase("&SYSMACRONAME"));
    if (rowCount=0) 
      then put logMsgPrefix " No user task candidates were found.";
      else put logMsgPrefix " " rowCount " records written to data set &sas_dsname";
    put logMsgPrefix " _lsafMsg_= " message ;
    put logMsgPrefix " _lsafRC_= " returnCode ;
    put;

    %***********************************;
    %* Set the macro variables         *;
    %***********************************;
    call symputx("_lsafRC_", returnCode);
    call symputx("_lsafMsg_", message);

  run;

%mend lsaf_getpfusertaskcandidates;
