%* lsaf_getpftasksbyquery                                                                         *;
%*                                                                                                *;
%* Gets a list of UserTask objects based on query criteria.                                       *;
%*                                                                                                *;
%* macrotype LSAF                                                                                 *;
%* since 2.5                                                                                      *;
%* exposure external                                                                              *;
%*                                                                                                *;
%* History                                                                                        *;
%* 2021-01-15  initial coding                                                                     *;
%* 2021-08-12  Remove header documentation                                                        *;
%* 2021-09-20  Remove parameter defaults (DE16775)                                                *;
%* 2021-12-02  2.6 Added tasklimit parameter & attrib fix                                         *;
%* 2024-07-05  2.7 DE17673 increase length of id variables                                        *;
%*                                                                                                *;

%macro lsaf_getpftasksbyquery(
  lsaf_status=, 
  lsaf_elementid=, 
  lsaf_elementname=, 
  lsaf_processflowname=, 
  lsaf_contextPath=, 
  lsaf_assigneetype=, 
  lsaf_assignees=, 
  lsaf_fromduedate=, 
  lsaf_toduedate=, 
  lsaf_priority=,
  lsaf_tasklimit=,
  sas_result_dsname=work.lsafgetpftasksbyquery
  ) /des='Get a list of all the process flow tasks in the system as defined by user input.';
  
  %if "&sas_result_dsname" EQ "" %then %let sas_result_dsname=work.lsafgetpftasksbyquery;

  %****************************************;
  %* Initialize macro variables           *;
  %****************************************;
  %lsaf_getProductName();

  %global _lsafMsg_
          _lsafRC_
          _lsafPFTaskQueryLimitReached_
  ;

  %local _lsafRowCount_
         logMsgPrefix
         logMsgPrimary
         macroName
  ;

  %let _lsafRC_ =%str(-999);
  %let _lsafMsg_=%str(The SAS Macro did not execute properly.  Unknown err%str(or).);
  %let _lsafRowCount_=0;
  %let macroName=%sysfunc(lowcase(&SYSMACRONAME));
  
  %let logMsgPrimary=%str(ERR%str(OR): &_LSAF_PRODUCT_NAME_ Macro: );
  %let logMsgPrefix=%str(NOTE: &_LSAF_PRODUCT_NAME_ Macro: );
 
 
  %**********************************************;
  %* Start the datastep to call the java code   *;
  %**********************************************;
  data &sas_result_dsname (label="Created by &SYSMACRONAME"
                    keep= processFlowId processFlowStatus elementId elementName elementType taskId taskDescription
                          taskCreated taskStatus assignees priority complexity dateStarted dateStartedSASFormat 
                          dueDate dueDateSASFormat
    );

    attrib returnCode           length=8       
           message              length=$200       
           logMsgPrefix         length=$70       
           logMsgPrimary        length=$70       
           logMsg               length=$300       
           rowCount             length=8       
           isLimitReached       length=3       

           processFlowId        length=$64    label="Process Flow ID"       
           processFlowStatus    length=$32    label="Process Flow Status"       
           elementId            length=$64    label="Element ID"       
           elementName          length=$32    label="Element Name"       
           elementType          length=$64    label="Element Type"       
           taskId               length=$64    label="Task ID"       
           taskDescription      length=$2048  label="Task Description"       
           taskCreated          length=$64    label="Task Created"       
           taskStatus           length=$32    label="Task Status"       
           assignees            length=$32    label="Assignees"       
           priority             length=$32    label="Priority"       
           complexity           length=$32    label="Complexity"       
           dateStarted          length=$64    label="Task Date Started"       
           dateStartedSASFormat length=$64    label="Task Date Started (SAS DateTime Format)"       
           dueDate              length=$64    label="Task Due Date"       
           dueDateSASFormat     length=$64    label="Task Due Date (SAS DateTime Format)"
    ;

    rowCount=0;
    isLimitReached=0;
      
    %***********************************;
    %* Declare the java objects        *;
    %***********************************;
    declare javaobj srv("com/sas/lsaf/macro/workflow/task/ProcessFlowTaskService");
    declare javaobj results("com/sas/lsaf/macro/workflow/task/SasTasksResult");

    %***********************************;
    %* Attempt the operation           *;
    %***********************************;  
    srv.callStaticBooleanMethod("getTasksByQuery", "%bquote(&lsaf_status)", "%bquote(&lsaf_elementid)", "%bquote(&lsaf_elementname)",  
      "%bquote(&lsaf_processflowname)", "%bquote(&lsaf_contextpath)", "%bquote(&lsaf_assigneetype)", "%bquote(&lsaf_assignees)", 
      "%bquote(&lsaf_fromduedate)", "%bquote(&lsaf_toduedate)", "%bquote(&lsaf_priority)", "%bquote(&lsaf_tasklimit)",
      results, isLimitReached);

    %***********************************;
    %* Retrieve the results            *;
    %***********************************;
    results.callIntMethod("getReturnCode", returnCode);
    results.callStringMethod("getReturnMessage", message);

    %***********************************;
    %* Process the results             *;
    %***********************************;
    logMsgPrefix="NOTE: &_LSAF_PRODUCT_NAME_ Macro: * ";
    if (returnCode ne 0) then
    do;
      logMsgPrimary="ERR%str(OR): &_LSAF_PRODUCT_NAME_ Macro:";
      logMsg=" Cannot retrieve list of queried tasks.";
    end;
    else
    do;
      logMsgPrimary=logMsgPrefix;
      logMsg=" List of all the queried tasks was returned.";
      results.callIntMethod("getSize", rowCount);
      do row=0 to rowCount - 1;
        results.callStringMethod("getCreated", row, taskCreated);
        results.callStringMethod("getDescription", row, taskDescription);
        results.callStringMethod("getElementId", row, elementId);
        results.callStringMethod("getTaskId", row, taskId);
        results.callStringMethod("getFlowStatus", row, processFlowStatus);
        results.callStringMethod("getElementName", row, elementName);
        results.callStringMethod("getProcessFlowId", row, processFlowId);
        results.callStringMethod("getElementType", row, elementType);
        results.callStringMethod("getStarted", row, dateStarted);
        results.callStringMethod("getStartedSasFormat", row, dateStartedSasFormat);
        results.callStringMethod("getTaskStatus", row, taskStatus);
        results.callStringMethod("getAssignee", row, assignees);
        results.callStringMethod("getDueDate", row, dueDate);
        results.callStringMethod("getDueDateSasFormat", row, dueDateSasFormat);
        results.callStringMethod("getPriority", row, priority);
        results.callStringMethod("getComplexity", row, complexity);
        output;
      end;
    end;

    %***********************************;
    %* Print messages in log           *;
    %***********************************;
    put;
    put logMsgPrimary " " logMsg;
    put logMsgPrefix " " %sysfunc(lowcase("&SYSMACRONAME"));
    if (rowCount=0) 
      then put logMsgPrefix " No queried tasks were found.";
      else put logMsgPrefix " " rowCount " records written to data set &sas_result_dsname";
    put logMsgPrefix " _lsafMsg_= " message ;
    put logMsgPrefix " _lsafRC_= " returnCode ;
    put logMsgPrefix " Task Limit Reached= " isLimitReached;
    put;

    %***********************************;
    %* Set the macro variables         *;
    %***********************************;
    call symputx("_lsafRC_", returnCode);
    call symputx("_lsafMsg_", message);
    call symputx("_lsafRowCount_", rowCount);
    call symputx("_lsafPFTaskQueryLimitReached_", isLimitReached);
  run;

  %***********************************;
  %* Sort the data set               *;
  %***********************************;
  %if (&_lsafRowCount_ > 0) %then
  %do;
      %let notesOption=%sysfunc(getoption(notes));
      options nonotes;
      proc sort data=&sas_result_dsname;
         by processFlowId dateStartedSasFormat dueDateSasFormat;
      run;
     options &notesOption;
  %end;

%mend lsaf_getpftasksbyquery;
