%* lsaf_getpfsetupusercandidates                                                                   *;
%*                                                                                                 *;
%* Gets the candidates for the user setup elements for the process flow within the specified       *;
%* context. The candidates are stored in a SAS data set.                                           *;
%*                                                                                                 *;
%* macrotype LSAF                                                                                  *;
%* since 1.5                                                                                       *;
%* exposure external                                                                               *;
%*                                                                                                 *;
%* History                                                                                         *;
%* 2016-12-13  original creation.                                                                  *;
%* 2019-02-12  update to 2.1                                                                       *;
%* 2019-03-15  rename from lsaf_getpfusercandidates to lsaf_getpfsetupusercandidates               *;
%* 2019-11-01  update to handle a blank SAS output file.                                           *;
%* 2021-08-12  Remove header documentation                                                         *;
%* 2021-10-13  US11739                                                                             *;
%* 2021-10-19  principal refactor                                                                  *;
%* 2023-03-10  DE17654 - SAS Notes option updates                                                  *;
%*                                                                                                 *;

%macro lsaf_getpfsetupusercandidates(
  lsaf_path=,
  lsaf_processflow=,
  sas_dsname=work.lsafgetpfsetupusercandidates
  ) /des='Get a list of process flow user setup element candidates for the specific process flow.';
  
  %if "&sas_dsname" EQ "" %then %let sas_dsname=work.lsafgetpfsetupusercandidates;

  %****************************************;
  %* Initialize macro variables           *;
  %****************************************;
  %lsaf_getProductName();

  %global _lsafMsg_
          _lsafRC_
  ;
  %local notesOption;

  %let _lsafRC_ =%str(-999);
  %let _lsafMsg_=%str(The SAS Macro did not execute properly.  Unknown err%str(or).);

  %local _lsafRowCount_;
  %let _lsafRowCount_=0;

  %**********************************************;
  %*  Start the datastep to call the java code  *;
  %**********************************************;
  data &sas_dsname (label="Created by &SYSMACRONAME"
                    keep=processFlowPath processFlowName elementID
                         type principalId description grpSrcCtxt
          );
    attrib returnCode      length=8
           message         length=$200
           logMsgPrefix    length=$70
           logMsgPrimary   length=$70
           logMsg          length=$300
           rowCount        length=8

           processFlowPath length=$2048  label="Context Path"
           processFlowName length=$256   label="Process Flow Name"
           elementID       length=$256   label="Element ID"
           type            length=$32    label="Principal Type"
           principalId     length=$256   label="Principal Name"
           description     length=$128   label="Principal Description"
           grpSrcCtxt      length=$2048  label="Principal Group Context Source Path"
    ;

    rowCount=0;

    %***********************************;
    %* Declare the java objects        *;
    %***********************************;
    declare javaobj srv("com/sas/lsaf/macro/workflow/setup/ProcessFlowSetupService");
    declare javaobj results("com/sas/lsaf/macro/workflow/setup/SasProcessFlowElementsPrincipalsResult");

    %***********************************;
    %* Attempt the operation           *;
    %***********************************;
    srv.callStaticVoidMethod("getElementPrincipalsByType", "%bquote(&lsaf_path)", "%bquote(&lsaf_processflow)",
         "%bquote(USER_TASK)", results) ;

    %***********************************;
    %* Retrieve the results            *;
    %***********************************;
    results.callIntMethod("getReturnCode", returnCode);
    results.callStringMethod("getReturnMessage", message);

    %***********************************;
    %* Process the results             *;
    %***********************************;
    logMsgPrefix="NOTE: &_LSAF_PRODUCT_NAME_ Macro: * ";
    if (returnCode ne 0) then
    do;
      logMsgPrimary="ERR%str(OR): &_LSAF_PRODUCT_NAME_ Macro:";
      logMsg=" Cannot retrieve process flow user element candidates for &lsaf_processflow in &lsaf_path" ;
    end;
    else
    do;
      logMsgPrimary=logMsgPrefix;
      logMsg=" Process flow user element candidates were returned for &lsaf_processflow in &lsaf_path" ;
      results.callIntMethod("getSize", rowCount);
	  processFlowPath = "&lsaf_path";
	  processFlowName = "&lsaf_processflow";
      do row=0 to rowCount - 1;
        results.callStringMethod("getElementId", row, elementID);
        results.callStringMethod("getType", row, type);
        results.callStringMethod("getName", row, principalId);
        results.callStringMethod("getGroupContext", row, grpSrcCtxt);
        results.callStringMethod("getDescription", row, description);
        output;
      end;
    end;

    %***********************************;
    %* Print messages in log           *;
    %***********************************;
    put;
    put logMsgPrimary " " logMsg;
    put logMsgPrefix " " %sysfunc(lowcase("&SYSMACRONAME"));
    if (rowCount=0) 
      then put logMsgPrefix " No candidates were found.";
      else put logMsgPrefix " " rowCount " records written to data set &sas_dsname";
    put logMsgPrefix " _lsafMsg_= " message ;
    put logMsgPrefix " _lsafRC_= " returnCode ;
    put;

    %***********************************;
    %* Set the macro variables         *;
    %***********************************;
    call symputx("_lsafRC_", returnCode);
    call symputx("_lsafMsg_", message);
    call symputx("_lsafRowCount_", rowCount);

  run;

  %***********************************;
  %* Sort the data set               *;
  %***********************************;
  %if (&_lsafRowCount_ > 0) %then
  %do;
    %let notesOption=%sysfunc(getoption(notes));
    options nonotes;
    proc sort data=&sas_dsname sortseq=linguistic(strength=primary);
      by elementID type principalId;
    run;
    options &notesOption.;
  %end;

%mend lsaf_getpfsetupusercandidates;
