%* lsaf_getpfsetuptimers                                                                           *;
%*                                                                                                 *;
%* Gets the data for the timer setup elements for the process flow within the specified context.   *;
%* The data is stored in a SAS data set.                                                           *;
%*                                                                                                 *;
%* macrotype LSAF                                                                                  *;
%* since 1.5                                                                                       *;
%* exposure external                                                                               *;
%*                                                                                                 *;
%* History                                                                                         *;
%* 2016-12-10  original creation.                                                                  *;
%* 2019-02-12  update to 2.1                                                                       *;
%* 2019-03-15  rename from lsaf_getpftimerelements to lsaf_getpfsetuptimers                        *;
%* 2019-11-01  update to handle a blank SAS output file.                                           *;
%* 2021-08-12  Remove header documentation                                                         *;
%* 2023-03-09  DE17654 - SAS Notes option updates                                                  *;
%*                                                                                                 *;

%macro lsaf_getpfsetuptimers(
  lsaf_path=,
  lsaf_processflow=,
  sas_dsname=work.lsafgetpfsetuptimers
  ) /des='Get a list of process flow timer setup element properties for the specific process flow.';
  
  %if "&sas_dsname" EQ "" %then %let sas_dsname=work.lsafgetpfsetuptimers;

  %****************************************;
  %* Initialize macro variables           *;
  %****************************************;
  %lsaf_getProductName();

  %global _lsafMsg_
          _lsafRC_
  ;

  %let _lsafRC_ =%str(-999);
  %let _lsafMsg_=%str(The SAS Macro did not execute properly.  Unknown err%str(or).);

  %**********************************************;
  %* Start the datastep to call the java code   *;
  %**********************************************;
  data &sas_dsname (label="Created by &SYSMACRONAME"
                    keep= processFlowPath processFlowName elementID name type description
                          value years months days hours minutes seconds);

    attrib returnCode          length=8
           message             length=$200
           logMsgPrefix        length=$70
           logMsgPrimary       length=$70
           logMsg              length=$300
           rowCount            length=8

           processFlowPath     length=$2048  label="Context Path"
           processFlowName     length=$256   label="Process Flow Name"
           elementID           length=$256   label="Element ID"
           description         length=$1024  label="Description"
           name                length=$256   label="Name"
           type                length=$32    label="Timer Type"
           value               length=$32    label="Timer Value"
           years               length=8      label="Years value for duration timer"
           months              length=8      label="Months value for duration timer"
           days                length=8      label="Days value for duration timer"
           hours               length=8      label="Hours value for duration timer"
           minutes             length=8      label="Minutes value for duration timer"
           seconds             length=8      label="Seconds value for duration timer"
    ;
    rowCount=0;

    %***********************************;
    %* Declare the java objects        *;
    %***********************************;
    declare javaobj srv("com/sas/lsaf/macro/workflow/setup/ProcessFlowSetupService") ;
    declare javaobj results("com/sas/lsaf/macro/workflow/setup/SasProcessFlowTimerElementResult");

    %***********************************;
    %* Attempt the operation           *;
    %***********************************;
    srv.callStaticVoidMethod("getTimerElements", "%bquote(&lsaf_path)", "%bquote(&lsaf_processflow)", results) ;

    %***********************************;
    %* Retrieve the results            *;
    %***********************************;
    results.callIntMethod("getReturnCode", returnCode);
    results.callStringMethod("getReturnMessage", message);

    %***********************************;
    %* Process the results             *;
    %***********************************;
    logMsgPrefix="NOTE: &_LSAF_PRODUCT_NAME_ Macro: * ";
    if (returnCode ne 0) then
    do;
      logMsgPrimary="ERR%str(OR): &_LSAF_PRODUCT_NAME_ Macro:";
      logMsg=" Cannot retrieve process flow timer element properties for &lsaf_processflow in &lsaf_path" ;
    end;
    else
    do;
      logMsgPrimary=logMsgPrefix;
      logMsg=" Process flow timer element properties were returned for &lsaf_processflow in &lsaf_path" ;
      results.callIntMethod("getSize", rowCount);
      do row=0 to rowCount - 1;
        results.callStringMethod("getProcessFlowContext", processFlowPath);
        results.callStringMethod("getProcessFlowName", processFlowName);
        results.callStringMethod("getElementID", row, elementID);
        results.callStringMethod("getName", row, name);
        results.callStringMethod("getDescription", row, description);
        results.callStringMethod("getTimerType", row, type);
        results.callStringMethod("getTimerValue", row, value);
        if (type EQ "DURATION") then
        do;
           results.callIntMethod("getYears", row, years);
           results.callIntMethod("getMonths", row, months);
           results.callIntMethod("getDays", row, days);
           results.callIntMethod("getHours", row, hours);
           results.callIntMethod("getMinutes", row, minutes);
           results.callIntMethod("getSeconds", row, seconds);
        end;
        else
        do;
           years=.;
           months=.;
           days=.;
           hours=.;
           minutes=.;
           seconds=.;
        end;
        output;
      end;
    end;

    %***********************************;
    %* Print messages in log           *;
    %***********************************;
    put;
    put logMsgPrimary " " logMsg;
    put logMsgPrefix " " %sysfunc(lowcase("&SYSMACRONAME"));
    if (rowCount=0) 
      then put logMsgPrefix " No process flow timer elements properties were found.";
      else put logMsgPrefix " " rowCount " records written to data set &sas_dsname";
    put logMsgPrefix " _lsafMsg_= " message ;
    put logMsgPrefix " _lsafRC_= " returnCode ;
    put;

    %***********************************;
    %* Set the macro variables         *;
    %***********************************;
    call symputx("_lsafRC_", returnCode);
    call symputx("_lsafMsg_", message);

  run;

%mend lsaf_getpfsetuptimers;
