%* lsaf_getpfsetupjobparameters                                                                     *;
%*                                                                                                  *;
%* Gets the override parameters for the job setup elements for the process flow within the          *;
%* specified context. The parameters are stored in a SAS data set.                                  *;
%*                                                                                                  *;
%* macrotype LSAF                                                                                   *;
%* since 1.5                                                                                        *;
%* exposure external                                                                                *;
%*                                                                                                  *;
%* History                                                                                          *;
%* 2016-12-14  original creation.                                                                   *;
%* 2019-02-12  update to 2.1                                                                        *;
%* 2019-03-15  rename from lsaf_getpfjobparameters to lsaf_getpfsetupjobparameters                  *;
%* 2019-11-01  update to handle a blank SAS output file.                                            *;
%* 2021-08-12  Remove header documentation                                                          *;
%* 2021-10-13  US11739                                                                              *;
%* 2023-03-09  DE17654 - SAS Notes option updates                                                   *;
%*                                                                                                  *;

%macro lsaf_getpfsetupjobparameters(
  lsaf_path=,
  lsaf_processflow=,
  sas_dsname=work.lsafGetPfSetupJobParameters
  ) /des='Get a list of process flow job setup element override parameters for the specific process flow.';
  
  %if "&sas_dsname" EQ "" %then %let sas_dsname=work.lsafGetPfSetupJobParameters;

  %****************************************;
  %* Initialize macro variables           *;
  %****************************************;
  %lsaf_getProductName();

  %global _lsafMsg_
          _lsafRC_
  ;

  %let _lsafRC_ =%str(-999);
  %let _lsafMsg_=%str(The SAS Macro did not execute properly.  Unknown err%str(or).);

  %local _lsafRowCount_
         notesOption;

  %let _lsafRowCount_=0;
  %let notesOption=%sysfunc(getoption(notes));

  %**********************************************;
  %* Start the datastep to call the java code   *;
  %**********************************************;
  data &sas_dsname (label="Created by &SYSMACRONAME"
                    keep= processFlowPath processFlowName elementID
                          jobPath jobVersion name type value fileVersion includeSubFolders);

    attrib returnCode        length=8
           message           length=$200
           logMsgPrefix      length=$70
           logMsgPrimary     length=$70
           logMsg            length=$300
           rowCount          length=8
           processFlowPath   length=$2048  label="Context Path"
           processFlowName   length=$256   label="Process Flow Name"
           elementID         length=$256   label="Element ID"
           jobPath           length=$2048  label="Job Path"
           jobVersion        length=$32    label="Job Version"
           name              length=$256   label="Parameter Name"
           type              length=$32    label="Parameter Type"
           value             length=$2048  label="Override Value"
           fileVersion       length=$32    label="File Version if File Type"
           includeSubFolders length=8      label="Include Container SubFolders"
    ;

    rowCount=0;

    %***********************************;
    %* Declare the java objects        *;
    %***********************************;
    declare javaobj     srv("com/sas/lsaf/macro/workflow/setup/ProcessFlowSetupService") ;
    declare javaobj results("com/sas/lsaf/macro/workflow/setup/SasProcessFlowElementsParametersResult");

    %***********************************;
    %* Attempt the operation           *;
    %***********************************;
    srv.callStaticVoidMethod("getJobElementsParameters", "%bquote(&lsaf_path)", "%bquote(&lsaf_processflow)", results) ;

    %***********************************;
    %* Retrieve the results            *;
    %***********************************;
    results.callIntMethod("getReturnCode", returnCode);
    results.callStringMethod("getReturnMessage", message);

    %***********************************;
    %* Process the results             *;
    %***********************************;
    logMsgPrefix="NOTE: &_LSAF_PRODUCT_NAME_ Macro: * ";
    if (returnCode ne 0) then
    do;
      logMsgPrimary="ERR%str(OR): &_LSAF_PRODUCT_NAME_ Macro:";
      logMsg=" Cannot retrieve process flow job element override parameters for &lsaf_processflow in &lsaf_path" ;
    end;
    else
    do;
      logMsgPrimary=logMsgPrefix;
      logMsg=" Process flow job element override parameters were returned for &lsaf_processflow in &lsaf_path" ;
      results.callIntMethod("getSize", rowCount);
      do row=0 to rowCount - 1;
        results.callStringMethod("getProcessFlowContext", processFlowPath);
        results.callStringMethod("getProcessFlowName", processFlowName);
        results.callStringMethod("getElementID", row, elementID);
        results.callStringMethod("getJobPath", row, jobPath);
        results.callStringMethod("getJobVersion", row, jobVersion);
        results.callStringMethod("getParameterName", row, name);
        results.callStringMethod("getParameterType", row, type);
        results.callStringMethod("getParameterValue", row, value);
        results.callStringMethod("getParameterFileVersion", row, fileVersion);
        results.callIntMethod("getParameterIncludeSubfolders", row, includeSubFolders);       
        output;
      end;
    end;

    %***********************************;
    %* Print messages in log           *;
    %***********************************;
    put;
    put logMsgPrimary " " logMsg;
    put logMsgPrefix " " %sysfunc(lowcase("&SYSMACRONAME"));
    if (rowCount=0) 
      then put logMsgPrefix " No process flow job elements override parameters were found.";
      else put logMsgPrefix " " rowCount " records written to data set &sas_dsname";
    put logMsgPrefix " _lsafMsg_= " message ;
    put logMsgPrefix " _lsafRC_= " returnCode ;
    put;

    %***********************************;
    %* Set the macro variables         *;
    %***********************************;
    call symputx("_lsafRC_", returnCode);
    call symputx("_lsafMsg_", message);
    call symputx("_lsafRowCount_", rowCount);

  run;

  %***********************************;
  %* Sort the data set               *;
  %***********************************;
  %if (&_lsafRowCount_ > 0) %then
  %do;
    %let notesOption=%sysfunc(getoption(notes));
    options nonotes;
    proc sort data=&sas_dsname sortseq=linguistic(strength=primary);
      by elementID name;
    run;
    options &notesOption.;
  %end;
  
%mend lsaf_getpfsetupjobparameters;
