%* lsaf_getpfcompletedelements                                                                     *;
%*                                                                                                 *;
%* Gets the metadata for the completed elements for the specified process flow within the          *;
%* specified context. The metadata is stored in a SAS data set.                                    *;
%*                                                                                                 *;
%* macrotype LSAF                                                                                  *;
%* since 2.3                                                                                       *;
%* exposure external                                                                               *;
%*                                                                                                 *;
%* History                                                                                         *;
%* 2020-02-19  initial coding.                                                                     *;
%* 2021-08-12  Remove header documentation                                                         *;
%*                                                                                                 *;

%macro lsaf_getpfcompletedelements(
  lsaf_path=,
  lsaf_processflow=,
  sas_dsname=work.lsafGetPFCompletedElements
  ) /des='Get a list of process flow completed elements of a specific process flow in a SAS Life Science Analytics Framework context.';
  
  %if "&sas_dsname" EQ "" %then %let sas_dsname=work.lsafGetPFCompletedElements;

  %****************************************;
  %* Initialize macro variables           *;
  %****************************************;
  %lsaf_getProductName();

  %global _lsafMsg_
          _lsafRC_
  ;

  %let _lsafRC_ =%str(-999);
  %let _lsafMsg_=%str(The SAS Macro did not execute properly.  Unknown err%str(or).);


  %**********************************************;
  %* Start the datastep to call the java code   *;
  %**********************************************;
  data &sas_dsname (label="Created by &SYSMACRONAME"
                    keep=processFlowPath processFlowName elementID elementName elementDescription elementType elementCreated
                         elementCompleted elementCompletedBy);
    attrib returnCode             length=8
           message                length=$200
           logMsgPrefix           length=$70
           logMsgPrimary          length=$70
           logMsg                 length=$300
           rowCount               length=8

           processFlowPath        length=$2048   label="Context Path"
           processFlowName        length=$256    label="Process Flow Name"
           elementType            length=$64     label="Element Type"
           elementID              length=$256    label="Element Name"
           elementName            length=$256    label="Element Display Name"
           elementDescription     length=$2048   label="Element Value"
           elementCreated         length=$64     label="Element Date Created"
           elementCompleted       length=$64     label="Element Completed"
           elementCompletedBy     length=$64     label="Element Completed By"
    ;

    rowCount=0;

    %***********************************;
    %* Declare the java objects        *;
    %***********************************;
    declare javaobj srv("com/sas/lsaf/macro/workflow/task/ProcessFlowTaskService") ;
    declare javaobj results("com/sas/lsaf/macro/workflow/task/SasProcessFlowCompletedElementsResult");

    %***********************************;
    %* Attempt the operation           *;
    %***********************************;
    srv.callStaticVoidMethod("getProcessCompletedElements", "%bquote(&lsaf_path)", "%bquote(&lsaf_processflow)", results) ;

    %***********************************;
    %* Retrieve the results            *;
    %***********************************;
    results.callIntMethod("getReturnCode", returnCode);
    results.callStringMethod("getReturnMessage", message);

    %***********************************;
    %* Process the results             *;
    %***********************************;
    logMsgPrefix="NOTE: &_LSAF_PRODUCT_NAME_ Macro: * ";
    if (returnCode ne 0) then
    do;
      logMsgPrimary="ERR%str(OR): &_LSAF_PRODUCT_NAME_ Macro:";
      logMsg=" Cannot retrieve completed elements for  process flow &lsaf_processflow in &lsaf_path" ;
    end;
    else
    do;
      logMsgPrimary=logMsgPrefix;
      logMsg=" Completed elements were returned for process flow &lsaf_processflow in &lsaf_path" ;
      results.callIntMethod("getSize", rowCount);
      do row=0 to rowCount - 1;
        results.callStringMethod("getProcessFlowContext", processFlowPath);
        results.callStringMethod("getProcessFlowName", processFlowName);
        results.callStringMethod("getName", row, elementName);
        results.callStringMethod("getId", row, elementID);
        results.callStringMethod("getDescription", row, elementDescription);
        results.callStringMethod("getType", row, elementType);
        results.callStringMethod("getCompleted", row, elementCompleted);
        results.callStringMethod("getCompletedBy", row, elementCompletedBy);
        results.callStringMethod("getCreated", row, elementCreated);
        output;
      end;
    end;

    %***********************************;
    %* Print messages in log           *;
    %***********************************;
    put;
    put logMsgPrimary " " logMsg;
    put logMsgPrefix " " %sysfunc(lowcase("&SYSMACRONAME"));
    if (rowCount=0) 
      then put logMsgPrefix " No completed elements for process flow were found.";
      else put logMsgPrefix " " rowCount " records written to data set &sas_dsname";
    put logMsgPrefix " _lsafMsg_= " message ;
    put logMsgPrefix " _lsafRC_= " returnCode ;
    put;

    %***********************************;
    %* Set the macro variables         *;
    %***********************************;
    call symputx("_lsafRC_", returnCode);
    call symputx("_lsafMsg_", message);

  run;

%mend lsaf_getpfcompletedelements;
