%* lsaf_getpfcompleteddetails                                                                       *;
%*                                                                                                  *;
%* Gets the details for the completed elements of the process flow that is in the specified         *;
%* context. The data is stored in a SAS data set.                                                   *;
%*                                                                                                  *;
%* macrotype LSAF                                                                                   *;
%* since 2.4                                                                                        *;
%* exposure external                                                                                *;
%*                                                                                                  *;
%* History                                                                                          *;
%* 2020-10-22  original creation.                                                                   *;
%* 2021-03-17  Rename instanceId to taskId, add additional columns, and sort the dataset.           *;
%* 2021-08-12  Remove header documentation                                                          *;
%* 2021-08-25  DE16777                                                                              *;
%* 2021-10-13  US11739                                                                              *;
%* 2023-03-09  DE17654 - SAS Notes option updates                                                   *;
%*                                                                                                  *;

%macro lsaf_getpfcompleteddetails(
  lsaf_path=,
  lsaf_processflow=,
  lsaf_elementid=,
  sas_dsname=work.lsafgetpfcompleteddetails
  ) /des='Get a list of the details of the completed elements for the process flow in a SAS Life Science Analytics Framework context.';
  
  %if "&sas_dsname" EQ "" %then %let sas_dsname=work.lsafgetpfcompleteddetails;

  %****************************************;
  %* Initialize macro variables           *;
  %****************************************;
  %lsaf_getProductName();

  %global _lsafMsg_
          _lsafRC_
  ;

  %local _lsafRowCount_
         notesOption;

  %let _lsafRC_ =%str(-999);
  %let _lsafMsg_=%str(The SAS Macro did not execute properly.  Unknown err%str(or).);
  %let _lsafRowCount_=0;

  %**********************************************;
  %* Start the datastep to call the java code   *;
  %**********************************************;
  data &sas_dsname (label="Created by &SYSMACRONAME"
                    keep=processFlowPath processFlowName elementID elementName elementType taskID elementCreated 
                    elementCreatedSasFormat elementCompleted elementCompletedSasFormat elementCompletedBy 
                    elementCompletedState recordTypeSortOrder recordType id name value completedDate);

    attrib returnCode       length=8
           message          length=$200
           logMsgPrefix     length=$70
           logMsgPrimary    length=$70
           logMsg           length=$300
           rowCount         length=8

           processFlowPath           length=$3000  label="Context Path"
           processFlowName           length=$256   label="Process Flow Name"
           elementID                 length=$256   label="Element Identifier"
           elementName               length=$256   label="Element Name"
           elementType               length=$64    label="Element Type"
           taskID                    length=$256   label="Task Identifier"
           elementCreated            length=$64    label="Element Date Created"
           elementCreatedSasFormat   length=$64    label="Element Date Created (SAS Format)"
           elementCompleted          length=$64    label="Element Completed"
           elementCompletedSasFormat length=$64    label="Element Completed (SAS Format)"
           elementCompletedBy        length=$64    label="Element Completed By"
           elementCompletedState     length=$64    label="Completed State"
           recordTypeSortOrder       length=8      label="Record Type Sort Order"
           recordType                length=$64    label="Record Type"
           id                        length=$64    label="Detail Identifier"
           name                      length=$256   label="Detail Name"
           value                     length=$2048  label="Detail Value"
           completedDate             length=8      label="Numeric Completed Date WILL BE DROPPED"
    ;

    rowCount=0;

    %***********************************;
    %* Declare the java objects        *;
    %***********************************;
    declare javaobj srv("com/sas/lsaf/macro/workflow/task/ProcessFlowTaskService") ;
    declare javaobj results("com/sas/lsaf/macro/workflow/task/SasCompletedElementResult");

    %***********************************;
    %* Attempt the operation           *;
    %***********************************;
    srv.callStaticVoidMethod("getProcessCompletedElementDetails", "%bquote(&lsaf_path)", "%bquote(&lsaf_processflow)",
        "%bquote(&lsaf_elementid)", results) ;

    %***********************************;
    %* Retrieve the results            *;
    %***********************************;
    results.callIntMethod("getReturnCode", returnCode);
    results.callStringMethod("getReturnMessage", message);

    %***********************************;
    %* Process the results             *;
    %***********************************;
    logMsgPrefix="NOTE: &_LSAF_PRODUCT_NAME_ Macro: * ";
    if (returnCode ne 0) then
    do;
      logMsgPrimary="ERR%str(OR): &_LSAF_PRODUCT_NAME_ Macro:";
      logMsg=" Cannot retrieve details of completed elements for &lsaf_processflow in &lsaf_path" ;
    end;
    else
    do;
      logMsgPrimary=logMsgPrefix;
      logMsg=" Details for completed elements were returned for &lsaf_processflow in &lsaf_path" ;
      results.callIntMethod("getSize", rowCount);
      results.callStringMethod("getProcessFlowContext", processFlowPath);
      results.callStringMethod("getProcessFlowName", processFlowName);
      do row=0 to rowCount - 1;
        results.callStringMethod("getElementId", row, elementID);
        results.callStringMethod("getElementName", row, elementName);
        results.callStringMethod("getElementType", row, elementType);
        results.callStringMethod("getInstanceId", row, taskID);
        results.callStringMethod("getCreated", row, elementCreated);
        results.callStringMethod("getCreatedSasFormat", row, elementCreatedSasFormat);
        results.callStringMethod("getCompleted", row, elementCompleted);
        results.callStringMethod("getCompletedSasFormat", row, elementCompletedSasFormat);
        completedDate=input(elementCompletedSasFormat,datetime20.);
        results.callStringMethod("getCompletedBy", row, elementCompletedBy);
        results.callStringMethod("getCompletedState", row, elementCompletedState);
        results.callIntMethod("getRecordTypeSortOrder", row, recordTypeSortOrder);
        results.callStringMethod("getRecordType", row, recordType);
        results.callStringMethod("getId", row, id);
        results.callStringMethod("getName", row, name);
        results.callStringMethod("getValue", row, value);
        output;
      end;
    end;

    %***********************************;
    %* Print messages in log           *;
    %***********************************;
    put;
    put logMsgPrimary " " logMsg;
    put logMsgPrefix " " %sysfunc(lowcase("&SYSMACRONAME"));
    if (rowCount=0) 
      then put logMsgPrefix " No process flow completed elements were found.";
      else put logMsgPrefix " " rowCount " records written to data set &sas_dsname";
    put logMsgPrefix " _lsafMsg_= " message ;
    put logMsgPrefix " _lsafRC_= " returnCode ;
    put;

    %***********************************;
    %* Set the macro variables         *;
    %***********************************;
    call symputx("_lsafRC_", returnCode);
    call symputx("_lsafMsg_", message);
    call symputx("_lsafRowCount_", rowCount);

  run;

  %let notesOption=%sysfunc(getoption(notes));
  options nonotes;

  %***********************************;
  %* Sort the data set               *;
  %***********************************;
  %if (&_lsafRowCount_ > 0) %then
  %do;
    proc sort data=&sas_dsname sortseq=linguistic(strength=primary);
      by descending completedDate taskId recordTypeSortOrder id name;
    run;
  %end;

  %***********************************;
  %* Cleanup                         *;
  %***********************************;
  proc sql;
     alter table &sas_dsname
        drop completedDate;
  quit;

  options &notesOption.;

%mend lsaf_getpfcompleteddetails;
