%* lsaf_getjobsubmissionbysessionid                                                                *;
%*                                                                                                 *;
%* Gets the properties of a job submission by the session identifier. The properties are stored    *;
%* in a SAS data set.                                                                              *;
%*                                                                                                 *;
%* macrotype LSAF                                                                                  *;
%* since 2.4                                                                                       *;
%* exposure external                                                                               *;
%*                                                                                                 *;
%* History                                                                                         *;
%* 2020-07-10  initial coding                                                                      *;
%* 2021-08-09  Remove header documentation                                                         *;
%* 2021-10-12  US11739                                                                             *;
%*                                                                                                 *;

%macro lsaf_getjobsubmissionbysessionid(
  lsaf_sessionid=,
  sas_dsname=work.lsafjobsubmissionbysessionid
  ) /des='Get a list of properties for a specified job submission in the SAS Life Science Analytics Framework Repository.';
  
  %if "&sas_dsname" EQ "" %then %let sas_dsname=work.lsafjobsubmissionbysessionid;

  %****************************************;
  %* Initialize macro variables           *;
  %****************************************;
  %lsaf_getProductName();

  %global _lsafMsg_
          _lsafRC_
  ;

  %let _lsafRC_ =%str(-999);
  %let _lsafMsg_=%str(The SAS Macro did not execute properly.  Unknown err%str(or).);

  %local notesOption
         _lsafRowCount_
  ;

  %let _lsafRowCount_=0;

  %**********************************************;
  %* Start the datastep to call the java code   *;
  %**********************************************;
  data &sas_dsname (label="Created by &SYSMACRONAME"
                    keep=sessionId name value);
    attrib returnCode      length=8
           message         length=$200
           logMsgPrefix    length=$70
           logMsgPrimary   length=$70
           logMsg          length=$300
           rowCount        length=8

           sessionId       length=$256     label="Job Submission Session ID"
           name            length=$256     label="Property Name"
           value           length=$2048    label="Property Value"
   ;

    rowCount=0;

    %***********************************;
    %* Declare the java objects        *;
    %***********************************;
    declare javaobj srv("com/sas/lsaf/macro/execution/jobsubmission/JobSubmissionService") ;
    declare javaobj results("com/sas/lsaf/macro/common/result/SasPropertiesResult");

    %***********************************;
    %* Attempt the operation           *;
    %***********************************;
    srv.callStaticVoidMethod("getJobSubmissionBySessionId", "%bquote(&lsaf_sessionid)", results);

    %***********************************;
    %* Retrieve the results            *;
    %***********************************;
    results.callIntMethod("getReturnCode", returnCode);
    results.callStringMethod("getReturnMessage", message);

    %***********************************;
    %* Process the results             *;
    %***********************************;
    logMsgPrefix="NOTE: &_LSAF_PRODUCT_NAME_ Macro: * ";
    if (returnCode ne 0) then
    do;
      logMsgPrimary="ERR%str(OR): &_LSAF_PRODUCT_NAME_ Macro:";
      logMsg=" Cannot retrieve list of job submission properties for &lsaf_sessionid" ;
    end;
    else
    do;
      logMsgPrimary=logMsgPrefix;
      logMsg=" List of job submission properties were returned for &lsaf_sessionid" ;
      results.callIntMethod("getSize", rowCount);
      do row=0 to rowCount - 1;
        sessionId=kstrip("&lsaf_sessionid");
        results.callStringMethod("getPropertyName", row, name);
        results.callStringMethod("getPropertyValue", row, value);
        output;
      end;
    end;

    %***********************************;
    %* Print messages in log           *;
    %***********************************;
    put;
    put logMsgPrimary " " logMsg;
    put logMsgPrefix " " %sysfunc(lowcase("&SYSMACRONAME"));
    if (rowCount=0) 
      then put logMsgPrefix " No job submission properties were retrieved.";
      else put logMsgPrefix " " rowCount " records written to data set &sas_dsname";
    put logMsgPrefix " _lsafMsg_= " message ;
    put logMsgPrefix " _lsafRC_= " returnCode ;
    put;

    %***********************************;
    %* Set the macro variables         *;
    %***********************************;
    call symputx("_lsafRC_", returnCode);
    call symputx("_lsafMsg_", message);
    call symputx("_lsafRowCount_", rowCount);

  run;

  %***********************************;
  %* Sort the data set               *;
  %***********************************;
  %if (&_lsafRowCount_ > 0) %then
  %do;
    %let notesOption=%sysfunc(getoption(notes));
    options nonotes;
    proc sort data=&sas_dsname sortseq=linguistic(strength=primary);
      by name;
    run;
    options &notesOption;
  %end;

%mend lsaf_getjobsubmissionbysessionid;
