%* lsaf_getjobpublishhistory                                                                       *;
%*                                                                                                 *;
%* Gets the publish history for the specified job in the repository. The metadata is stored in a   *;
%* SAS data set.                                                                                   *;
%*                                                                                                 *;
%* macrotype LSAF                                                                                  *;
%* since 2.3                                                                                       *;
%* exposure external                                                                               *;
%*                                                                                                 *;
%* History                                                                                         *;
%* 2019-11-13  initial coding.                                                                     *;
%* 2020-06-30  update service location                                                             *;
%* 2020-07-09  add jobId and owner variables to output data set                                    *;
%* 2021-08-09  Remove header documentation                                                         *;
%*                                                                                                 *;

%macro lsaf_getjobpublishhistory(
   lsaf_path=,
   sas_dsname=work.lsafGetJobPublishHistory
   ) /des='Gets the publish history for a job manifest.';

   %if "&sas_dsname" EQ "" %then %let sas_dsname=work.lsafGetJobPublishHistory;

   %****************************************;
   %* Initialize macro variables           *;
   %****************************************;
  %lsaf_getProductName();

   %global _lsafMsg_
           _lsafRC_
   ;

   %let _lsafRC_ =%str(-999);
   %let _lsafMsg_=%str(The SAS Macro did not execute properly.  Unknown err%str(or).);

   %**********************************************;
   %* Start the datastep to call the java code   *;
   %**********************************************;
   data &sas_dsname (label="Created by &SYSMACRONAME"
                     keep=path id jobId version sessionId submittedBy submittedDate submittedDateSAS
                          completedDate completedDateSAS summaryStatus detailStatus owner runAsUser
   );
   
   attrib returnCode      length=8
          message         length=$200
          logMsgPrefix    length=$70
          logMsgPrimary   length=$70
          logMsg          length=$300
          rowCount        length=8

          path               length=$2048     label="Job Path"
          id                 length=$256      label="Job Submission Identifier"
          jobId              length=$256      label="Job Identifier"
          version            length=$32       label="Job Version"
          sessionId          length=$256      label="Job Session Identifier"
          submittedBy        length=$32       label="Submitted By"
          submittedDate      length=$32       label="Date Submitted"
          submittedDateSAS   length=8         label="Date Submitted (SAS Datetime Format)"
          completedDate      length=$32       label="Date Completed"
          completedDateSAS   length=8         label="Date Completed (SAS Datetime Format)"
          summaryStatus      length=$32       label="Job Submission Summary Status"
          detailStatus       length=$32       label="Job Submission Detail Status"
          owner              length=$32       label="Job Owner"
          runAsUser          length=$32       label="Run as User"
          
          submittedDateSAStemp   length=$64   label="TEMPORARY - WILL BE DROPPED"
	  completedDateSAStemp   length=$64   label="TEMPORARY - WILL BE DROPPED"

   ;

   format submittedDateSAS completedDateSAS datetime.;
   rowCount=0;

   %***********************************;
   %* Declare the java objects        *;
   %***********************************;
   declare javaobj srv("com/sas/lsaf/macro/execution/jobsubmission/JobSubmissionService") ;
   declare javaobj results("com/sas/lsaf/macro/execution/jobsubmission/SasJobSubmissionsResult");

   %***********************************;
   %* Attempt the operation           *;
   %***********************************;
   srv.callStaticVoidMethod("getJobSubmissionHistory", "%bquote(&lsaf_path)", results) ;

   %***********************************;
   %* Retrieve the results            *;
   %***********************************;
   results.callIntMethod("getReturnCode", returnCode);
   results.callStringMethod("getReturnMessage", message);

   %***********************************;
   %* Process the results             *;
   %***********************************;
   logMsgPrefix="NOTE: &_LSAF_PRODUCT_NAME_ Macro: * ";
   if (returnCode ne 0) then
   do;
      logMsgPrimary="ERR%str(OR): &_LSAF_PRODUCT_NAME_ Macro:";
      logMsg=" Cannot retrieve publish history for job &lsaf_path" ;
   end;
   else
   do;
      logMsgPrimary=logMsgPrefix;
      logMsg=" Publish history was returned for job &lsaf_path" ;
      results.callIntMethod("getSize", rowCount);
      do row=0 to rowCount - 1;
         results.callStringMethod("getId", row, id);
         results.callStringMethod("getJobPath", row, path);
         results.callStringMethod("getJobId", row, jobId);
         results.callStringMethod("getJobVersion", row, version);
         results.callStringMethod("getSessionId", row, sessionId);
         results.callStringMethod("getSubmittedBy", row, submittedBy);
         results.callStringMethod("getSubmitted", row, submittedDate);
         results.callStringMethod("getSubmittedSasFormat", row, submittedDateSAStemp);
         results.callStringMethod("getCompleted", row, completedDate);
         results.callStringMethod("getCompletedSasFormat", row, completedDateSAStemp);
         results.callStringMethod("getJobSubmissionSummaryStatus", row, summaryStatus);
         results.callStringMethod("getJobSubmissionDetailStatus", row, detailStatus);
         results.callStringMethod("getJobOwner", row, owner);
         results.callStringMethod("getRunAsUser", row, runAsUser);
         submittedDateSAS=input(submittedDateSAStemp, datetime.);
         completedDateSAS=input(completedDateSAStemp, datetime.);
         output;
      end;
   end;

   %***********************************;
   %* Print messages in log           *;
   %***********************************;
   put;
   put logMsgPrimary " " logMsg;
   put logMsgPrefix " " %sysfunc(lowcase("&SYSMACRONAME"));
   if (rowCount=0) 
     then put logMsgPrefix " No publish history was found.";
     else put logMsgPrefix " " rowCount " records written to data set &sas_dsname";
   put logMsgPrefix " _lsafMsg_= " message ;
   put logMsgPrefix " _lsafRC_= " returnCode ;
   put;

   %***********************************;
   %* Set the macro variables         *;
   %***********************************;
   call symputx("_lsafRC_", returnCode);
   call symputx("_lsafMsg_", message);

   run;

%mend lsaf_getjobpublishhistory;