%* lsaf_getjobmanifestresultlocs                                                                  *;
%*                                                                                                *;
%* Gets the metadata for SAS Life Science Analytics Framework repository job manifest result      *;
%* locations.                                                                                     *;
%*                                                                                                *;
%* macrotype LSAF                                                                                 *;
%* since 2.4                                                                                      *;
%* exposure external                                                                              *;
%*                                                                                                *;
%* History                                                                                        *;
%* 2020-07-29  initial coding.                                                                    *;
%* 2021-07-06  add lsaf_location parameter.                                                       *;
%* 2021-08-09  Remove header documentation                                                        *;
%* 2021-10-12  US11739                                                                            *;
%*                                                                                                *;

%macro lsaf_getjobmanifestresultlocs(
  lsaf_location=,
  lsaf_path=,
  lsaf_version=,
  sas_dsname=work.lsafGetJobManifestResultLocs
  ) /des='Get the metadata for SAS Life Science Analytics Framework repository job manifest result locations.';
  
  %if "&sas_dsname" EQ "" %then %let sas_dsname=work.lsafGetJobManifestResultLocs;

  %****************************************;
  %* Initialize macro variables           *;
  %****************************************;
  %lsaf_getProductName();

  %global _lsafMsg_
          _lsafRC_
  ;

  %let _lsafRC_ =%str(-999);
  %let _lsafMsg_=%str(The SAS Macro did not execute properly.  Unknown err%str(or).);

  %local _lsafRowCount_
         notesOption
  ;

  %let _lsafRowCount_=0;

  %**********************************************;
  %* Start the datastep to call the java code   *;
  %**********************************************;
  data &sas_dsname (label="Created by &SYSMACRONAME"
                    keep=jobManifestLocation jobManifestPath jobManifestVersion listType resultPath resultVersion resultName engineType dateModified srcLocation fileSize fileSizeFmt);
    attrib returnCode      length=8
           message         length=$200
           logMsgPrefix    length=$70
           logMsgPrimary   length=$70
           logMsg          length=$300

           jobManifestLocation length=$32      label="Job Manifest Source Location"
           jobManifestPath     length=$2048    label="Job Manifest Path"
           jobManifestVersion  length=$32      label="Job Manifest Version"
           listType            length=$8       label="List Type"
           resultPath          length=$2048    label="Result Path"
           resultVersion       length=$32      label="Result Version Number"
           resultName          length=$256     label="Result Name"
           engineType          length=$32      label="Engine Type"
           dateModified        length=$32      label="Date Last Modified"
           srcLocation         length=$2048    label="Source Location"
           fileSize            length=8        label="Result File Size"
           fileSizeFmt         length=$10      label="Result File Size (Formatted)"
    ;

    rowCount=0;

    %***********************************;
    %* Declare the java objects        *;
    %***********************************;
    declare javaobj srv("com/sas/lsaf/macro/execution/jobmanifest/JobManifestService") ;
    declare javaobj results("com/sas/lsaf/macro/execution/jobmanifest/SasManifestSystemFilesResult");

    %***********************************;
    %* Attempt the operation           *;
    %***********************************;
    srv.callStaticVoidMethod("getJobManifestResults", "%bquote(&lsaf_location)", "%bquote(&lsaf_path)", "%bquote(&lsaf_version)", results);

    %***********************************;
    %* Retrieve the results            *;
    %***********************************;
    results.callIntMethod("getReturnCode", returnCode);
    results.callStringMethod("getReturnMessage", message);

    %***********************************;
    %* Process the results             *;
    %***********************************;
    logMsgPrefix="NOTE: &_LSAF_PRODUCT_NAME_ Macro: * ";
    if (returnCode ne 0) then
    do;
      logMsgPrimary="ERR%str(OR): &_LSAF_PRODUCT_NAME_ Macro:";
      logMsg=" Could not retrieve job manifest result locations." ;
    end;
    else
    do;
      logMsgPrimary=logMsgPrefix;
      logMsg=" Retrieved job manifest information.";
      results.callStringMethod("getManifestSourceLocation", jobManifestLocation);
      results.callStringMethod("getManifestPath", jobManifestPath);
      results.callStringMethod("getManifestVersion", jobManifestVersion);
      results.callIntMethod("getSize", rowcount);
      results.callStringMethod("getListType", listType);         
      do row=0 to rowCount - 1;
        results.callStringMethod("getName", row, resultName);
        results.callStringMethod("getVersion", row, resultVersion);
        results.callStringMethod("getPath", row, resultPath);
        results.callStringMethod("getEngineType", row, engineType);
        results.callStringMethod("getLastModified", row, dateModified);
        results.callStringMethod("getSourceLocation", row, srcLocation);
        results.callLongMethod("getFileSize", row, fileSize);      
        results.callStringMethod("getFormattedFileSize", row, fileSizeFmt);
        output;
      end;
    end;

    %***********************************;
    %* Print messages in log           *;
    %***********************************;
    put;
    put logMsgPrimary " " logMsg;
    put logMsgPrefix " " %sysfunc(lowcase("&SYSMACRONAME"));
    if (rowCount=0) 
      then put logMsgPrefix " The job manifest result locations were not found.";
      else put logMsgPrefix  rowCount " records written to data set &sas_dsname";
    put logMsgPrefix " _lsafMsg_= " message ;
    put logMsgPrefix " _lsafRC_= " returnCode ;
    put;

    %***********************************;
    %* Set the macro variables         *;
    %***********************************;
    call symputx("_lsafRC_", returnCode);
    call symputx("_lsafMsg_", message);
    call symputx("_lsafRowCount_", rowCount);

  run;

  %***********************************;
  %* Sort the data set               *;
  %***********************************;
  %if (&_lsafRowCount_ > 0) %then
  %do;
    %let notesOption=%sysfunc(getoption(notes));
    options nonotes;
    proc sort data=&sas_dsname sortseq=linguistic(strength=primary);
      by resultName;
    run;
    options &notesOption;
  %end;

%mend lsaf_getjobmanifestresultlocs;
