%* lsaf_getjobmanifestinfo                                                                        *;
%*                                                                                                *;
%* Gets the information for a job manifest in the repository. The information is stored in a SAS  *;
%* data set. The output data set can be used as an input data set for the lsaf_createworkspacejob *;
%* macro.                                                                                         *;
%*                                                                                                *;
%* macrotype LSAF                                                                                 *;
%* since 2.3                                                                                      *;
%* exposure external                                                                              *;
%*                                                                                                *;
%* History                                                                                        *;
%* 2019-12-03  initial coding.                                                                    *;
%* 2021-07-06  add lsaf_location parameter.                                                       *;
%* 2021-08-09  Remove header documentation                                                        *;
%* 2021-10-12  US11739                                                                            *;
%*                                                                                                *;

%macro lsaf_getjobmanifestinfo(
  lsaf_location=,
  lsaf_path=,
  lsaf_version=,
  sas_dsname=work.lsafGetJobManifestInfo
  ) /des='Get the metadata for SAS Life Science Analytics Framework repository job manifest.';
  
  %if "&sas_dsname" EQ "" %then %let sas_dsname=work.lsafGetJobManifestInfo;

  %****************************************;
  %* Initialize macro variables           *;
  %****************************************;
  %lsaf_getProductName();

  %global _lsafMsg_
          _lsafRC_
  ;

  %let _lsafRC_ =%str(-999);
  %let _lsafMsg_=%str(The SAS Macro did not execute properly.  Unknown err%str(or).);

  %local _lsafRowCount_
         notesOption
  ;

  %let _lsafRowCount_=0;

  %**********************************************;
  %* Start the datastep to call the java code   *;
  %**********************************************;
  data &sas_dsname (label="Created by &SYSMACRONAME"
                    keep=jobManifestLocation jobManifestPath jobManifestVersion name value);
    attrib returnCode      length=8
           message         length=$200
           logMsgPrefix    length=$70
           logMsgPrimary   length=$70
           logMsg          length=$300

           jobManifestLocation length=$32   label="Job Manifest Source Location"
           jobManifestPath     length=$2048 label="Job Manifest Path"
           jobManifestVersion  length=$32   label="Job Manifest Version"
           name                length=$256  label="Property Name"
           value               length=$2048 label="Property Value"
    ;

    rowCount=0;

    %***********************************;
    %* Declare the java objects        *;
    %***********************************;
    declare javaobj srv("com/sas/lsaf/macro/execution/jobmanifest/JobManifestService") ;
    declare javaobj results("com/sas/lsaf/macro/execution/jobmanifest/SasJobManifestInfoResult");

    %***********************************;
    %* Attempt the operation           *;
    %***********************************;
    srv.callStaticVoidMethod("getJobManifestInfo", "%bquote(&lsaf_location)", "%bquote(&lsaf_path)", "%bquote(&lsaf_version)", results);

    %***********************************;
    %* Retrieve the results            *;
    %***********************************;
    results.callIntMethod("getReturnCode", returnCode);
    results.callStringMethod("getReturnMessage", message);

    %***********************************;
    %* Process the results             *;
    %***********************************;
    logMsgPrefix="NOTE: &_LSAF_PRODUCT_NAME_ Macro: * ";
    if (returnCode ne 0) then
    do;
      logMsgPrimary="ERR%str(OR): &_LSAF_PRODUCT_NAME_ Macro:";
      logMsg=" Could not retrieve job manifest information." ;
    end;
    else
    do;
      logMsgPrimary=logMsgPrefix;
      logMsg=" Retrieved job manifest information.";
      results.callStringMethod("getManifestSourceLocation", jobManifestLocation);
      results.callStringMethod("getManifestPath", jobManifestPath);
      results.callStringMethod("getManifestVersion", jobManifestVersion);
      results.callIntMethod("getSize", rowcount);
      do row=0 to rowCount - 1;
         results.callStringMethod("getName", row, name);
         results.callStringMethod("getValue", row, value);
         output;
      end;
    end;

    %***********************************;
    %* Print messages in log           *;
    %***********************************;
    put;
    put logMsgPrimary " " logMsg;
    put logMsgPrefix " " %sysfunc(lowcase("&SYSMACRONAME"));
    if (rowCount=0) 
      then put logMsgPrefix " The job manifest information was not found.";
      else put logMsgPrefix  rowCount " records written to data set &sas_dsname";
    put logMsgPrefix " _lsafMsg_= " message ;
    put logMsgPrefix " _lsafRC_= " returnCode ;
    put;

    %***********************************;
    %* Set the macro variables         *;
    %***********************************;
    call symputx("_lsafRC_", returnCode);
    call symputx("_lsafMsg_", message);
    call symputx("_lsafRowCount_", rowCount);

  run;

  %***********************************;
  %* Sort the data set               *;
  %***********************************;
  %if (&_lsafRowCount_ > 0) %then
  %do;
    %let notesOption=%sysfunc(getoption(notes));
    options nonotes;
    proc sort data=&sas_dsname sortseq=linguistic(strength=primary);
      by name;
    run;
    options &notesOption;
  %end;

%mend lsaf_getjobmanifestinfo;
