%* lsaf_getjobmanifestexternalrefs                                                                    *;
%*                                                                                                    *;
%* Gets the metadata for the external references for the specified job manifest in the                *;
%* repository. The metadata is stored in a SAS data set.                                              *;
%*                                                                                                    *;
%* macrotype LSAF                                                                                     *;
%* since 2.4                                                                                          *;
%* exposure external                                                                                  *;
%*                                                                                                    *;
%* History                                                                                            *;
%* 2020-09-04  initial coding.                                                                        *;
%* 2021-07-06  add lsaf_location parameter.                                                           *;
%* 2021-08-09  Remove header documentation                                                            *;
%*                                                                                                    *;

%macro lsaf_getjobmanifestexternalrefs(
   lsaf_location=,
   lsaf_path=,
   lsaf_version=,
   lsaf_type=,
   sas_dsname=work.lsafGetJobManifestExternalRefs
   ) /des='Get a list of the external references for a job manifest.';
   
   %if "&sas_dsname" EQ "" %then %let sas_dsname=work.lsafGetJobManifestExternalRefs;

   %****************************************;
   %* Initialize macro variables           *;
   %****************************************;
   %lsaf_getProductName();

   %global _lsafMsg_
           _lsafRC_
   ;

   %let _lsafRC_ =%str(-999);
   %let _lsafMsg_=%str(The SAS Macro did not execute properly.  Unknown err%str(or).);

   %**********************************************;
   %* Start the datastep to call the java code   *;
   %**********************************************;
   data &sas_dsname (label="Created by &SYSMACRONAME"
                     keep=jobManifestLocation jobManifestPath jobManifestVersion path type referencedProgram referenceType
   );
   attrib returnCode         length=8
          message            length=$200
          logMsgPrefix       length=$70
          logMsgPrimary      length=$70
          logMsg             length=$300
          rowCount           length=8
          
          jobManifestLocation length=$32   label="Job Manifest Source Location"
          jobManifestPath     length=$2048 label="Job Manifest Path"
          jobManifestVersion  length=$32   label="Job Manifest Version"
          path                length=$2048 label="Path"
          type                length=$10   label="Type"
          referencedProgram   length=$100  label="Referenced SAS Program"
          referencetype       length=$32   label="Reference Type"
   ;

   rowCount=0;

   %***********************************;
   %* Declare the java objects        *;
   %***********************************;
   declare javaobj srv("com/sas/lsaf/macro/execution/jobmanifest/JobManifestService") ;
   declare javaobj results("com/sas/lsaf/macro/execution/jobmanifest/SasManifestExternalReferencesResult");

   %***********************************;
   %* Attempt the operation           *;
   %***********************************;
   srv.callStaticVoidMethod("getJobManifestExternalRefs", "%bquote(&lsaf_location)", "%bquote(&lsaf_path)", "%bquote(&lsaf_version)","%bquote(&lsaf_type)", results) ;

   %***********************************;
   %* Retrieve the results            *;
   %***********************************;
   results.callIntMethod("getReturnCode", returnCode);
   results.callStringMethod("getReturnMessage", message);

   %***********************************;
   %* Process the results             *;
   %***********************************;
   logMsgPrefix="NOTE: &_LSAF_PRODUCT_NAME_ Macro: * ";
   if (returnCode ne 0) then
   do;
     logMsgPrimary="ERR%str(OR): &_LSAF_PRODUCT_NAME_ Macro:";
     if "&lsaf_version" NE ""
       then logMsg=" Cannot retrieve external references for job manifest &lsaf_path, version &lsaf_version.";
       else logMsg=" Cannot retrieve external references for job manifest &lsaf_path.";
   end;
   else
   do;
     logMsgPrimary=logMsgPrefix;
     logMsg=" Inputs were returned for job manifest &lsaf_path" ;
     results.callStringMethod("getManifestSourceLocation", jobManifestLocation);
     results.callStringMethod("getManifestPath", jobManifestPath);
     results.callStringMethod("getManifestVersion", jobManifestVersion);
     results.callStringMethod("getListType", referenceType);
     results.callIntMethod("getSize", rowCount);
     do row=0 to rowCount - 1;
       results.callStringMethod("getPath", row, path);
       results.callStringMethod("getType", row, type);
       results.callStringMethod("getListType", referenceType);
       results.callStringMethod("getReferencingProgram", row, referencedProgram);
       output;
     end;
   end;

   %***********************************;
   %* Print messages in log           *;
   %***********************************;
   put;
   put logMsgPrimary " " logMsg;
   put logMsgPrefix " " %sysfunc(lowcase("&SYSMACRONAME"));
   if (rowCount=0) 
     then put logMsgPrefix " No external references were found.";
     else put logMsgPrefix " " rowCount " records written to data set &sas_dsname";
   put logMsgPrefix " _lsafMsg_= " message ;
   put logMsgPrefix " _lsafRC_= " returnCode ;
   put;

   %***********************************;
   %* Set the macro variables         *;
   %***********************************;
   call symputx("_lsafRC_", returnCode);
   call symputx("_lsafMsg_", message);

   run;

%mend lsaf_getjobmanifestexternalrefs;
