%* lsaf_getjobinfo                                                                                *;
%*                                                                                                *;
%* Gets the information for a job in the repository. The information is stored in a SAS data set. *;
%*                                                                                                *;
%* macrotype LSAF                                                                                 *;
%* since 1.4                                                                                      *;
%* exposure external                                                                              *;
%*                                                                                                *;
%* History                                                                                        *;
%* 2014-04-08  initial coding.                                                                    *;
%* 2016-08-26  rebrand as LSAF                                                                    *;
%* 2019-02-11  update to 2.1                                                                      *;
%* 2019-11-01  update to handle a blank SAS output file.                                          *;
%* 2021-08-06  Remove header documentation                                                        *;
%* 2021-10-12  US11739                                                                            *;
%*                                                                                                *;

%macro lsaf_getjobinfo(
  lsaf_path=,
  lsaf_version=,
  sas_dsname=work.lsafGetJobInfo
  ) /des='Get the metadata for SAS Life Science Analytics Framework repository job.';
  
  %if "&sas_dsname" EQ "" %then %let sas_dsname=work.lsafGetJobInfo;

  %****************************************;
  %* Initialize macro variables           *;
  %****************************************;
  %lsaf_getProductName();

  %global _lsafMsg_
          _lsafRC_
  ;

  %let _lsafRC_ =%str(-999);
  %let _lsafMsg_=%str(The SAS Macro did not execute properly.  Unknown err%str(or).);

  %local _lsafRowCount_
         notesOption
  ;

  %let _lsafRowCount_=0;

  %**********************************************;
  %* Start the datastep to call the java code   *;
  %**********************************************;
  data &sas_dsname (label="Created by &SYSMACRONAME"
                    keep=path version name value);
    attrib returnCode      length=8
           message         length=$200
           logMsgPrefix    length=$70
           logMsgPrimary   length=$70
           logMsg          length=$300

           path            length=$2048    label="Job Path"
           version         length=$32      label="Job Version Number"
           name            length=$256     label="Property Name"
           value           length=$2048    label="Property Value"
    ;

    rowCount=0;

    %***********************************;
    %* Declare the java objects        *;
    %***********************************;
    declare javaobj srv("com/sas/lsaf/macro/execution/job/JobService") ;
    declare javaobj results("com/sas/lsaf/macro/execution/job/SasJobPropertiesResult");

    %***********************************;
    %* Attempt the operation           *;
    %***********************************;
    srv.callStaticVoidMethod("getRepositoryJobInfo", "%bquote(&lsaf_path)", "%bquote(&lsaf_version)", results);

    %***********************************;
    %* Retrieve the results            *;
    %***********************************;
    results.callIntMethod("getReturnCode", returnCode);
    results.callStringMethod("getReturnMessage", message);

    %***********************************;
    %* Process the results             *;
    %***********************************;
    logMsgPrefix="NOTE: &_LSAF_PRODUCT_NAME_ Macro: * ";
    if (returnCode ne 0) then
    do;
      logMsgPrimary="ERR%str(OR): &_LSAF_PRODUCT_NAME_ Macro:";
      logMsg=" Could not retrieve job information." ;
    end;
    else
    do;
      logMsgPrimary=logMsgPrefix;
      logMsg=" Retrieved job information.";
      results.callStringMethod("getJobPath", path);
      results.callStringMethod("getJobVersion", version);
      results.callIntMethod("getSize", rowcount);
      do row=0 to rowCount - 1;
         results.callStringMethod("getPropertyName", row, name);
         results.callStringMethod("getPropertyValue", row, value);
         output;
      end;
    end;

    %***********************************;
    %* Print messages in log           *;
    %***********************************;
    put;
    put logMsgPrimary " " logMsg;
    put logMsgPrefix " " %sysfunc(lowcase("&SYSMACRONAME"));
    if (rowCount=0) 
      then put logMsgPrefix " The job information was not found.";
      else put logMsgPrefix  rowCount " records written to data set &sas_dsname";
    put logMsgPrefix " _lsafMsg_= " message ;
    put logMsgPrefix " _lsafRC_= " returnCode ;
    put;

    %***********************************;
    %* Set the macro variables         *;
    %***********************************;
    call symputx("_lsafRC_", returnCode);
    call symputx("_lsafMsg_", message);
    call symputx("_lsafRowCount_", rowCount);

  run;

  %***********************************;
  %* Sort the data set               *;
  %***********************************;
  %if (&_lsafRowCount_ > 0) %then
  %do;
    %let notesOption=%sysfunc(getoption(notes));
    options nonotes;
    proc sort data=&sas_dsname sortseq=linguistic(strength=primary);
      by name;
    run;
    options &notesOption;
  %end;

%mend lsaf_getjobinfo;
