%* lsaf_getinheritedroles                                                                         *;
%*                                                                                                *;
%* Gets all inherited roles across contexts that are associated with the specified role.          *;
%*                                                                                                *;
%* macrotype LSAF                                                                                 *;
%* since 2.6                                                                                      *;
%* exposure external                                                                              *;
%*                                                                                                *;
%* History                                                                                        *;
%* 2021-11-24  initial code                                                                       *;
%*                                                                                                *;

%macro lsaf_getinheritedroles(
  lsaf_path=,
  lsaf_role=,
  sas_dsname=
  ) /des='Get a list of inherited roles that are associated with the specified role.';
  
  %if "&sas_dsname" EQ "" %then %let sas_dsname=work.lsafGetInheritedRoles;

  %****************************************;
  %* Initialize macro variables           *;
  %****************************************;
  %lsaf_getProductName();

  %global _lsafMsg_
          _lsafRC_
  ;

  %let _lsafRC_ =%str(-999);
  %let _lsafMsg_=%str(The SAS Macro did not execute properly.  Unknown err%str(or).);

  %local _lsafRowCount_;

  %let _lsafRowCount_=0;

  %**********************************************;
  %* Start the datastep to call the java code   *;
  %**********************************************;
  data &sas_dsname (label="Created by &SYSMACRONAME"
                    keep=contextPath name roleSrcCtxt
  );
    attrib returnCode      length=8
           message         length=$200
           logMsgPrefix    length=$70
           logMsgPrimary   length=$70
           logMsg          length=$300
           rowCount        length=8

           contextPath     length=$2048     label="Context Path" 
           name            length=$256      label="Role Name"
           roleSrcCtxt     length=$2048     label="Role Context Source Path"
    ;

    rowCount=0;

    %***********************************;
    %* Declare the java objects        *;
    %***********************************;
    declare javaobj srv("com/sas/lsaf/macro/security/role/RoleService") ;
    declare javaobj results("com/sas/lsaf/macro/security/role/SasRolesResult");

    %***********************************;
    %* Attempt the operation           *;
    %***********************************;
    srv.callStaticVoidMethod("getInheritedRolesByRole", "%bquote(&lsaf_path)", "%bquote(&lsaf_role)", results) ;

    %***********************************;
    %* Retrieve the results            *;
    %***********************************;
    results.callIntMethod("getReturnCode", returnCode);
    results.callStringMethod("getReturnMessage", message);

    %***********************************;
    %* Process the results             *;
    %***********************************;
    logMsgPrefix="NOTE: &_LSAF_PRODUCT_NAME_ Macro: * ";
    if (returnCode ne 0) then
    do;
      logMsgPrimary="ERR%str(OR): &_LSAF_PRODUCT_NAME_ Macro:";
      logMsg=" Cannot retrieve inherited roles for &lsaf_role in &lsaf_path" ;
    end;
    else
    do;
      logMsgPrimary=logMsgPrefix;
      logMsg=" Inherited roles were returned for &lsaf_role in &lsaf_path" ;
      results.callIntMethod("getSize", rowCount);
      do row=0 to rowCount - 1;
        contextPath="&lsaf_path";
        results.callStringMethod("getDefinedPath", row, contextPath);
        results.callStringMethod("getName", row, name);
        results.callStringMethod("getAssignedPath", row, roleSrcCtxt);
        output;
      end;
    end;

    %***********************************;
    %* Print messages in log           *;
    %***********************************;
    put;
    put logMsgPrimary " " logMsg;
    put logMsgPrefix " " %sysfunc(lowcase("&SYSMACRONAME"));
    if (rowCount=0) 
      then put logMsgPrefix " No inherited roles were found.";
      else put logMsgPrefix " " rowCount " records written to data set &sas_dsname";
    put logMsgPrefix " _lsafMsg_= " message ;
    put logMsgPrefix " _lsafRC_= " returnCode ;
    put;

    %***********************************;
    %* Set the macro variables         *;
    %***********************************;
    call symputx("_lsafRC_", returnCode);
    call symputx("_lsafMsg_", message);
    call symputx("_lsafRowCount_", rowCount);

  run;

  %***********************************;
  %* Sort by name                    *;
  %***********************************;
  %if (&_lsafRowCount_ > 0) %then
  %do;
    %let notesOption=%sysfunc(getoption(notes));
    options nonotes;
    proc sort data=&sas_dsname sortseq=linguistic(strength=primary);
      by roleSrcCtxt;
    run;
    options &notesOption;
  %end;

%mend lsaf_getinheritedroles;
