%* lsaf_getgroups                                                                                  *;
%*                                                                                                 *;
%* Gets the metadata for all of the groups that are defined within the specified context. The      *;
%* metadata is stored in a SAS data set.                                                           *;
%*                                                                                                 *;
%* macrotype LSAF                                                                                  *;
%* since 1.2                                                                                       *;
%* exposure external                                                                               *;
%*                                                                                                 *;
%* History                                                                                         *;
%* 2013-08-30  updates for standardization and coding conventions                                  *;
%* 2016-08-26  rebrand as LSAF                                                                     *;
%* 2019-01-21  updates for LSAF 5.1 behavior                                                       *;
%* 2019-11-01  update to handle a blank SAS output file.                                           *;
%* 2020-12-04  updates for service refactor                                                        *;
%* 2021-08-06  Remove header documentation                                                         *;
%* 2021-10-13  US11739                                                                             *;
%*                                                                                                 *;

%macro lsaf_getgroups(
  lsaf_path=,
  sas_dsname=work.lsafGetGroups
  ) /des='Get a list of groups defined in an LSAF context.';
  
  %if "&sas_dsname" EQ "" %then %let sas_dsname=work.lsafGetGroups;

  %****************************************;
  %* Initialize macro variables           *;
  %****************************************;
  %lsaf_getProductName();

  %global _lsafMsg_
          _lsafRC_
  ;

  %local _lsafRowCount_;

  %let _lsafRC_ =%str(-999);
  %let _lsafMsg_=%str(The SAS Macro did not execute properly.  Unknown err%str(or).);
  %let _lsafRowCount_=0;

  %**********************************************;
  %* Start the datastep to call the java code   *;
  %**********************************************;
  data &sas_dsname (label="Created by &SYSMACRONAME"
                    keep=contextPath name description);
    attrib returnCode      length=8;
    attrib message         length=$200;
    attrib logMsgPrefix    length=$70;
    attrib logMsgPrimary   length=$70;
    attrib logMsg          length=$300;
    attrib rowCount        length=8;

    attrib contextPath     length=$2048    label="Context Path" ;
    attrib name            length=$256     label="Group Name";
    attrib description     length=$128     label="Group Description";

    rowCount=0;

    %***********************************;
    %* Declare the java objects        *;
    %***********************************;
    declare javaobj srv("com/sas/lsaf/macro/security/group/GroupService") ;
    declare javaobj results("com/sas/lsaf/macro/security/group/SasGroupsResult");

    %***********************************;
    %* Attempt the operation           *;
    %***********************************;
    srv.callStaticVoidMethod("getGroups", "%bquote(&lsaf_path)", results);

    %***********************************;
    %* Retrieve the results            *;
    %***********************************;
    results.callIntMethod("getReturnCode", returnCode);
    results.callStringMethod("getReturnMessage", message);

    %***********************************;
    %* Process the results             *;
    %***********************************;
    logMsgPrefix="NOTE: &_LSAF_PRODUCT_NAME_ Macro: * ";
    if (returnCode ne 0) then
    do;
      logMsgPrimary="ERR%str(OR): &_LSAF_PRODUCT_NAME_ Macro:";
      logMsg=" Cannot retrieve list of groups in &lsaf_path..";
    end;
    else
    do;
      logMsgPrimary=logMsgPrefix;
      logMsg=" List of groups was returned for &lsaf_path.." ;
      results.callIntMethod("getSize", rowCount);
      do row=0 to rowCount - 1;
        contextPath="&lsaf_path";
        results.callStringMethod("getName", row, name);
        results.callStringMethod("getDescription", row, description);
        output;
      end;
    end;

    %***********************************;
    %* Print messages in log           *;
    %***********************************;
    put;
    put logMsgPrimary " " logMsg;
    put logMsgPrefix " " %sysfunc(lowcase("&SYSMACRONAME"));
    if (rowCount=0) then
      put logMsgPrefix " No groups were found.";
    else
      put logMsgPrefix " " rowCount " records written to data set &sas_dsname";
    put logMsgPrefix " _lsafMsg_= " message ;
    put logMsgPrefix " _lsafRC_= " returnCode ;
    put;

    %***********************************;
    %* Set the macro variables         *;
    %***********************************;
    call symputx("_lsafRC_", returnCode);
    call symputx("_lsafMsg_", message);
    call symputx("_lsafRowCount_", rowCount);

  run;

  %***********************************;
  %* Sort the data set               *;
  %***********************************;
  %if (&_lsafRowCount_ > 0) %then
  %do;
    %let notesOption=%sysfunc(getoption(notes));
    options nonotes;
    proc sort data=&sas_dsname sortseq=linguistic(strength=primary);
      by name;
    run;
    options &notesOption;
  %end;

%mend lsaf_getgroups;
