%* lsaf_getgroupmembers                                                                          *;
%*                                                                                               *;
%* Gets the metadata for all of the members that are assigned to a group within the specified    *;
%* context. The metadata is stored in a SAS data set.                                            *;
%*                                                                                               *;
%* macrotype LSAF                                                                                *;
%* since 1.2                                                                                     *;
%* exposure external                                                                             *;
%*                                                                                               *;
%* History                                                                                       *;
%* 2013-08-30  updates for standardization and coding conventions                                *;
%* 2016-08-26  rebrand as LSAF                                                                   *;
%* 2019-01-21  updates for LSAF 5.1 functionality.                                               *;
%* 2019-11-01  update to handle a blank SAS output file.                                         *;
%* 2020-12-04  updates for service refactor                                                      *;
%* 2021-08-06  Remove header documentation                                                       *;
%* 2021-10-13  US11739                                                                           *;
%* 2021-10-19  principal refactor                                                                *;
%* 2021-12-08  2.6, DE17193 add group name to data set                                           *;
%*                                                                                               *;

%macro lsaf_getgroupmembers(
  lsaf_path=,
  lsaf_group=,
  sas_dsname=work.lsafGetGroupMembers
  ) /des='Get a list of the members of a group in an LSAF context.';
  
  %if "&sas_dsname" EQ "" %then %let sas_dsname=work.lsafGetGroupMembers;

  %****************************************;
  %* Initialize macro variables           *;
  %****************************************;
  %lsaf_getProductName();

  %global _lsafMsg_
          _lsafRC_
  ;

  %local _lsafRowCount_;

  %let _lsafRC_ =%str(-999);
  %let _lsafMsg_=%str(The SAS Macro did not execute properly.  Unknown err%str(or).);
  %let _lsafRowCount_=0;

  %**********************************************;
  %*  Start the datastep to call the java code  *;
  %**********************************************;
  data &sas_dsname (label="Created by &SYSMACRONAME"
                    keep=contextPath group type principalId description grpSrcCtxt);
    attrib returnCode      length=8      
           message         length=$200      
           logMsgPrefix    length=$70      
           logMsgPrimary   length=$70      
           logMsg          length=$300      
           rowCount        length=8      

           contextPath     length=$2048     label="Context Path"       
           group           length=$256      label="Group Name"      
           type            length=$32       label="Principal Type"      
           principalId     length=$256      label="Principal Name"      
           description     length=$128      label="Principal Description"      
           grpSrcCtxt      length=$2048     label="Principal Group Context Source Path"
    ;

    rowCount=0;

    %***********************************;
    %* Declare the java objects        *;
    %***********************************;
    declare javaobj srv("com/sas/lsaf/macro/security/group/GroupService");
    declare javaobj results("com/sas/lsaf/macro/security/SasPrincipalsResult");

    %***********************************;
    %* Attempt the operation           *;
    %***********************************;
    srv.callStaticVoidMethod("getMembers", "%bquote(&lsaf_path)", "%bquote(&lsaf_group)", results);

    %***********************************;
    %* Retrieve the results            *;
    %***********************************;
    results.callIntMethod("getReturnCode", returnCode);
    results.callStringMethod("getReturnMessage", message);

    %***********************************;
    %* Process the results             *;
    %***********************************;
    logMsgPrefix="NOTE: &_LSAF_PRODUCT_NAME_ Macro: * ";
    if (returnCode ne 0) then
    do;
      logMsgPrimary="ERR%str(OR): &_LSAF_PRODUCT_NAME_ Macro:";
      logMsg=" Cannot retrieve members from &lsaf_group in &lsaf_path.." ;
    end;
    else
    do;
      logMsgPrimary=logMsgPrefix;
      logMsg=" Members were returned for &lsaf_group for &lsaf_path.." ;
      results.callIntMethod("getSize", rowCount);
      do row=0 to rowCount - 1;
        contextPath="&lsaf_path";
        group="&lsaf_group";
        results.callStringMethod("getType", row, type);
        results.callStringMethod("getName", row, principalId);
        results.callStringMethod("getDescription", row, description);
        results.callStringMethod("getContextPath", row, grpSrcCtxt);
        output;
      end;
    end;

    %***********************************;
    %* Print messages in log           *;
    %***********************************;
    put;
    put logMsgPrimary " " logMsg;
    put logMsgPrefix " " %sysfunc(lowcase("&SYSMACRONAME"));
    if (rowCount=0) then
      put logMsgPrefix " No members were found.";
    else
      put logMsgPrefix " " rowCount " records written to data set &sas_dsname";
    put logMsgPrefix " _lsafMsg_= " message ;
    put logMsgPrefix " _lsafRC_= " returnCode ;
    put;

    %***********************************;
    %* Set the macro variables         *;
    %***********************************;
    call symputx("_lsafRC_", returnCode);
    call symputx("_lsafMsg_", message);
    call symputx("_lsafRowCount_", rowCount);

   run;

  %***********************************;
  %* Sort the data set               *;
  %***********************************;
  %if (&_lsafRowCount_ > 0) %then
  %do;
    %let notesOption=%sysfunc(getoption(notes));
    options nonotes;
    proc sort data=&sas_dsname sortseq=linguistic(strength=primary);
      by type principalId;
    run;
    options &notesOption;
  %end;

%mend lsaf_getgroupmembers;
