%* lsaf_getglobaltlftemplates                                                                       *;
%*                                                                                                  *;
%* Gets the metadata for the global Tables, Listing, and Figure templates. The metadata are stored  *;
%* in a SAS data set.                                                                               *;
%*                                                                                                  *;
%* macrotype LSAF                                                                                   *;
%* since 2.5                                                                                        *;
%* exposure external                                                                                *;
%*                                                                                                  *;
%* History                                                                                          *;
%* 2021-06-09  initial coding                                                                       *;
%* 2021-08-17  Remove header documentation                                                          *;
%*                                                                                                  *;

%macro lsaf_getglobaltlftemplates(
  sas_dsname=work.lsafgetglobaltlftemplates
  ) /des='Gets the metadata for the global Tables, Listing, and Figure templates';
  
  %if "&sas_dsname" EQ "" %then %let sas_dsname=work.lsafgetglobaltlftemplates;

  %****************************************;
  %* Initialize macro variables           *;
  %****************************************;
  %lsaf_getProductName();

  %global _lsafMsg_
          _lsafRC_
  ;

  %local _lsafRowCount_;

  %let _lsafRC_ =%str(-999);
  %let _lsafMsg_=%str(The SAS Macro did not execute properly.  Unknown err%str(or).);
  %let _lsafRowCount_=0;

  %**********************************************;
  %* Start the datastep to call the java code   *;
  %**********************************************;
  data &sas_dsname (label="Created by &SYSMACRONAME"
                    keep=id name description createdBy createdDate createdDateSASFormat lastModifiedBy 
                         lastModifiedDate lastModifiedDateSASFormat);

    attrib returnCode                 length=8;
    attrib message                    length=$200;
    attrib logMsgPrefix               length=$70;
    attrib logMsgPrimary              length=$70;
    attrib logMsg                     length=$300;
    attrib rowCount                   length=8;

    attrib id                         length=$70    label="TLF Template ID";
    attrib name                       length=$70    label="TLF Template Name";
    attrib description                length=$2000  label="TLF Template Description";
    attrib createdBy                  length=$64    label="The ID of the User who Created the TLF Template";
    attrib createdDate                length=$64    label="TLF Template Creation Date";
    attrib createdDateSASFormat       length=$64    label="TLF Template Creation Date (SAS Format)";
    attrib lastModifiedBy             length=$64    label="The ID of the User who Last Modified the TLF Template";
    attrib lastModifiedDate           length=$64    label="TLF Template Modified Date";
    attrib lastModifiedDateSASFormat  length=$64    label="TLF Template Modified Date (SAS Format)";
    
    rowCount=0;

    %***********************************;
    %* Declare the java objects        *;
    %***********************************;
    declare javaobj srv("com/sas/lsaf/macro/clinical/tlf/TlfTemplateService");
    declare javaobj results("com/sas/lsaf/macro/clinical/tlf/SasTlfTemplatesResult");

    %***********************************;
    %* Attempt the operation           *;
    %***********************************;
    srv.callStaticVoidMethod("getTlfTemplates", results);

    %***********************************;
    %* Retrieve the results            *;
    %***********************************;
    results.callIntMethod("getReturnCode", returnCode);
    results.callStringMethod("getReturnMessage", message);

    %***********************************;
    %* Process the results             *;
    %***********************************;
    logMsgPrefix="NOTE: &_LSAF_PRODUCT_NAME_ Macro: * ";
    if (returnCode ne 0) then
    do;
      logMsgPrimary="ERR%str(OR): &_LSAF_PRODUCT_NAME_ Macro:";
      logMsg=" Cannot retrieve global TLF templates.";
    end;
    else
    do;
      logMsgPrimary=logMsgPrefix;
      logMsg=" Global TLF templates were returned.";
      results.callIntMethod("getSize", rowCount);
      do row=0 to rowCount - 1;
        results.callStringMethod("getId", row, id);
        results.callStringMethod("getName", row, name);
        results.callStringMethod("getDescription", row, description);
        results.callStringMethod("getCreatedBy", row, createdBy);
        results.callStringMethod("getCreated", row, createdDate);
        results.callStringMethod("getCreatedSasFormat", row, createdDateSASFormat);
        results.callStringMethod("getLastModifiedBy", row, lastModifiedBy);
        results.callStringMethod("getLastModified", row, lastModifiedDate);
        results.callStringMethod("getLastModifiedSasFormat", row, lastModifiedDateSASFormat);
        output;
      end;
    end;

    %***********************************;
    %* Print messages in log           *;
    %***********************************;
    put;
    put logMsgPrimary " " logMsg;
    put logMsgPrefix " " %sysfunc(lowcase("&SYSMACRONAME"));
    put logMsgPrefix " " rowCount " records written to data set &sas_dsname";
    put logMsgPrefix " _lsafMsg_= " message ;
    put logMsgPrefix " _lsafRC_= " returnCode ;
    put;

    %***********************************;
    %* Set the macro variables         *;
    %***********************************;
    call symputx("_lsafRC_", returnCode);
    call symputx("_lsafMsg_", message);
    call symputx("_lsafRowCount_", rowCount);

  run;

%mend lsaf_getglobaltlftemplates;
