%* lsaf_getglobalprivileges                                                                         *;
%*                                                                                                  *;
%* Gets the metadata for all of the global privileges that are available. The metadata is stored    *;
%* in a SAS data set.                                                                               *;
%*                                                                                                  *;
%* macrotype LSAF                                                                                   *;
%* since 1.3                                                                                        *;
%* exposure external                                                                                *;
%*                                                                                                  *;
%* History                                                                                          *;
%* 2013-11-01  initial coding                                                                       *;
%* 2014-04-21  updated documentation                                                                *;
%* 2016-08-26  rebrand as LSAF                                                                      *;
%* 2019-11-01  update to handle a blank SAS output file.                                            *;
%* 2020-07-13  update for refactor of java service hierarchy.                                       *;
%* 2021-08-05  Remove header documentation                                                          *;
%* 2021-10-13  US11739                                                                              *;
%*                                                                                                  *;

%macro lsaf_getglobalprivileges(
  sas_dsname=work.lsafGetGlobalPrivileges
  )  /des='Get a list of all global privileges.';
  
  %if "&sas_dsname" EQ "" %then %let sas_dsname=work.lsafGetGlobalPrivileges;

  %****************************************;
  %* Initialize macro variables           *;
  %****************************************;
  %lsaf_getProductName();

  %global _lsafMsg_
          _lsafRC_
  ;

  %local _lsafRowCount_;

  %let _lsafRC_ =%str(-999);
  %let _lsafMsg_=%str(The SAS Macro did not execute properly.  Unknown err%str(or).);
  %let _lsafRowCount_=0;

  %**********************************************;
  %* Start the datastep to call the java code   *;
  %**********************************************;
  data &sas_dsname (label="Created by &SYSMACRONAME"
                    keep=privilegeType privilegeId privilegeName);
    attrib returnCode      length=8;
    attrib message         length=$200;
    attrib logMsgPrefix    length=$70;
    attrib logMsgPrimary   length=$70;
    attrib logMsg          length=$300;
    attrib rowCount        length=8;

    attrib privilegeType   length=$20      label="Privilege Type" ;
    attrib privilegeId     length=$64      label="Privilege Identifier";
    attrib privilegeName   length=$128     label="Privilege Name";

    rowCount=0;

    %***********************************;
    %* Declare the java objects        *;
    %***********************************;
    declare javaobj srv("com/sas/lsaf/macro/security/privilege/global/GlobalPrivilegeService");
    declare javaobj results("com/sas/lsaf/macro/security/privilege/SasPrivilegesResult");

    %***********************************;
    %* Attempt the operation           *;
    %***********************************;
    srv.callStaticVoidMethod("getGlobalPrivileges", results);

    %***********************************;
    %* Retrieve the results            *;
    %***********************************;
    results.callIntMethod("getReturnCode", returnCode);
    results.callStringMethod("getReturnMessage", message);

    %***********************************;
    %* Process the results             *;
    %***********************************;
    logMsgPrefix="NOTE: &_LSAF_PRODUCT_NAME_ Macro: * ";
    if (returnCode ne 0) then
    do;
      logMsgPrimary="ERR%str(OR): &_LSAF_PRODUCT_NAME_ Macro:";
      logMsg=" Cannot retrieve list of global privileges." ;
    end;
      else
    do;
      logMsgPrimary=logMsgPrefix;
      logMsg=" List of global privileges were returned." ;
      results.callStringMethod("getType", privilegeType);
      results.callIntMethod("getSize", rowCount);
      do row=0 to rowCount - 1;
        results.callStringMethod("getId", row, privilegeId);
        results.callStringMethod("getName", row, privilegeName);
        output;
      end;
    end;

    %***********************************;
    %* Print messages in log           *;
    %***********************************;
    put;
    put logMsgPrimary " " logMsg;
    put logMsgPrefix " " %sysfunc(lowcase("&SYSMACRONAME"));
    put logMsgPrefix " " rowCount " records written to data set &sas_dsname";
    put logMsgPrefix " _lsafMsg_= " message ;
    put logMsgPrefix " _lsafRC_= " returnCode ;
    put;

    %***********************************;
    %* Set the macro variables         *;
    %***********************************;
    call symputx("_lsafRC_", returnCode);
    call symputx("_lsafMsg_", message);
    call symputx("_lsafRowCount_", rowCount);

  run;

  %***********************************;
  %* Sort by privilege id            *;
  %***********************************;
  %if (&_lsafRowCount_ > 0) %then
  %do;
    %let notesOption=%sysfunc(getoption(notes));
    options nonotes;
    proc sort data=&sas_dsname sortseq=linguistic(strength=primary);
      by privilegeId;
    run;
    options &notesOption;
  %end;

%mend lsaf_getglobalprivileges;
