%* lsaf_getexternaldictionaries                                                                      *;
%*                                                                                                   *;
%* Gets the metadata for all of the external dictionaries. The metadata is stored in a SAS data set. *;
%*                                                                                                   *;
%* macrotype LSAF                                                                                    *;
%* since 2.2                                                                                         *;
%* exposure external                                                                                 *;
%*                                                                                                   *;
%* History                                                                                           *;
%* 2019-07-08  initial coding                                                                        *;
%* 2019-11-01  update to handle a blank SAS output file.                                             *;
%* 2020-07-14  update for refactor of java service hierarchy.                                        *;
%* 2021-08-05  Remove header documentation                                                           *;
%* 2021-10-12  US11739                                                                               *;
%*                                                                                                   *;

%macro lsaf_getexternaldictionaries(
  sas_dsname=work.lsafgetexternaldictionaries
  ) /des='Get a list of all external dictionaries in the system.';

  %if "&sas_dsname" EQ "" %then %let sas_dsname=work.lsafgetexternaldictionaries;

  %****************************************;
  %* Initialize macro variables           *;
  %****************************************;
  %lsaf_getProductName();

  %global _lsafMsg_
          _lsafRC_
  ;

  %local _lsafRowCount_;

  %let _lsafRC_ =%str(-999);
  %let _lsafMsg_=%str(The SAS Macro did not execute properly.  Unknown err%str(or).);
  %let _lsafRowCount_=0;

  %**********************************************;
  %* Start the datastep to call the java code   *;
  %**********************************************;
  data &sas_dsname (label="Created by &SYSMACRONAME"
                    keep= id dictionary version name description dataType shortName href
                          createdDate createdDateSASFormat createdBy associatedStudiesCount
                          lastModifiedDate lastModifiedDateSASFormat lastModifiedBy
  );

    attrib returnCode                length=8;
    attrib message                   length=$200;
    attrib logMsgPrefix              length=$70;
    attrib logMsgPrimary             length=$70;
    attrib logMsg                    length=$300;
    attrib rowCount                  length=8;

    attrib id                        length=$64    label="ID";
    attrib name                      length=$128   label="Name";
    attrib description               length=$64    label="Description";
    attrib dataType                  length=$64    label="Data Type";
    attrib shortName                 length=$64    label="Short Name";
    attrib dictionary                length=$64    label="Dictionary";
    attrib version                   length=$64    label="Version";
    attrib href                      length=$64    label="HRef";
    attrib createdDate               length=$64    label="Date Created";
    attrib createdDateSASFormat      length=$64    label="Date Created (SAS Format)";
    attrib createdBy                 Length=$64    Label="Created by";
    attrib associatedStudiesCount    length=8      label="Associated Studies Count";
    attrib lastModifiedDate          length=$64    label="Date Last Modified";
    attrib lastModifiedDateSASFormat length=$64    label="Date Last Modified (SAS Format)";
    attrib lastModifiedBy            length=$64    label="Last Modified By";    

    rowCount=0;

    %***********************************;
    %* Declare the java objects        *;
    %***********************************;
    declare javaobj     srv("com/sas/lsaf/macro/clinical/externaldictionary/ExternalDictionaryService");
    declare javaobj results("com/sas/lsaf/macro/clinical/externaldictionary/SasExternalDictionariesResult");

    %***********************************;
    %* Attempt the operation           *;
    %***********************************;
    srv.callStaticVoidMethod("getExternalDictionaries", results);

    %***********************************;
    %* Retrieve the results            *;
    %***********************************;
    results.callIntMethod("getReturnCode", returnCode);
    results.callStringMethod("getReturnMessage", message);

    %***********************************;
    %* Process the results             *;
    %***********************************;
    logMsgPrefix="NOTE: &_LSAF_PRODUCT_NAME_ Macro: * ";
    if (returnCode ne 0) then
    do;
      logMsgPrimary="ERR%str(OR): &_LSAF_PRODUCT_NAME_ Macro:";
      logMsg=" Cannot retrieve list of external dictionaries.";
    end;
    else
    do;
      logMsgPrimary=logMsgPrefix;
      logMsg=" List of external dictionaries was returned.";
      results.callIntMethod("getSize", rowCount);
      do row=0 to rowCount - 1;
        results.callStringMethod("getId", row, id);
        results.callStringMethod("getName", row, name);
        results.callStringMethod("getDescription", row, description);
        results.callStringMethod("getDataType", row, dataType);
        results.callStringMethod("getShortName", row, shortName);
        results.callStringMethod("getDictionary", row, dictionary);
        results.callStringMethod("getVersion", row, version);
        results.callStringMethod("getHRef", row, href);
        results.callStringMethod("getCreated", row, createdDate);
        results.callStringMethod("getCreatedSasFormat", row, createdDateSasFormat);
        results.callStringMethod("getCreatedBy", row, createdBy);
        results.callLongMethod("getAssociatedStudiesCount", row, associatedStudiesCount);
        results.callStringMethod("getLastModified", row, lastModifiedDate);
        results.callStringMethod("getLastModifiedSasFormat", row, lastModifiedDateSASFormat);
        results.callStringMethod("getLastModifiedBy", row, lastModifiedBy);
        output;
      end;
    end;

    %***********************************;
    %* Print messages in log           *;
    %***********************************;
    put;
    put logMsgPrimary " " logMsg;
    put logMsgPrefix " " %sysfunc(lowcase("&SYSMACRONAME"));
    if (rowCount<=0) 
      then put logMsgPrefix " No external dictionaries were found.";
      else put logMsgPrefix " " rowCount " records written to data set &sas_dsname";
    put logMsgPrefix " _lsafMsg_= " message ;
    put logMsgPrefix " _lsafRC_= " returnCode ;
    put;

    %***********************************;
    %* Set the macro variables         *;
    %***********************************;
    call symputx("_lsafRC_", returnCode);
    call symputx("_lsafMsg_", message);
    call symputx("_lsafRowCount_", rowCount);

  run;

  %***********************************;
  %* Sort the data set               *;
  %***********************************;
  %if (&_lsafRowCount_ > 0) %then
  %do;
    %let notesOption=%sysfunc(getoption(notes));
    options nonotes;
    proc sort data=&sas_dsname sortseq=linguistic(strength=primary);
      by name;
    run;
    options &notesOption;
  %end;

%mend lsaf_getexternaldictionaries;