%* lsaf_getcontexttypes                                                                            *;                                                       
%*                                                                                                 *;                                                 
%* Gets the metadata for all of the context types that are defined in the repository. The metadata *;
%* is stored in a SAS data set.                                                                    *;
%*                                                                                                 *;                                                 
%* macrotype LSAF                                                                                  *;                                                
%* since 2.1                                                                                       *;                                                 
%* exposure external                                                                               *;                                                 
%*                                                                                                 *;                                                 
%* History                                                                                         *;
%* 2019-11-01  update to handle a blank SAS output file.                                           *;
%* 2020-07-13  update for refactor of java service hierarchy.                                      *;
%* 2021-08-17  Remove header documentation                                                         *;
%* 2021-09-07  Update proc sort code (DE16792)                                                     *;
%* 2021-09-20  Update allowableChildren variable length (DE16993)                                  *;
%*                                                                                                 *;

%macro lsaf_getcontexttypes(                                                                                                                              
  sas_dsname=work.lsafGetContextTypes                                                                                                                     
  ) /des='Get a list of all context types defined in the system.';                                                                                            
  
  %if "&sas_dsname" EQ "" %then %let sas_dsname=work.lsafGetContextTypes;
                                                                                                                                                      
  %****************************************;                                                                                                          
  %* Initialize macro variables           *;                                                                                                          
  %****************************************;                                                                                                          
  %lsaf_getProductName();

  %global _lsafMsg_                                                                                                                                   
          _lsafRC_                                                                                                                                    
  ;                                                                                                                                                   
                                                                                                                                                      
  %local _lsafRowCount_;                                                                                                                              
                                                                                                                                                      
  %let _lsafRC_ =%str(-999);                                                                                                                          
  %let _lsafMsg_=%str(The SAS Macro did not execute properly.  Unknown err%str(or).);                                                                 
  %let _lsafRowCount_=0;                                                                                                                              
                                                                                                                                                      
  %**********************************************;                                                                                                    
  %* Start the datastep to call the java code   *;                                                                                                    
  %**********************************************;                                                                                                    
  data &sas_dsname (label="Created by &SYSMACRONAME"                                                                                                  
                    keep= id name description lsaftype allowablechildren capabilities listType);                                                                                                                                                  
                                                                                                                                                      
    attrib returnCode              length=8;                                                                                                          
    attrib message                 length=$200;                                                                                                       
    attrib logMsgPrefix            length=$70;                                                                                                        
    attrib logMsgPrimary           length=$70;                                                                                                        
    attrib logMsg                  length=$300;                                                                                                       
    attrib rowCount                length=8;                                                                                                          
             
    attrib listType                length=$32    label="LSAF List Type";   
    attrib id                      length=$64    label="Context ID";                                                                                     
    attrib name                    length=$128   label="Context Name";                                                                                
    attrib description             length=$1024  label="Context Description";                                                                                  
    attrib lsafType                length=$64    label="LSAF Context Type";                                                                                  
    attrib allowableChildren       length=$2048  label="Allowable Children Type";                                                                                   
    attrib capabilities            length=$128   label="Capabilities";                                                                              
                                                                                                                                                      
    rowCount=0;                                                                                                                                       
                                                                                                                                                      
    %***********************************;                                                                                                             
    %* Declare the java objects        *;                                                                                                             
    %***********************************;                                                                                                             
    declare javaobj srv("com/sas/lsaf/macro/core/type/TypeService");                                                                                  
    declare javaobj results("com/sas/lsaf/macro/core/type/SasTypesResult");                                                                            
                                                                                                                                                      
    %***********************************;                                                                                                             
    %* Attempt the operation           *;                                                                                                             
    %***********************************;                                                                                                             
    srv.callStaticVoidMethod("getContextTypes", results);                                                                                                 
                                                                                                                                                      
    %***********************************;                                                                                                             
    %* Retrieve the results            *;                                                                                                             
    %***********************************;                                                                                                             
    results.callIntMethod("getReturnCode", returnCode);                                                                                               
    results.callStringMethod("getReturnMessage", message);                                                                                            
                                                                                                                                                      
    %***********************************;                                                                                                             
    %* Process the results             *;                                                                                                             
    %***********************************;
    logMsgPrefix="NOTE: &_LSAF_PRODUCT_NAME_ Macro: * ";
    if (returnCode ne 0) then                                                                                                                         
    do;                                                                                                                                               
      logMsgPrimary="ERR%str(OR): &_LSAF_PRODUCT_NAME_ Macro:";                                                                                        
      logMsg=" Cannot retrieve list of context types.";                                                                                                       
    end;                                                                                                                                              
    else                                                                                                                                              
    do;                                                                                                                                               
      logMsgPrimary=logMsgPrefix;                                                                                            
      logMsg=" List of context types was returned.";                                                                                                          
      results.callIntMethod("getSize", rowCount);                                                                                                     
      results.callStringMethod("getListType", listType);                                                             
      do row=0 to rowCount - 1;                                                                                                                       
        results.callStringMethod("getId", row, id);                                                             
        results.callStringMethod("getName", row, name);                                                             
        results.callStringMethod("getDescription", row, description);                                                         
        results.callStringMethod("getLsafType", row, lsafType);                                                                         
        results.callStringMethod("getAllowableChildTypes", row, allowableChildren);                                                                     
        results.callStringMethod("getCapabilities", row, capabilities);                                                                         
        output;                                                                                                                                       
      end;                                                                                                                                            
    end;                                                                                                                                              
                                                                                                                                                      
    %***********************************;                                                                                                             
    %* Print messages in log           *;                                                                                                             
    %***********************************;                                                                                                             
    put;                                                                                                                                              
    put logMsgPrimary " " logMsg;                                                                                                                      
    put logMsgPrefix " " %sysfunc(lowcase("&SYSMACRONAME"));                                                                                          
    if (rowCount=0) 
      then put logMsgPrefix " No context types were found.";                                                                                                       
      else put logMsgPrefix " " rowCount " records written to data set &sas_dsname";                                                                       
    put logMsgPrefix " _lsafMsg_= " message ;                                                                                                         
    put logMsgPrefix " _lsafRC_= " returnCode ;                                                                                                       
    put;                                                                                                                                              
                                                                                                                                                      
    %***********************************;                                                                                                             
    %* Set the macro variables         *;                                                                                                             
    %***********************************;                                                                                                             
    call symputx("_lsafRC_", returnCode);                                                                                                             
    call symputx("_lsafMsg_", message);                                                                                                               
    call symputx("_lsafRowCount_", rowCount);                                                                                                         
                                                                                                                                                      
  run;
  
  %***********************************;
  %* Sort the data set               *;                                                                                                               
  %***********************************;                                                                                                               
  %if (&_lsafRowCount_ > 0) %then                                                                                                                     
  %do;                                                                                                                                                
    %let notesOption=%sysfunc(getoption(notes));                                                                                                        
    options nonotes;                                                                                                                                    
    proc sort data=&sas_dsname sortseq=linguistic(strength=primary);                                                                                                                       
      by id;                                                                                                                                    
    run;                                                                                                                                              
    options &notesOption;                                                                                                                               
  %end;                                                                                                                                               
                                                                                                                                                      
%mend lsaf_getcontexttypes;
