%* lsaf_getchildren                                                                              *;
%*                                                                                               *;
%* Gets the metadata for the items that are within a container in the repository. The metadata   *;
%* is stored in a SAS data set. To retrieve the children of the root level, call the             *;
%* lsaf_gettopcontexts macro.                                                                    *;
%*                                                                                               *;
%* macrotype LSAF                                                                                *;
%* since 1.1                                                                                     *;
%* exposure external                                                                             *;
%*                                                                                               *;
%* History                                                                                       *;
%* 2013-06-28  version 1.2 update - path length                                                  *;
%* 2013-08-30  updates for standardization and coding conventions                                *;
%* 2014-05-07  updated documentation and fixed data set processing.                              *;
%* 2016-08-29  rebrand as LSAF                                                                   *;
%* 2019-08-15  documentation updated to reference lsaf_getTopLevelContexts.                      *;
%* 2019-11-01  update to handle a blank SAS output file.                                         *;
%* 2020-11-11  add formattedSize and update to set a -1 value for file size to missing.          *;
%* 2021-08-13  Remove header documentation                                                       *;
%* 2021-08-31  DE16792                                                                           *;
%* 2021-09-20  Remove parameter defaults (DE16775)                                               *;
%* 2021-10-12  US11739                                                                           *;
%* 2024-07-26  2.7: Notes option updates.                                                        *;
%*                                                                                               *;

%macro lsaf_getchildren(
  lsaf_path=,
  sas_dsname=work.lsafGetChildren,
  lsaf_recursive=
  ) /des='Get a list of the items in a folder in SAS Life Science Analytics Framework.';
  
  %if "&sas_dsname" EQ "" %then %let sas_dsname=work.lsafGetChildren;

  %****************************************;
  %* Initialize macro variables           *;
  %****************************************;
  %lsaf_getProductName();

  %global _lsafMsg_
          _lsafRC_
  ;

  %let _lsafRC_ =%str(-999);
  %let _lsafMsg_=%str(The SAS Macro did not execute properly.  Unknown err%str(or).);

  %local _lsafRowCount_
         notesOption
  ;

  %let _lsafRowCount_=0;

  %**********************************************;
  %* Start the datastep to call the java code   *;
  %**********************************************;
  data &sas_dsname (label="Created by &SYSMACRONAME"
                    keep=path name description itemType isContainer
                         description version isVersioned isCheckedOut isLocked signatureStatus size formattedsize createdBy created dateCreated
                         lastModifiedBy lastModified dateLastModified propertiesLastModifiedBy propertiesLastModified datePropertiesLastModified
                         state signatureStatus);

    attrib returnCode                length=8
           message                   length=$200
           logMsgPrefix              length=$70
           logMsgPrimary             length=$70
           logMsg                    length=$300
           rowCount                  length=8

           path                      length=$2048    label="Item Path"
           name                      length=$256     label="Item Name"
           itemType                  length=$32      label="Item Type"
           isContainer               length=8        label="Is Item A Container?"

           description               length=$512     label="Description"
           version                   length=$32      label="Version Number"
           isVersioned               length=8        label="Is File Versioned"
           isCheckedOut              length=8        label="Is File Checked out"
           isLocked                  length=8        label="Is File Locked"
           signatureStatus           length=$32      label="Is File Signed"
           size                      length=8        label="Size In Bytes"
           formattedsize             length=$20      label="Size (Formatted)"

           createdBy                 length=$64      label="Userid that created the item"
           created                   length=$64      label="Item Creation Date"
           dateCreated                               label="Date the item was created (SAS DateTime Format)"

           lastModifiedBy            length=$64      label="Userid that last modified the item content"
           lastModified              length=$64      label="Last Content Modification Date"
           dateLastModified                          label="Date the item's content was last modified (SAS DateTime Format)"

           propertiesLastModifiedBy  length=$64      label="Userid that last modified the item properties"
           propertiesLastModified    length=$64      label="Last Properties Modification Date"
           datePropertiesLastModified                label="Date the item's properties were last modified (SAS DateTime Format)"

           state                     length=$64      label="State of an Analysis or Project"

           createdTempSAS                 length=$64 label="TEMPORARY - WILL BE DROPPED"
           lastModifiedTempSAS            length=$64 label="TEMPORARY - WILL BE DROPPED"
           propertiesLastModifiedTempSAS  length=$64 label="TEMPORARY - WILL BE DROPPED"
    ;

    format dateCreated
           dateLastModified
           datePropertiesLastModified datetime.
    ;

    rowCount=0;

    %***********************************;
    %* Declare the java items        *;
    %***********************************;
    declare javaobj srv("com/sas/lsaf/macro/content/repository/RepositoryService") ;
    declare javaobj results("com/sas/lsaf/macro/content/repository/SasRepositoryItemsResult");

    %***********************************;
    %* Attempt the operation           *;
    %***********************************;
    srv.callStaticVoidMethod("getChildren", "%bquote(&lsaf_path)", "%bquote(&lsaf_recursive)", results);

    %***********************************;
    %* Retrieve the results            *;
    %***********************************;
    results.callIntMethod("getReturnCode", returnCode);
    results.callStringMethod("getReturnMessage", message);

    %***********************************;
    %* Process the results             *;
    %***********************************;
    logMsgPrefix="NOTE: &_LSAF_PRODUCT_NAME_ Macro: * ";
    if (returnCode ne 0) then
    do;
      logMsgPrimary="ERR%str(OR): &_LSAF_PRODUCT_NAME_ Macro:";
      logMsg=" Could not retrieve list of items from %bquote(&lsaf_path)" ;
    end;
    else
    do;
      logMsgPrimary=logMsgPrefix;
      logMsg=" Returned a list of items in %bquote(&lsaf_path)";
      results.callIntMethod("getSize", rowCount);
      do row=0 to rowCount - 1;
        results.callStringMethod("getPath", row, path);
        results.callStringMethod("getName", row, name);
        results.callStringMethod("getDescription", row, description);
        results.callStringMethod("getVersion", row, version);
        results.callIntMethod("isContainer", row, isContainer);
        results.callIntMethod("isVersioned", row, isVersioned);
        results.callIntMethod("isCheckedOut", row, isCheckedOut);
        results.callIntMethod("isLocked", row, isLocked);
        results.callStringMethod("getSignatureStatus", row, signatureStatus);
        results.callLongMethod("getFileSize", row, size);
        if size < 0 then size=.;
        results.callStringMethod("getFormattedFileSize", row, formattedsize); 
        results.callStringMethod("getItemType", row, itemType);
        results.callStringMethod("getCreated", row, created);
        results.callStringMethod("getCreatedBy", row, createdBy);
        results.callStringMethod("getLastModified", row, lastModified);
        results.callStringMethod("getLastModifiedBy", row, lastModifiedBy);
        results.callStringMethod("getPropertiesLastModified", row, propertiesLastModified);
        results.callStringMethod("getPropertiesLastModifiedBy", row, propertiesLastModifiedBy);
        results.callStringMethod("getState", row, state);

        results.callStringMethod("getCreatedSasFormat", row, createdTempSas);
        results.callStringMethod("getLastModifiedSasFormat", row, lastModifiedTempSas);
        results.callStringMethod("getPropertiesLastModifiedSasFormat", row, propertiesLastModifiedTempSAS);

        dateCreated=input(createdTempSAS, datetime.);
        dateLastModified=input(lastModifiedTempSas, datetime.);
        datePropertiesLastModified=input(propertiesLastModifiedTempSAS, datetime.);

        output;
      end;
    end;

    %***********************************;
    %* Print messages in log           *;
    %***********************************;
    put;
    put logMsgPrimary " " logMsg;
    put logMsgPrefix " " %sysfunc(lowcase("&SYSMACRONAME"));
    if (rowCount=0) then 
       put logMsgPrefix " No children were found.";
    else 
       put logMsgPrefix " " rowCount " records written to data set &sas_dsname";
    put logMsgPrefix " _lsafMsg_= " message ;
    put logMsgPrefix " _lsafRC_= " returnCode ;
    put logMsgPrefix " lsaf_recursive= &lsaf_recursive" ;
    put;

    %***********************************;
    %* Set the macro variables         *;
    %***********************************;
    call symputx("_lsafRC_", returnCode);
    call symputx("_lsafMsg_", message);
    call symputx("_lsafRowCount_", rowCount);

  run;

  %***********************************;
  %* Sort the data set               *;
  %***********************************;
  %if (&_lsafRowCount_ > 0) %then
  %do;
    %let notesOption=%sysfunc(getoption(notes));
    options nonotes;
    proc sort data=&sas_dsname sortseq=linguistic(strength=primary);
      by path;
    run;
    options &notesOption.;
  %end;

%mend lsaf_getchildren;
