%* lsaf_getcheckedoutfiles                                                                         *;
%*                                                                                                 *;
%* Gets the checked out file information for the specified user,file, or location within the       *;
%* repository. The information is stored in a SAS data set.                                        *;
%*                                                                                                 *;
%* macrotype LSAF                                                                                  *;
%* since 1.5                                                                                       *;
%* exposure external                                                                               *;
%*                                                                                                 *;
%* History                                                                                         *;
%* 2016-12-16  initial coding.                                                                     *;
%* 2019-11-01  update to handle a blank SAS output file.                                           *;
%* 2021-08-13  Remove header documentation                                                         *;
%*                                                                                                 *;

%macro lsaf_getcheckedoutfiles(
   lsaf_checkedoutby=%str(_LSAF_VALUE_NOT_PROVIDED_),
   lsaf_name=%str(_LSAF_VALUE_NOT_PROVIDED_),
   lsaf_location=%str(_LSAF_VALUE_NOT_PROVIDED_),
   sas_dsname=work.lsafGetCheckedOutFiles,
   lsaf_printLimitReachedWarning=1
   ) /des='Gets a list of the checked out files.';
  
   %if "&sas_dsname" EQ "" %then %let sas_dsname=work.lsafGetCheckedOutFiles;

   %****************************************;
   %* Initialize macro variables           *;
   %****************************************;
   %lsaf_getProductName();

   %global _lsafMsg_
           _lsafRC_
           _lsaf_printmessage_
           _lsafCOFSearchLimitReached_
   ;

   %let _lsafRC_ =%str(-999);
   %let _lsafMsg_=%str(The SAS Macro did not execute properly.  Unknown err%str(or).);

   %**********************************************;
   %* Start the datastep to call the java code   *;
   %**********************************************;
   data &sas_dsname (label="Created by &SYSMACRONAME"
                     keep=name location size formattedsize checkedoutby modifieddate version status
   );

   attrib
          rowCount        length=8
          returnCode      length=8
          message         length=$200
          logMsgPrefix    length=$70
          logMsg          length=$300
          logMsgPrimary   length=$200
          limitReached    length=8

          name            length=$256   label="File name"
          location        length=$2048  label="File location within LSAF"
          size            length=8      label="File size"
          formattedsize   length=$32    label="Formatted file size"
          checkedoutby    length=$64    label="Checked out by"
          modifieddate    length=$64    label="Date last modified"
          version         length=$32    label="File version"
          status          length=$32    label="File workspace status"
   ;

   rowCount=0;

   %***********************************;
   %* Declare the java objects        *;
   %***********************************;
   declare javaobj srv("com/sas/lsaf/macro/content/repository/RepositoryService") ;
   declare javaobj results("com/sas/lsaf/macro/content/repository/SasCheckedOutFilesResult");

   %***********************************;
   %* Attempt the operation           *;
   %***********************************;
   srv.callStaticVoidMethod("getCheckedOutFiles", "%bquote(&lsaf_checkedoutby)", "%bquote(&lsaf_name)", "%bquote(&lsaf_location)", results);

   %***********************************;
   %* Retrieve the results            *;
   %***********************************;
   results.callIntMethod("getReturnCode", returnCode);
   results.callStringMethod("getReturnMessage", message);
   results.callBooleanMethod("isLimitReached", limitReached);

   %***********************************;
   %* Process the results             *;
   %***********************************;
   logMsgPrefix="NOTE: &_LSAF_PRODUCT_NAME_ Macro: * ";
   if (returnCode ne 0) then
   do;
     logMsgPrimary="ERR%str(OR): &_LSAF_PRODUCT_NAME_ Macro:";
     logMsg=" Cannot retrieve checked out files.";
   end;
   else
   do;
     results.callIntMethod("getSize", rowCount);
     if limitReached then
     do;
       %lsaf_printwarningmessage(lsaf_printmessage=&lsaf_printlimitreachedwarning);
       logMsgPrimary="&_lsaf_PrintMessage_";
       logMsg=" Checked out files information written to data set &sas_dsname";
     end;
     else 
     do;
       logMsgPrimary=logMsgPrefix;
       if (rowCount=0) 
         then logMsg = " No checked out files information was found.";
         else logMsg = " Checked out files information written to data set &sas_dsname";
     end;

     do row=0 to rowCount - 1;
       results.callStringMethod("getName", row, name);
       results.callStringMethod("getLocation", row, location);
       results.callLongMethod("getFileSize", row, size);
       results.callStringMethod("getFormattedSize", row, formattedsize);
       results.callStringMethod("getCheckedOutBy", row, checkedoutby);
       results.callStringMethod("getModifiedDate", row, modifieddate);
       results.callStringMethod("getVersion", row, version);
       results.callStringMethod("getWsStatus", row, status);
       output;
     end;
   end;

   %***********************************;
   %* Print messages in log           *;
   %***********************************;
   put;
   put logMsgPrimary " " logMsg;
   put logMsgPrefix " " %sysfunc(lowcase("&SYSMACRONAME"));
   put logMsgPrefix " _lsafMsg_= " message ;
   put logMsgPrefix " _lsafRC_= " returnCode ;
   put logMsgPrefix " _lsafCOFSearchLimitReached_= " limitReached;
   put;

   %***********************************;
   %* Set the macro variables         *;
   %***********************************;
   call symputx("_lsafRC_", returnCode);
   call symputx("_lsafMsg_", message);
   call symputx("_lsafCOFSearchLimitReached_", limitReached);

   run;

%mend lsaf_getcheckedoutfiles;
