%* lsaf_getauditactions                                                                             *;
%*                                                                                                  *;
%* Gets the metadata for the auditable actions that are applicable to the specified object type or  *;
%* all of the auditable actions for the system, if no object type is specified.                     *;
%*                                                                                                  *;
%* macrotype LSAF                                                                                   *;
%* since 2.3                                                                                        *;
%* exposure external                                                                                *;
%*                                                                                                  *;
%* History                                                                                          *;
%* 2020-02-26  initial coding                                                                       *;
%* 2020-07-13  update for refactor of java service hierarchy.                                       *;
%* 2021-02-09  increase displayName from $32 to $64 (DE16001).                                      *;
%* 2021-08-05  Remove header documentation                                                          *;
%* 2021-10-13  US11739                                                                              *;
%*                                                                                                  *;


%macro lsaf_getauditactions(
  lsaf_typeid=,
  sas_dsname=work.lsafgetauditactions
  ) /des='Get a list of the metadata for the audit actions.';
  
  %if "&sas_dsname" EQ "" %then %let sas_dsname=work.lsafgetauditactions;

  %****************************************;
  %* Initialize macro variables           *;
  %****************************************;
  %lsaf_getProductName();

  %global _lsafMsg_
          _lsafRC_
  ;

  %local _lsafRowCount_;

  %let _lsafRC_ =%str(-999);
  %let _lsafMsg_=%str(The SAS Macro did not execute properly.  Unknown err%str(or).);
  %let _lsafRowCount_=0;

  %**********************************************;
  %* Start the datastep to call the java code   *;
  %**********************************************;
  data &sas_dsname (label="Created by &SYSMACRONAME"
                    keep=typeID action displayName
    );

    attrib returnCode     length=8;
    attrib message        length=$200;
    attrib logMsgPrefix   length=$70;
    attrib logMsgPrimary  length=$70;
    attrib logMsg         length=$300;
    attrib rowCount       length=8;

    attrib typeID         length=$64    label="Object Type Identifier";
    attrib action         length=$32    label="Audit Action";
    attrib displayName    length=$64    label="Display Name";

    rowCount=0;

    %***********************************;
    %* Declare the java objects        *;
    %***********************************;
    declare javaobj srv("com/sas/lsaf/macro/security/audit/AuditService") ;
    declare javaobj results("com/sas/lsaf/macro/security/audit/SasAuditActionsResult");

    %***********************************;
    %* Attempt the operation           *;
    %***********************************;
    srv.callStaticVoidMethod("getAuditActions", "%bquote(&lsaf_typeid)", results);

    %***********************************;
    %* Retrieve the results            *;
    %***********************************;
    results.callIntMethod("getReturnCode", returnCode);
    results.callStringMethod("getReturnMessage", message);

    %***********************************;
    %* Process the results             *;
    %***********************************;
    logMsgPrefix="NOTE: &_LSAF_PRODUCT_NAME_ Macro: * ";
    if (returnCode ne 0) then
    do;
      logMsgPrimary="ERR%str(OR): &_LSAF_PRODUCT_NAME_ Macro:";
      logMsg=" Cannot retrieve list of audit actions for object type &lsaf_typeid..";
    end;
    else
    do;
      logMsgPrimary=logMsgPrefix;
      logMsg=" List of all the audit actions was returned.";
      results.callIntMethod("getSize", rowCount);
      do row=0 to rowCount - 1;
        results.callStringMethod("getTypeId", row, typeID);
        results.callStringMethod("getAction", row, action);
        results.callStringMethod("getDisplayName", row, displayName);
        output;
      end;
    end;

    %***********************************;
    %* Print messages in log           *;
    %***********************************;
    put;
    put logMsgPrimary " " logMsg;
    put logMsgPrefix " " %sysfunc(lowcase("&SYSMACRONAME"));
    if (rowCount=0) 
      then 
      do;
        if "&lsaf_typeid" ne "" then put logMsgPrefix " No audit actions were found for object type &lsaf_typeid..";
        else put logMsgPrefix " No audit actions were found.";
      end;
      else put logMsgPrefix " " rowCount " records written to data set &sas_dsname";
    put logMsgPrefix " _lsafMsg_= " message ;
    put logMsgPrefix " _lsafRC_= " returnCode ;
    put;

    %***********************************;
    %* Set the macro variables         *;
    %***********************************;
    call symputx("_lsafRC_", returnCode);
    call symputx("_lsafMsg_", message);
    call symputx("_lsafRowCount_", rowCount);

  run;
  
  %***********************************;
  %* Sort the data set               *;
  %***********************************;
  %if (&_lsafRowCount_ > 0) %then
  %do;
    %let notesOption=%sysfunc(getoption(notes));
    options nonotes;
    proc sort data=&sas_dsname sortseq=linguistic(strength=primary);
      by typeID action;
    run;
    options &notesOption;
  %end;

%mend lsaf_getauditactions;