%* lsaf_getallusers                                                                                *;
%*                                                                                                 *;
%* Gets the metadata for all of the user accounts that are defined. The metadata is stored in a    *;
%* SAS data set.                                                                                   *;
%*                                                                                                 *;
%* macrotype LSAF                                                                                  *;
%* since 1.3                                                                                       *;
%* exposure external                                                                               *;
%*                                                                                                 *;
%* History                                                                                         *;
%* 2013-10-30  initial coding                                                                      *;
%* 2016-08-26  rebrand as LSAF                                                                     *;
%* 2016-12-07  added account license type                                                          *;
%* 2019-11-01  update to handle a blank SAS output file.                                           *;
%* 2020-12-04  updates for service refactor                                                        *;
%* 2021-08-18  Remove header documentation                                                         *;
%* 2021-09-07  Update proc sort code (DE16792)                                                     *;
%* 2021-09-22  Add additional variables (US11398)                                                  *;
%*                                                                                                 *;

%macro lsaf_getallusers(
  sas_dsname=work.lsafGetAllUsers
  ) /des='Get a list of all users defined in the system.';
  
  %if "&sas_dsname" EQ "" %then %let sas_dsname=work.lsafGetAllUsers;

  %****************************************;
  %* Initialize macro variables           *;
  %****************************************;
  %lsaf_getProductName();

  %global _lsafMsg_
          _lsafRC_
  ;

  %local _lsafRowCount_;

  %let _lsafRC_ =%str(-999);
  %let _lsafMsg_=%str(The SAS Macro did not execute properly.  Unknown err%str(or).);
  %let _lsafRowCount_=0;

  %**********************************************;
  %* Start the datastep to call the java code   *;
  %**********************************************;
  data &sas_dsname (label="Created by &SYSMACRONAME"
                    keep= userId displayName firstName lastName email department phone
                          accountLicenseType isAccountActive accountActivationDate accountDeactivationDate
                          isAccountLocked accountLockedBy accountLockedDate
                          accountType lastLogin loginAttempts
                          passwordCreated passwordExpiration isPasswordExpired
                          dateAccountActivation dateAccountDeactivation dateAccountLocked
                          dateLastLogin datePasswordCreated datePasswordExpiration
                          dateCreated createdDate createdBy dateLastModified lastModifiedDate lastModifiedBy
  );

    attrib returnCode              length=8;
    attrib message                 length=$200;
    attrib logMsgPrefix            length=$70;
    attrib logMsgPrimary           length=$70;
    attrib logMsg                  length=$300;
    attrib rowCount                length=8;

    attrib userId                  length=$64    label="User ID";
    attrib displayName             length=$128   label="Display Name";
    attrib firstName               length=$64    label="First Name";
    attrib lastName                length=$64    label="Last Name";
    attrib email                   length=$128   label="Email Address";
    attrib department              length=$128   label="Department";
    attrib phone                   length=$64    label="Phone Number";

    attrib isAccountActive         length=8         label="Is Account Active";
    attrib accountLicenseType      length=$64       label="Account License Type";
    attrib accountActivationDate   length=$64       label="Account Activation Date";
    attrib dateAccountActivation   format=datetime. label="Account Activation Date (SAS DateTime Format)";
    attrib accountDeactivationDate length=$64       label="Account Deactivation Date";
    attrib dateAccountDeactivation format=datetime. label="Account Deactivation Date (SAS DateTime Format)";
    attrib isAccountLocked         length=8         label="Is Account Locked";
    attrib accountLockedBy         length=$64       label="Account Locked By";
    attrib accountLockedDate       length=$64       label="Account Locked Date";
    attrib dateAccountLocked       format=datetime. label="Account Locked Date (SAS DateTime Format)";

    attrib accountType             length=$32       label="Account Type";
    attrib lastLogin               length=$64       label="Last Login Date";
    attrib dateLastLogin           format=datetime. label="Last Login Date (SAS DateTime Format)";
    attrib loginAttempts           length=8         label="Failed Login Attempts (Reset on Login)";

    attrib passwordCreated         length=$64       label="Password Creation Date";
    attrib datePasswordCreated     format=datetime. label="Password Creation Date (SAS DateTime Format)";
    attrib passwordExpiration      length=$64       label="Password Expiration Date";
    attrib datePasswordExpiration  format=datetime. label="Password Expiration Date (SAS DateTime Format)";
    attrib isPasswordExpired       length=8         label="Is Password Expired";

    attrib createdDate             length=$64       label="User Creation Date";
    attrib dateCreated             format=datetime. label="User Creation Date (SAS DateTime Format)";
    attrib createdBy               length=$64       label="User Created By";
    attrib lastModifiedDate        length=$64       label="User Last Modified Date";
    attrib dateLastModified        format=datetime. label="User Last Modified Date (SAS DateTime Format)";
    attrib lastModifiedBy          length=$64       label="User Last Modified By";

    attrib accountActivationDateSasFormat        length=$64  label="Account Activation Date: SAS Format: USED FOR FORMATTING - will be dropped";
    attrib accountDeactivationDateSasFormat      length=$64  label="Account Deactivation Date: SAS Format: USED FOR FORMATTING - will be dropped";
    attrib accountLockedDateSasFormat            length=$64  label="Account Locked Date: SAS Format: USED FOR FORMATTING - will be dropped";
    attrib lastLoginSasFormat                    length=$64  label="Last Login Date: SAS Format: USED FOR FORMATTING - will be dropped";
    attrib passwordCreatedSasFormat              length=$64  label="Password Creation Date: SAS Format: USED FOR FORMATTING - will be dropped";
    attrib passwordExpirationSasFormat           length=$64  label="Password Expiration Date: SAS Format: USED FOR FORMATTING - will be dropped";
    attrib creationDateSasFormat                 length=$64  label="Creation Date: SAS Format: USED FOR FORMATTING - will be dropped";
    attrib lastModifiedSasFormat                 length=$64  label="Last Modified Date: SAS Format: USED FOR FORMATTING - will be dropped";

    rowCount=0;

    %***********************************;
    %* Declare the java objects        *;
    %***********************************;
    declare javaobj srv("com/sas/lsaf/macro/security/user/UserService") ;
    declare javaobj results("com/sas/lsaf/macro/security/user/SasUsersResult");

    %***********************************;
    %* Attempt the operation           *;
    %***********************************;
    srv.callStaticVoidMethod("getAllUsers", results);

    %***********************************;
    %* Retrieve the results            *;
    %***********************************;
    results.callIntMethod("getReturnCode", returnCode);
    results.callStringMethod("getReturnMessage", message);

    %***********************************;
    %* Process the results             *;
    %***********************************;
    logMsgPrefix="NOTE: &_LSAF_PRODUCT_NAME_ Macro: * ";
    if (returnCode ne 0) then
    do;
      logMsgPrimary="ERR%str(OR): &_LSAF_PRODUCT_NAME_ Macro:";
      logMsg=" Cannot retrieve list of users.";
    end;
    else
    do;
      logMsgPrimary=logMsgPrefix;
      logMsg=" List of users was returned.";
      results.callIntMethod("getSize", rowCount);
      do row=0 to rowCount - 1;
        results.callStringMethod("getAccountLicenseTypeId", row, accountLicenseType);
        results.callStringMethod("getAccountActivationDate", row, accountActivationDate);
        results.callStringMethod("getAccountDeactivationDate", row, accountDeactivationDate);
        results.callStringMethod("getAccountLockedBy", row, accountLockedBy);
        results.callStringMethod("getAccountLockedDate", row, accountLockedDate);
        results.callStringMethod("getAccountType", row, accountType);
        results.callStringMethod("getDepartment", row, department);
        results.callStringMethod("getDisplayName", row, displayName);
        results.callStringMethod("getEmail", row, email);
        results.callStringMethod("getFirstName", row, firstName);
        results.callStringMethod("getLastLogin", row, lastLogin);
        results.callStringMethod("getLastName", row, lastName);
        results.callLongMethod("getLoginAttempts", row, loginAttempts);
        results.callStringMethod("getPasswordCreated", row, passwordCreated);
        results.callStringMethod("getPasswordExpiration", row, passwordExpiration);
        results.callStringMethod("getPhone", row, phone);
        results.callStringMethod("getUserId", row, userId);
        results.callIntMethod("isAccountActive", row, isAccountActive);
        results.callIntMethod("isAccountLocked", row, isAccountLocked);
        results.callIntMethod("isPasswordExpired", row, isPasswordExpired);
        results.callStringMethod("getCreatedBy", row, createdBy);
        results.callStringMethod("getCreated", row, createdDate);
        results.callStringMethod("getLastModifiedBy", row, lastModifiedBy);
        results.callStringMethod("getLastModified", row, lastModifiedDate);

        results.callStringMethod("getAccountActivationDateSasFormat", row, accountActivationDateSasFormat);
        results.callStringMethod("getAccountDeactivationDateSasFormat", row, accountDeactivationDateSasFormat);
        results.callStringMethod("getAccountLockedDateSasFormat", row, accountLockedDateSasFormat);
        results.callStringMethod("getCreatedSasFormat", row, creationDateSasFormat);
        results.callStringMethod("getLastLoginSasFormat", row, lastLoginSasFormat);
        results.callStringMethod("getLastModifiedSasFormat", row, lastModifiedSasFormat);
        results.callStringMethod("getPasswordCreatedSasFormat", row, passwordCreatedSasFormat);
        results.callStringMethod("getPasswordExpirationSasFormat", row, passwordExpirationSasFormat);

        dateAccountActivation = input(accountActivationDateSasFormat, datetime.);
        dateAccountDeactivation = input(accountDeactivationDateSasFormat, datetime.);
        dateAccountLocked = input(accountLockedDateSasFormat, datetime.);
        dateLastLogin = input(lastLoginSasFormat, datetime.);
        datePasswordCreated = input(passwordCreatedSasFormat, datetime.);
        datePasswordExpiration = input(passwordExpirationSasFormat, datetime.);
        dateCreated = input(creationDateSasFormat, datetime.);
        dateLastModified =  input(lastModifiedSasFormat, datetime.);

        output;
      end;
    end;

    %***********************************;
    %* Print messages in log           *;
    %***********************************;
    put;
    put logMsgPrimary " " logMsg;
    put logMsgPrefix " " %sysfunc(lowcase("&SYSMACRONAME"));
    if (rowCount=0) 
      then put logMsgPrefix " No users were found.";
      else put logMsgPrefix " " rowCount " records written to data set &sas_dsname";
    put logMsgPrefix " _lsafMsg_= " message ;
    put logMsgPrefix " _lsafRC_= " returnCode ;
    put;

    %***********************************;
    %* Set the macro variables         *;
    %***********************************;
    call symputx("_lsafRC_", returnCode);
    call symputx("_lsafMsg_", message);
    call symputx("_lsafRowCount_", rowCount);

  run;

  %***********************************;
  %* Sort the data set               *;
  %***********************************;
  %if (&_lsafRowCount_ > 0) %then
  %do;
    %let notesOption=%sysfunc(getoption(notes));
    options nonotes;
    proc sort data=&sas_dsname sortseq=linguistic(strength=primary);
      by userid;
    run;
    options &notesOption;
  %end;

%mend lsaf_getallusers;
