%* lsaf_getallschedules                                                                            *;
%*                                                                                                 *;
%* Gets the metadata for all of the schedules. The metadata is stored in a SAS data set.           *;
%*                                                                                                 *;
%* macrotype LSAF                                                                                  *;
%* since 2.3                                                                                       *;
%* exposure external                                                                               *;
%*                                                                                                 *;
%* History                                                                                         *;
%* 2019-10-01  initial code                                                                        *;
%* 2020-06-30  update service location                                                             *;
%* 2021-08-16  Remove header documentation                                                         *;
%* 2021-10-12  US11739                                                                             *;
%*                                                                                                 *;

%macro lsaf_getallschedules(
  sas_dsname=work.lsafGetAllSchedules
  ) /des='Get a list of all schedules.';

  %if "&sas_dsname" EQ "" %then %let sas_dsname=work.lsafGetAllSchedules;

  %****************************************;
  %* Initialize macro variables           *;
  %****************************************;
  %lsaf_getProductName();

  %global _lsafMsg_
          _lsafRC_
  ;

  %let _lsafRC_ =%str(-999);
  %let _lsafMsg_=%str(The SAS Macro did not execute properly.  Unknown err%str(or).);

  %local _lsafRowCount_;

  %let _lsafRowCount_=0;

  %**********************************************;
  %* Start the datastep to call the java code   *;
  %**********************************************;
  data &sas_dsname (label="Created by &SYSMACRONAME"
                    keep=scheduleID name description owner status 
                    lastModified dateLastModified)
  ;
    attrib returnCode          length=8
           message             length=$200
           logMsgPrefix        length=$70
           logMsgPrimary       length=$70
           logMsg              length=$300
           rowCount            length=8

           scheduleID          length=$256   label="Schedule ID"
           name                length=$256   label="Schedule Name"
           description         length=$1024  label="Schedule Description"
           owner               length=$32    label="Schedule Owner"
           status              length=$32    label="Schedule Status"
           lastModified        length=$64    label="Last Schedule Modification Date"
           dateLastModified                  label="Date the schedule was last modified (SAS DateTime Format)"
           
           lastModifiedTempSAS  length=$64   label="TEMPORARY - WILL BE DROPPED"
    ;

    format dateLastModified datetime.
    ;

    rowCount=0;

    %***********************************;
    %* Declare the java objects        *;
    %***********************************;
    declare javaobj srv("com/sas/lsaf/macro/execution/schedule/ScheduleService") ;
    declare javaobj results("com/sas/lsaf/macro/execution/schedule/SasSchedulesResult");

    %***********************************;
    %* Attempt the operation           *;
    %***********************************;
    srv.callStaticVoidMethod("getAllSchedules", results) ;

    %***********************************;
    %* Retrieve the results            *;
    %***********************************;
    results.callIntMethod("getReturnCode", returnCode);
    results.callStringMethod("getReturnMessage", message);

    %***********************************;
    %* Process the results             *;
    %***********************************;
    logMsgPrefix="NOTE: &_LSAF_PRODUCT_NAME_ Macro: * ";
    if (returnCode ne 0) then
    do;
      logMsgPrimary="ERR%str(OR): &_LSAF_PRODUCT_NAME_ Macro:";
      logMsg=" Cannot retrieve schedules." ;
    end;
    else
    do;
      logMsgPrimary=logMsgPrefix;
      logMsg=" Schedules were returned." ;
      results.callIntMethod("getSize", rowCount);
      do row=0 to rowCount - 1;
        results.callStringMethod("getId", row, scheduleID);
        results.callStringMethod("getName", row, name);
        results.callStringMethod("getDescription", row, description);
        results.callStringMethod("getOwner", row, owner);
        results.callStringMethod("getStatus", row, status);
        results.callStringMethod("getLastModified", row, lastModified);
        results.callStringMethod("getLastModifiedSasFormat", row, lastModifiedTempSas);
        dateLastModified=input(lastModifiedTempSas, datetime.);
        output;
      end;
    end;

    %***********************************;
    %* Print messages in log           *;
    %***********************************;
    put;
    put logMsgPrimary " " logMsg;
    put logMsgPrefix " " %sysfunc(lowcase("&SYSMACRONAME"));
    if (rowCount=0) 
      then put logMsgPrefix " No schedules were found.";
      else put logMsgPrefix " " rowCount " records written to data set &sas_dsname";
    put logMsgPrefix " _lsafMsg_= " message ;
    put logMsgPrefix " _lsafRC_= " returnCode ;
    put;

    %***********************************;
    %* Set the macro variables         *;
    %***********************************;
    call symputx("_lsafRC_", returnCode);
    call symputx("_lsafMsg_", message);
    call symputx("_lsafRowCount_", rowCount);

  run;

  %***********************************;
  %* Sort by name                    *;
  %***********************************;
  %if (&_lsafRowCount_ > 0) %then
  %do;
    %let notesOption=%sysfunc(getoption(notes));
    options nonotes;
    proc sort data=&sas_dsname sortseq=linguistic(strength=primary);
      by scheduleID;
    run;
    options &notesOption;
  %end;

%mend lsaf_getallschedules;
