%* lsaf_getacls                                                                                  *;
%*                                                                                               *;
%* Gets the access control list for items in the repository. The access control list is stored   *;
%* in a SAS data set.                                                                            *;
%*                                                                                               *;
%* macrotype LSAF                                                                                *;
%* since 1.2                                                                                     *;
%* exposure external                                                                             *;
%*                                                                                               *;
%* History                                                                                       *;
%* 2013-08-30  updates for standardization and coding conventions                                *;
%* 2016-08-26  rebrand as LSAF                                                                   *;
%* 2019-02-05  update to 2.1                                                                     *;
%* 2019-11-01  update to handle a blank SAS output file.                                         *;
%* 2020-07-13  update for refactor of java service hierarchy.                                    *;
%* 2021-08-05  update for refactor of java service hierarchy.                                    *;
%* 2021-09-07  Clean up extraneous code (DE16975).                                               *;
%* 2021-09-20  Remove parameter defaults (DE16775)                                               *;
%* 2021-10-12  US11739                                                                           *;
%* 2021-10-19  Label update                                                                      *;
%*                                                                                               *;

%macro lsaf_getacls(
  lsaf_path=,
  sas_dsname=work.lsafGetACLs,
  lsaf_recursive=
  ) /des='Get a list of the permissions for items in the SAS Life Science Analytics Framework repository.';
  
  %if "&sas_dsname" EQ "" %then %let sas_dsname=work.lsafGetACLs;

  %****************************************;
  %* Initialize macro variables           *;
  %****************************************;
  %lsaf_getProductName();

  %global _lsafMsg_
          _lsafRC_
  ;

  %local _lsafRowCount_;

  %let _lsafRC_ =%str(-999);
  %let _lsafMsg_=%str(The SAS Macro did not execute properly.  Unknown err%str(or).);

  %**********************************************;
  %* Start the datastep to call the java code   *;
  %**********************************************;
  data &sas_dsname(label="Created by &SYSMACRONAME"
                   keep=aclType path name itemType owner aclPrincipalType aclPrincipalName grpSrcCtxt isContainer
                        adminPermission readPermission writePropPerm writeContentPerm deletePermission ptype_sort);
    attrib returnCode               length=8;
    attrib message                  length=$200;
    attrib logMsgPrefix             length=$70;
    attrib logMsgPrimary            length=$70;
    attrib logMsg                   length=$300;
    attrib rowCount                 length=8;
    attrib aclCount                 length=8;
    attrib aceCount                 length=8;

    attrib path                     length=$2048    label="Item Path";
    attrib name                     length=$256     label="Item Name";
    attrib itemType                 length=$32      label="Item Type";
    attrib isContainer              length=8        label="Is Item A Container?";

    attrib owner                    length=$64      label="Owner";
    attrib aclType                  length=$32      label="ACL Type";
    attrib aclPrincipalType         length=$32      label="ACE Principal Type";
    attrib aclPrincipalName         length=$256     label="ACE Principal Name";
    attrib grpSrcCtxt               length=$2048    label="ACE Group Context Source Path";
    attrib adminPermission          length=8        label="Admin Permission";
    attrib readPermission           length=8        label="Read Permission";
    attrib writePropPerm            length=8        label="Write Properties Permission";
    attrib writeContentPerm         length=8        label="Write Content Permission";
    attrib deletePermission         length=8        label="Delete Permission";

    attrib ptype_sort               length=8        label="Temp var - sorting value";

    rowCount=0;

    %***********************************;
    %* Declare the java objects        *;
    %***********************************;
    declare javaobj srv("com/sas/lsaf/macro/security/acl/AclService") ;
    declare javaobj results("com/sas/lsaf/macro/security/acl/SasAclsResult");

    %***********************************;
    %* Attempt the operation           *;
    %***********************************;
    srv.callStaticVoidMethod("getAcls", "%bquote(&lsaf_path)", "%bquote(&lsaf_recursive)", results);

    %***********************************;
    %* Retrieve the results            *;
    %***********************************;
    results.callIntMethod("getReturnCode", returnCode);
    results.callStringMethod("getReturnMessage", message);

    %***********************************;
    %* Process the results             *;
    %***********************************;
    logMsgPrefix="NOTE: &_LSAF_PRODUCT_NAME_ Macro: * ";
    if (returnCode ne 0) then
    do;
      logMsgPrimary="ERR%str(OR): &_LSAF_PRODUCT_NAME_ Macro:";
      logMsg=" Cannot retrieve access control lists from &lsaf_path" ;
    end;
    else
    do;
      logMsgPrimary=logMsgPrefix;
      logMsg=" Access control lists were returned for &lsaf_path" ;
      results.callIntMethod("getAclsSize", aclCount);
      do aclRow=0 to aclCount - 1;
        results.callStringMethod("getPath", aclRow, path);
        results.callStringMethod("getName", aclRow, name);
        results.callStringMethod("getOwner", aclRow, owner);
        results.callIntMethod("getIsContainer", aclRow, isContainer);
        results.callStringMethod("getItemType", aclRow, itemType);
        results.callStringMethod("getAclType", aclRow, aclType);

        results.callIntMethod("getAcesSize", aclRow, aceCount);
        do aceRow=0 to aceCount - 1;
          rowCount+1;
          results.callStringMethod("getAclPrincipalType", aclRow, aceRow, aclPrincipalType);
          results.callStringMethod("getAclPrincipalName", aclRow, aceRow, aclPrincipalName);
          results.callStringMethod("getGroupContextPath", aclRow, aceRow, grpSrcCtxt);
          results.callIntMethod("getReadPermission", aclRow, aceRow, readPermission);
          results.callIntMethod("getWriteContentPermission", aclRow, aceRow, writeContentPerm);
          results.callIntMethod("getWritePropertiesPermission", aclRow, aceRow, writePropPerm);
          results.callIntMethod("getDeletePermission", aclRow, aceRow, deletePermission);
          results.callIntMethod("getAdminPermission", aclRow, aceRow, adminPermission);

          if (trim(aclPrincipalType) eq "ACLOWNER") 
            then ptype_sort=1;
            else if (trim(aclPrincipalType) eq "ACLMEMBERS") 
              then ptype_sort=2;
              else ptype_sort=99;

          output;
        end;
      end;
    end;

    %***********************************;
    %* Print messages in log           *;
    %***********************************;
    put;
    put logMsgPrimary " " logMsg;
    put logMsgPrefix " " %sysfunc(lowcase("&SYSMACRONAME"));
    put logMsgPrefix " " rowCount " records written to data set &sas_dsname";
    put logMsgPrefix " _lsafMsg_= " message ;
    put logMsgPrefix " _lsafRC_= " returnCode ;
    put logMsgPrefix " lsaf_recursive= &lsaf_recursive" ;
    put;

    %***********************************;
    %* Set the macro variables         *;
    %***********************************;
    call symputx("_lsafRC_", returnCode);
    call symputx("_lsafMsg_", message);
    call symputx("_lsafRowCount_", rowCount);

   run;

  %***********************************;
  %* Sort the data set               *;
  %***********************************;
  %let notesOption=%sysfunc(getoption(notes));
  options nonotes;
  %if (&_lsafRowCount_ > 0) %then
  %do;
    %***********************************;
    %* Sort by path and name           *;
    %***********************************;
    proc sort data=&sas_dsname out=&sas_dsname (label="Created by &SYSMACRONAME") sortseq=linguistic(strength=primary);
      by path aclType ptype_sort aclPrincipalName;
    run;
  %end;
  proc sql;
    alter table &sas_dsname drop ptype_sort;
  quit;
  options &notesOption;

%mend lsaf_getacls;