%* lsaf_exportstudydocs                                                                               *;
%*                                                                                                    *;
%* Exports the metadata for the study supporting documents. The metadata is stored in a SAS data set. *;
%*                                                                                                    *;
%* macrotype LSAF                                                                                     *;
%* since 2.2                                                                                          *;
%* exposure external                                                                                  *;
%*                                                                                                    *;
%* History                                                                                            *;
%* 2019-08-21  initial coding                                                                         *;
%* 2020-11-23  update for versioning strategy.                                                        *;
%* 2021-08-16  Remove header documentation                                                            *;
%*                                                                                                    *;


%macro lsaf_exportstudydocs(
  lsaf_studyid=,
  lsaf_basestandardname=,
  lsaf_modelid=,
  lsaf_location=,
  lsaf_path=,
  lsaf_overwrite=,
  lsaf_enableversioning=,
  lsaf_versiontype=,
  lsaf_customversion=,
  lsaf_comment=
  ) /des='Exports study supporting documents to a specified location in the SAS Life Science Analytics Framework.';

  %****************************************;
  %* Initialize macro variables           *;
  %****************************************;
  %lsaf_getProductName();

  %global _lsafMsg_
          _lsafRC_
		  _lsafexportstudydocspath_
  ;

  %let _lsafRC_ =%str(-999);
  %let _lsafMsg_=%str(The SAS Macro did not execute properly.  Unknown err%str(or).);

  %**********************************************;
  %* Start the datastep to call the java code   *;
  %**********************************************;
  data _null_;
	attrib returnCode         length=8
           message            length=$200
           logMsgPrefix       length=$70
           logMsgPrimary      length=$70
           logMsg             length=$300
		   exportPath         length=$2048
    ;

    %*****************************;
    %* Declare the java objects  *;
    %*****************************;
    declare javaobj srv("com/sas/lsaf/macro/clinical/study/StudyService") ;
    declare javaobj exportData("com/sas/lsaf/macro/content/SasFileCreateParameters"); 
    declare javaobj results("com/sas/lsaf/macro/common/result/Result");
 
    %***********************************;
    %* Attempt the operation           *;
    %***********************************;
    exportData.callVoidMethod("updateData", "%bquote(&lsaf_location)", "%bquote(&lsaf_path)", "%bquote(&lsaf_overwrite)", 
                                "%bquote(&lsaf_enableversioning)", "%bquote(&lsaf_versiontype)", "%bquote(&lsaf_customversion)", 
                                "%bquote(&lsaf_comment)");
                                
    srv.callStaticStringMethod("exportStudySupportingDocuments", "%bquote(&lsaf_studyid)", "%bquote(&lsaf_basestandardname)", "%bquote(&lsaf_modelid)", 
         exportData, results, exportPath);

    %***********************************;
    %* Retrieve the results            *;
    %***********************************;
    results.callIntMethod("getReturnCode", returnCode);
    results.callStringMethod("getReturnMessage", message);
    logMsgPrefix="NOTE: &_LSAF_PRODUCT_NAME_ Macro: * ";
    if (returnCode ne 0) then
    do;
      logMsgPrimary="ERR%str(OR): &_LSAF_PRODUCT_NAME_ Macro:";
      logMsg=" Could not export the study supporting documents." ;
    end;
    else
    do;
      logMsgPrimary=logMsgPrefix;
      logMsg=" Returned the path for the created study supporting documents file.";
    end;

    %***********************************;
    %* Print messages in log           *;
    %***********************************;
    put;
    put logMsgPrimary " " logMsg;
    put logMsgPrefix " " %sysfunc(lowcase("&SYSMACRONAME"));
    if (returnCode ne 0) 
      then put logMsgPrefix " The study supporting documents were not exported.";
      else put logMsgPrefix " The study supporting documents were exported to " exportPath;
    put logMsgPrefix " _lsafMsg_= " message ;
    put logMsgPrefix " _lsafRC_= " returnCode ;
    put logMsgPrefix " _lsafexportstudydocspath_=" exportPath;
    put;

    %***********************************;
    %* Set the macro variables         *;
    %***********************************;
    call symputx("_lsafRC_", returnCode);
    call symputx("_lsafMsg_", message);
    call symputx("_lsafexportstudydocspath_", exportPath);
  run;

%mend lsaf_exportstudydocs;
