%* lsaf_exportstandard                                                                                *;
%*                                                                                                    *;
%* Exports the metadata for a global standard. The metadata is stored in a SAS data set.              *;
%*                                                                                                    *;
%* macrotype LSAF                                                                                     *;
%* since 2.2                                                                                          *;
%* exposure external                                                                                  *;
%*                                                                                                    *;
%* History                                                                                            *;
%* 2019-06-07  initial coding                                                                         *;
%* 2019-11-01  update to handle a blank SAS output file.                                              *;
%* 2021-08-16  Remove header documentation                                                            *;
%* 2021-08-31  DE16792                                                                                *;
%*                                                                                                    *;

%macro lsaf_exportstandard(
  lsaf_standardid=,
  lsaf_status=,
  lsaf_standardversion=,
  lsaf_path=,
  lsaf_prefix=,
  lsaf_overwrite=,
  lsaf_enableversioning=,
  lsaf_versiontypefornewfiles=,
  lsaf_customversfornewfiles=,
  lsaf_versiontypeforexistingfiles=,
  lsaf_customversforexistingfiles=,
  lsaf_comment=,
  sas_result_dsname=work.lsafexportstandard
  ) /des='Exports a global standard to a specified location in the SAS Life Science Analytics Framework.';

  %if "&sas_result_dsname" EQ "" %then %let sas_result_dsname=work.lsafexportstandard;

  %****************************************;
  %* Initialize macro variables           *;
  %****************************************;
  %lsaf_getProductName();

  %global _lsafMsg_
          _lsafRC_
  ;

  %local _lsafRowCount_;

  %let _lsafRC_ =%str(-999);
  %let _lsafMsg_=%str(The SAS Macro did not execute properly.  Unknown err%str(or).);
  %let _lsafRowCount_=0;

  %**********************************************;
  %* Start the datastep to call the java code   *;
  %**********************************************;
  data &sas_result_dsname (label="Created by &SYSMACRONAME" keep=location path name);
    attrib returnCode         length=8
           message            length=$200
           logMsgPrefix       length=$70
           logMsgPrimary      length=$70
           logMsg             length=$300
           rowCount           length=8
           location           length=$10      label="Location"
           path               length=$2048    label="Export file Path"
           name               length=$256     label="Export file Name"
    ;

    %*****************************;
    %* Declare the java objects  *;
    %*****************************;
    declare javaobj srv("com/sas/lsaf/macro/clinical/standards/StandardsService");
    declare javaobj exportData("com/sas/lsaf/macro/content/SasBatchFileCreateParameters");
    declare javaobj results("com/sas/lsaf/macro/common/result/SasPathResult");
 
    %***********************************;
    %* Attempt the operation           *;
    %***********************************;
    exportData.callVoidMethod("updateBatchData", "%bquote(&lsaf_path)", "%bquote(&lsaf_overwrite)", "%bquote(&lsaf_enableversioning)", 
      "%bquote(&lsaf_versiontypefornewfiles)", "%bquote(&lsaf_customversfornewfiles)", "%bquote(&lsaf_comment)", 
      "%bquote(&lsaf_versiontypeforexistingfiles)", "%bquote(&lsaf_customversforexistingfiles)", 
      "%bquote(&lsaf_prefix)");

    srv.callStaticVoidMethod("exportStandardToDataSetsInRepository", "%bquote(&lsaf_standardid)", "%bquote(&lsaf_status)", "%bquote(&lsaf_standardversion)", exportData, results);

    %***********************************;
    %* Retrieve the results            *;
    %***********************************;
    results.callIntMethod("getReturnCode", returnCode);
    results.callStringMethod("getReturnMessage", message);
    logMsgPrefix="NOTE: &_LSAF_PRODUCT_NAME_ Macro: * ";
    if (returnCode ne 0) then
    do;
      logMsgPrimary="ERR%str(OR): &_LSAF_PRODUCT_NAME_ Macro:";
      logMsg=" Could not export the global standard." ;
    end;
    else
    do;
      logMsgPrimary=logMsgPrefix;
      logMsg=" Returned the list of paths for the exported global standard.";
      results.callIntMethod("getSize", rowCount);
      results.callStringMethod("getLocation", location);
      do row=0 to rowCount - 1;
        results.callStringMethod("getPath", row, path);
        results.callStringMethod("getName", row, name);
        output;
      end;
    end;

    %***********************************;
    %* Print messages in log           *;
    %***********************************;
    put;
    put logMsgPrimary " " logMsg;
    put logMsgPrefix " " %sysfunc(lowcase("&SYSMACRONAME"));
    if (rowCount<=0) 
      then put logMsgPrefix " The global standard was not exported.";
      else put logMsgPrefix " " rowCount " records written to data set &sas_result_dsname";
    put logMsgPrefix " _lsafMsg_= " message ;
    put logMsgPrefix " _lsafRC_= " returnCode ;
    put;

    %***********************************;
    %* Set the macro variables         *;
    %***********************************;
    call symputx("_lsafRC_", returnCode);
    call symputx("_lsafMsg_", message);
    call symputx("_lsafRowCount_", rowCount);
  run;

  %***********************************;
  %* Sort the data set               *;
  %***********************************;
  %let notesOption=%sysfunc(getoption(notes));
  options nonotes;

  %if (&_lsafRowCount_ > 0) %then
  %do;
    proc sort data=&sas_result_dsname out=&sas_result_dsname sortseq=linguistic(strength=primary);
      by path;
    run;
  %end;
  
  options &notesOption;

%mend lsaf_exportstandard;
