%* lsaf_exportglobaltlfs                                                                              *;
%*                                                                                                    *;
%* Exports the metadata for the global tables, listings, and figures (TLF) to the specified area in   *;
%* the repository. The metadata is stored in SAS data sets.                                           *;
%*                                                                                                    *;
%* macrotype LSAF                                                                                     *;
%* since 2.5                                                                                          *;
%* exposure external                                                                                  *;
%*                                                                                                    *;
%* History                                                                                            *;
%* 2021-06-09  initial coding                                                                         *;
%* 2021-08-17  Remove header documentation                                                            *;
%* 2021-08-31  DE16792                                                                                *;
%* 2021-10-12  US11739                                                                                *;
%*                                                                                                    *;

%macro lsaf_exportglobaltlfs(
  lsaf_tlftemplateid=,
  lsaf_path=,
  lsaf_prefix=,
  lsaf_includeresults=,
  lsaf_overwrite=,
  lsaf_enableversioning=,
  lsaf_versiontypefornewfiles=,
  lsaf_customversfornewfiles=,
  lsaf_versiontypeforexistingfiles=,
  lsaf_customversforexistingfiles=,
  lsaf_comment=,
  sas_result_dsname=lsafexportglobaltlfs
  ) /des='Exports metadata for global tables, figures, and listings to data sets in the repository.';
  
  %if "&sas_result_dsname" EQ "" %then %let sas_result_dsname=work.lsafexportglobaltlfs;

  %****************************************;
  %* Initialize macro variables           *;
  %****************************************;
  %lsaf_getProductName();

  %global _lsafMsg_
          _lsafRC_
  ;

  %local _lsafRowCount_;

  %let _lsafRC_ =%str(-999);
  %let _lsafMsg_=%str(The SAS Macro did not execute properly.  Unknown err%str(or).);
  %let _lsafRowCount_=0;

  %**********************************************;
  %* Start the datastep to call the java code   *;
  %**********************************************;
  data &sas_result_dsname (label="Created by &SYSMACRONAME" keep=location path name);
    attrib returnCode         length=8
           message            length=$200
           logMsgPrefix       length=$70
           logMsgPrimary      length=$70
           logMsg             length=$300
           rowCount           length=8
           location           length=$10      label="Location"
           path               length=$2048    label="Export file Path"
           name               length=$256     label="Export file Name"
    ;
    %*****************************;
    %* Declare the java objects  *;
    %*****************************;
    declare javaobj srv("com/sas/lsaf/macro/clinical/tlf/TlfTemplateService") ;
    declare javaobj exportData("com/sas/lsaf/macro/content/SasBatchFileCreateParameters"); 
    declare javaobj results("com/sas/lsaf/macro/common/result/SasPathResult");
 
    %***********************************;
    %* Attempt the operation           *;
    %***********************************;
    exportData.callVoidMethod("updateBatchData", "%bquote(&lsaf_path)", "%bquote(&lsaf_overwrite)", "%bquote(&lsaf_enableversioning)", 
      "%bquote(&lsaf_versiontypefornewfiles)", "%bquote(&lsaf_customversfornewfiles)", "%bquote(&lsaf_comment)", 
      "%bquote(&lsaf_versiontypeforexistingfiles)", "%bquote(&lsaf_customversforexistingfiles)", 
      "%bquote(&lsaf_prefix)");
 
    srv.callStaticVoidMethod("exportTlfsToRepository", "%bquote(&lsaf_tlftemplateid)", exportData, "%bquote(&lsaf_includeresults)", results);

    %***********************************;
    %* Retrieve the results            *;
    %***********************************;
    results.callIntMethod("getReturnCode", returnCode);
    results.callStringMethod("getReturnMessage", message);
    logMsgPrefix="NOTE: &_LSAF_PRODUCT_NAME_ Macro: * ";
    if (returnCode ne 0) then
    do;
      logMsgPrimary="ERR%str(OR): &_LSAF_PRODUCT_NAME_ Macro:";
      logMsg=" Could not export the global tables, listings, and figures." ;
    end;
    else
    do;
      logMsgPrimary=logMsgPrefix;
      logMsg=" Returned the list of paths for the global tables, listings, and figures.";
      results.callIntMethod("getSize", rowCount);
      results.callStringMethod("getLocation", location);
      do row=0 to rowCount - 1;
        results.callStringMethod("getPath", row, path);
        results.callStringMethod("getName", row, name);
        output;
      end;
    end;
    %***********************************;
    %* Print messages in log           *;
    %***********************************;
    put;
    put logMsgPrimary " " logMsg;
    put logMsgPrefix " " %sysfunc(lowcase("&SYSMACRONAME"));
    if (rowCount<=0) 
      then put logMsgPrefix " The global tables, listings, and figures were not exported.";
      else put logMsgPrefix " " rowCount " records written to data set &sas_result_dsname";
    put logMsgPrefix " _lsafMsg_= " message ;
    put logMsgPrefix " _lsafRC_= " returnCode ;
    put;

    %***********************************;
    %* Set the macro variables         *;
    %***********************************;
    call symputx("_lsafRC_", returnCode);
    call symputx("_lsafMsg_", message);
    call symputx("_lsafRowCount_", rowCount);
  run;

  %***********************************;
  %* Sort the data set               *;
  %***********************************;
  %if (&_lsafRowCount_ > 0) %then
  %do;
    %let notesOption=%sysfunc(getoption(notes));
    options nonotes;
    proc sort data=&sas_result_dsname out=&sas_result_dsname sortseq=linguistic(strength=primary);
      by path;
    run;
    options &notesOption;
  %end;

%mend lsaf_exportglobaltlfs;
