%* lsaf_exportextdicts                                                                                  *;
%*                                                                                                      *;
%* Exports the metadata for external dictionaries. The metadata is stored in a SAS data set in the      *;
%* specified location (WORKSPACE or REPOSITORY).                                                        *;
%*                                                                                                      *;
%* macrotype LSAF                                                                                       *;
%* since 2.2                                                                                            *;
%* exposure external                                                                                    *;
%*                                                                                                      *;
%* History                                                                                              *;
%* 2019-07-08  initial coding                                                                           *;
%* 2020-07-14  update for refactor of java service hierarchy.                                           *;
%* 2021-08-05  Remove header documentation                                                              *;
%*                                                                                                      *;

%macro lsaf_exportextdicts(
  lsaf_location=,
  lsaf_path=,
  lsaf_overwrite=,
  lsaf_enableversioning=,
  lsaf_versiontype=,
  lsaf_customversion=,
  lsaf_comment=
  ) /des='Exports external dictionaries to a specified location in the SAS Life Science Analytics Framework.';

  %****************************************;
  %* Initialize macro variables           *;
  %****************************************;
  %lsaf_getProductName();

  %global _lsafMsg_
          _lsafRC_
          _lsafExtDictionaryPath_
  ;


  %let _lsafRC_ =%str(-999);
  %let _lsafMsg_=%str(The SAS Macro did not execute properly.  Unknown err%str(or).);

  %**********************************************;
  %* Start the datastep to call the java code   *;
  %**********************************************;
  data _null_;
    attrib returnCode      length=8
           message         length=$200
           dictionaryPath  length=$2048
           logMsgPrefix    length=$70
           logMsgPrimary   length=$70
           logMsg          length=$300
    ;
    
    dictionaryPath="";
    
    %*****************************;
    %* Declare the java objects  *;
    %*****************************;
    declare javaobj srv("com/sas/lsaf/macro/clinical/externaldictionary/ExternalDictionaryService") ;
    declare javaobj exportData("com/sas/lsaf/macro/content/SasFileCreateParameters"); 
    declare javaobj results("com/sas/lsaf/macro/common/result/Result");
 
    %***********************************;
    %* Attempt the operation           *;
    %***********************************;
    exportData.callVoidMethod("updateData", "%bquote(&lsaf_location)", "%bquote(&lsaf_path)", "%bquote(&lsaf_overwrite)", 
                                "%bquote(&lsaf_enableversioning)", "%bquote(&lsaf_versiontype)", "%bquote(&lsaf_customversion)", 
                                "%bquote(&lsaf_comment)");
    srv.callStaticStringMethod("exportExtDictionariesToDataSet", exportData, results, dictionaryPath);

    %***********************************;
    %* Retrieve the results            *;
    %***********************************;
    results.callIntMethod("getReturnCode", returnCode);
    results.callStringMethod("getReturnMessage", message);
    logMsgPrefix="NOTE: &_LSAF_PRODUCT_NAME_ Macro: * ";
    if (returnCode ne 0) then
    do;
      logMsgPrimary="ERR%str(OR): &_LSAF_PRODUCT_NAME_ Macro:";
      logMsg=" Could not export the external dictionaries." ;
    end;
    else
    do;
      logMsgPrimary=logMsgPrefix;
      logMsg=" Returned the path for the created external dictionary file.";
    end;

    %***********************************;
    %* Print messages in log           *;
    %***********************************;
    put;
    put logMsgPrimary " " logMsg;
    put logMsgPrefix " " %sysfunc(lowcase("&SYSMACRONAME"));
    if (returnCode ne 0) 
      then put logMsgPrefix " The external dictionary was not exported.";
      else put logMsgPrefix " The external dictionary was exported to " dictionaryPath;
    put logMsgPrefix " _lsafMsg_= " message ;
    put logMsgPrefix " _lsafRC_= " returnCode ;
    put logMsgPrefix " _lsafExtDictionaryPath_=" dictionaryPath;
    put;

    %***********************************;
    %* Set the macro variables         *;
    %***********************************;
    call symputx("_lsafRC_", returnCode);
    call symputx("_lsafMsg_", message);
    call symputx("_lsafExtDictionaryPath_", dictionaryPath);
  run;

%mend lsaf_exportextdicts;
