%* lsaf_deployprocessdefinition                                                                     *;
%*                                                                                                  *;
%* Deploys a process flow definition using a BPMN file from the repository.                         *;
%*                                                                                                  *;
%* macrotype LSAF                                                                                   *;
%* since 2.2                                                                                        *;
%* exposure external                                                                                *;
%*                                                                                                  *;
%* History                                                                                          *;
%* 2019-09-12  initial coding                                                                       *;
%* 2021-08-10  Remove header documentation                                                          *;
%* 2021-08-25  DE16786                                                                              *;
%* 2021-09-20  Remove parameter defaults (DE16775)                                                  *;
%* 2023-03-09  DE17654 - SAS Notes option updates                                                   *;
%*                                                                                                  *;

%macro lsaf_deployprocessdefinition(
  lsaf_path=,
  lsaf_version=,
  sas_dsname=,
  lsaf_comment=,
  lsaf_activate=,
  lsaf_overwrite=
  ) /des='Deploy the process flow definitions for a SAS Life Science Analytics Framework process.';


  %****************************************;
  %* Initialize macro variables           *;
  %****************************************;
  %lsaf_getProductName();

  %global _lsafMsg_
          _lsafRC_
          _lsafProcessDefinitionId_
  ;

  %local logMsgPrefix
         logMsgPrimary
         macroName
		 notesOption
  ;

  %let _lsafRC_ =%str(-999);
  %let _lsafMsg_=%str(The SAS Macro did not execute properly.  Unknown err%str(or).);
  %let macroName=%sysfunc(lowcase(&SYSMACRONAME));
  
  %let logMsgPrimary=%str(ERR%str(OR): &_LSAF_PRODUCT_NAME_ Macro: );
  %let logMsgPrefix=%str(NOTE: &_LSAF_PRODUCT_NAME_ Macro: );

  %*************************************************************;
  %* Turn off the notes generated for all the pre-processing.  *;
  %* Save original option setting so it can be reset properly. *;
  %*************************************************************;
  %let notesOption=%sysfunc(getoption(notes));
  options nonotes;

  %***************************************************************************;
  %* Validate the input data set                                             *;
  %* sets global macro variable _lsafRC_=0 if validation successful          *;
  %***************************************************************************;
  %lsaf_validate_sasdataset(callingMacro=&macroName,
        sas_dsn=&sas_dsname,
        required_variables=%str(id)
        );
  options &notesOption.;
  %if (&_lsafRC_ ne 0) %then
  %do;
    %put &logMsgPrimary _lsafMsg_= &_lsafMsg_ ;
    %put &logMsgPrefix &macroName ;
    %put &logMsgPrefix Input data set = &sas_dsname;
    %put &logMsgPrefix _lsafRC_= &_lsafRC_ ;
    %put;
    %return;
  %end;

  %***************************************************************************;
  %* If data set is valid, process it.                                       *;
  %***************************************************************************;
  %put NOTE: &macroName: Processing data set &sas_dsname;

  %***********************;
  %* Process the records *;
  %***********************;

  data _null_;
    attrib returnCode    length=8
           message       length=$200
           logMsgPrefix  length=$70
           logMsgPrimary length=$70
           logMsg        length=$300
           updateDef     length=$300
    ;
    set &sas_dsname end=eof;

    %**********************************;
    %* Declare the java object, once  *;
    %**********************************;
    if (_n_ = 1) then
    do;
      declare javaobj contextTypeIds("com/sas/lsaf/macro/common/SasStrings");
    end;

    %************************************;
    %* Populate the contextTypeIds list *; 
    %************************************;
    contextTypeIds.callVoidMethod("addValue", kstrip(id));

    if eof then 
    do;
      %***********************************;
      %* Call the update                 *;
      %***********************************; 
      declare javaobj srv("com/sas/lsaf/macro/workflow/definition/ProcessDefinitionService");
      declare javaobj results("com/sas/lsaf/macro/common/result/Result");
      
      srv.callStaticStringMethod("deployProcessDefinitionFromRepository", "%bquote(&lsaf_path)", "%bquote(&lsaf_version)", contextTypeIds,
                               "%bquote(&lsaf_comment)", "%bquote(&lsaf_activate)", "%bquote(&lsaf_overwrite)", results, updateDef);

      %***********************************;
      %* Retrieve the results            *;
      %***********************************;
      results.callIntMethod("getReturnCode", returnCode);
      results.callStringMethod("getReturnMessage", message);

      %***********************************;
      %* Process the results             *;
      %***********************************;
      logMsgPrefix="NOTE: &_LSAF_PRODUCT_NAME_ Macro: * ";
      if (returnCode ne 0) then
      do;
        logMsgPrimary="ERR%str(OR): &_LSAF_PRODUCT_NAME_ Macro:";
        logMsg=" Cannot deploy process flow definition.";
      end;
      else
      do;
        logMsgPrimary=logMsgPrefix;
        logMsg=" Deployment of process defintion was performed.";
      end;

      %***********************************;
      %* Print messages in log           *;
      %***********************************;
      put;
      put logMsgPrimary " " logMsg;
      put logMsgPrefix " " %sysfunc(lowcase("&SYSMACRONAME"));
      put logMsgPrefix " _lsafMsg_= " message ;
      put logMsgPrefix " _lsafRC_= " returnCode ;
      put logMsgPrefix " Input data set = &sas_dsname";
      put logMsgPrefix " Deployed process definition ID = " updateDef;
      put;

      %***********************************;
      %* Set the macro variables         *;
      %***********************************;
      call symputx("_lsafRC_", returnCode);
      call symputx("_lsafMsg_", message);
      call symputx("_lsafProcessDefinitionId_", updateDef);
    end;
  run;

%mend lsaf_deployprocessdefinition;
