%* lsaf_deleterecyclebinitems                                                                      *;
%*                                                                                                 *;
%* Permanently deletes items from the SAS Life Science Analytics Framework recycle bin using a     *;
%* data set as input. Returns a SAS data set that contains the paths for the items that were       *;
%* permanently deleted from the SAS Life Science Analytics Framework recycle bin.                  *;
%*                                                                                                 *;
%* macrotype LSAF                                                                                  *;
%* since 1.4                                                                                       *;
%* exposure external                                                                               *;
%*                                                                                                 *;
%* History                                                                                         *;
%* 2013-03-27  initial code                                                                        *;
%* 2016-08-26  rebrand as LSAF                                                                     *;
%* 2019-02-20  update to 2.1 and rename from lsaf_deleterbitems to lsaf_deleterecyclebinitems      *;
%* 2019-11-01  update to handle a blank SAS output file.                                           *;
%* 2020-07-13  update for refactor of java service hierarchy.                                      *;
%* 2021-08-12  Remove header documentation                                                         *;
%* 2021-08-31  DE16792                                                                             *;
%* 2021-10-12  US11739                                                                             *;
%* 2021-11-22  SAS Macros 2.6: RecycleBin refactor                                                 *;
%* 2023-03-09  DE17654 - SAS Notes option updates                                                  *;
%*                                                                                                 *;

%macro lsaf_deleterecyclebinitems(
  sas_dsname=,
  sas_result_dsname=work.lsafdeleterecyclebinitems
  ) /des='Deletes the specified items from the SAS Life Science Analytics Framework recycle bin.';

  %if "&sas_result_dsname" EQ "" %then %let sas_result_dsname=work.lsafdeleterecyclebinitems;

  %****************************************;
  %* Initialize macro variables           *;
  %****************************************;
  %lsaf_getProductName();

  %global _lsafMsg_
          _lsafRC_
  ;

  %let _lsafRC_ =%str(-999);
  %let _lsafMsg_=%str(The SAS Macro did not execute properly.  Unknown err%str(or).);

  %local _lsafRowCount_
         logMsgPrefix
         logMsgPrimary
         macroName
         notesOption
  ;

  %*************************************************************;
  %* Turn off the notes generated for all the pre-processing.  *;
  %* Save original option setting so it can be reset properly. *;
  %*************************************************************;
  %let notesOption=%sysfunc(getoption(notes));
  options nonotes;

  %let _lsafRowCount_=0;

  %*******************************************************************;
  %* Validate the input data set                                     *;
  %* sets global macro variable _lsafRC_=0 if validation successful  *;
  %*******************************************************************;
  %let macroName=%sysfunc(lowcase(&SYSMACRONAME));
  %lsaf_validate_sasdataset(callingMacro=&macroName, sas_dsn=&sas_dsname, required_variables=%str(identifier));
  
  options &notesOption.;
  %if (&_lsafRC_ ne 0) %then
  %do;
    %let logMsgPrimary=%str(ERR%str(OR): &_LSAF_PRODUCT_NAME_ Macro: );
    %let logMsgPrefix=%str(NOTE: &_LSAF_PRODUCT_NAME_ Macro: );
    %put &logMsgPrimary _lsafMsg_= &_lsafMsg_ ;
    %put &logMsgPrefix &_LSAF_PRODUCT_NAME_ Macro &macroName ;
    %put &logMsgPrefix Input data set = &sas_dsname;
    %put &logMsgPrefix _lsafRC_= &_lsafRC_ ;
    %put;
    %return;
  %end;

  %***************************************************************************;
  %* If data set is valid, process it.                                       *;
  %***************************************************************************;
  %put NOTE: &macroName: Processing data set &sas_dsname;

  %**********************************************;
  %* Start the datastep to call the java code   *;
  %**********************************************;
  data &sas_result_dsname (label="Created by &SYSMACRONAME" keep=path name);
    attrib returnCode         length=8
           message            length=$200
           logMsgPrefix       length=$70
           logMsgPrimary      length=$70
           logMsg             length=$300
           rowCount           length=8
           path               length=$2048    label="Item Path"
           name               length=$256     label="Item Name"
    ;

    set &sas_dsname end=eof;

    %***********************************;
    %* Declare the java objects, once  *;
    %***********************************;
    if (_n_ = 1) then
    do;
      declare javaobj srv("com/sas/lsaf/macro/content/repository/recyclebin/RecycleBinService") ;
      declare javaobj rbItems("com/sas/lsaf/macro/common/SasStrings");
      declare javaobj results("com/sas/lsaf/macro/common/result/SasPathResult");
    end;

    %*******************************************;
    %* Update the result with the identifiers  *;
    %*******************************************;

    rbItems.callVoidMethod("addValue", kstrip(identifier));

    if (eof);

    rowCount=0;

    %***********************************;
    %* Attempt the operation           *;
    %***********************************;
    srv.callStaticVoidMethod("deleteItems", rbItems, results);

    %***********************************;
    %* Retrieve the results            *;
    %***********************************;
    results.callIntMethod("getReturnCode", returnCode);
    results.callStringMethod("getReturnMessage", message);
    logMsgPrefix="NOTE: &_LSAF_PRODUCT_NAME_ Macro: * ";
    if (returnCode ne 0) then
    do;
      logMsgPrimary="ERR%str(OR): &_LSAF_PRODUCT_NAME_ Macro:";
      logMsg=" Could not permanently delete items from the recycle bin." ;
    end;
    else
    do;
      logMsgPrimary=logMsgPrefix;
      logMsg=" Returned the list of paths deleted from the recycle bin.";
      results.callIntMethod("getSize", rowCount);
      do row=0 to rowCount - 1;
        results.callStringMethod("getPath", row, path);
        results.callStringMethod("getName", row, name);
        output;
      end;
    end;

    %***********************************;
    %* Print messages in log           *;
    %***********************************;
    put;
    put logMsgPrimary " " logMsg;
    put logMsgPrefix " " %sysfunc(lowcase("&SYSMACRONAME"));
    if (rowCount=0) 
      then put logMsgPrefix " No items were deleted from the recycle bin.";
      else put logMsgPrefix " " rowCount " records written to data set &sas_result_dsname";
    put logMsgPrefix " _lsafMsg_= " message ;
    put logMsgPrefix " _lsafRC_= " returnCode ;
    put;

    %***********************************;
    %* Set the macro variables         *;
    %***********************************;
    call symputx("_lsafRC_", returnCode);
    call symputx("_lsafMsg_", message);
    call symputx("_lsafRowCount_", rowCount);
  run;

  %***********************************;
  %* Sort the data set               *;
  %***********************************;
  %if (&_lsafRowCount_ > 0) %then
  %do;
    options nonotes;
    proc sort data=&sas_result_dsname out=&sas_result_dsname sortseq=linguistic(strength=primary) ;
      by path;
    run;
    options &notesOption.;
  %end;

%mend lsaf_deleterecyclebinitems;
