%* lsaf_deleterbitemsbyquery                                                                     *;
%*                                                                                               *;
%* Permanently deletes items from the recycle bin that match the specified criteria. The paths   *;
%* of the items that were permanently deleted are returned in a SAS data set.                    *;
%*                                                                                               *;
%* macrotype LSAF                                                                                *;
%* since 2.2                                                                                     *;
%* exposure external                                                                             *;
%*                                                                                               *;
%* History                                                                                       *;
%* 2019-08-29  original coding.                                                                  *;
%* 2019-11-01  update to handle a blank SAS output file.                                         *;
%* 2020-07-13  update for refactor of java service hierarchy.                                    *;
%* 2021-08-12  Remove header documentation                                                       *;
%* 2021-08-31  DE16792                                                                           *;
%* 2021-10-12  US11739                                                                           *;
%* 2021-11-22  SAS Macros 2.6: RecycleBin refactor                                               *;
%*                                                                                               *;

%macro lsaf_deleterbitemsbyquery(
  lsaf_location=,
  lsaf_name=,
  lsaf_deletedby=,
  lsaf_from=,
  lsaf_to=,
  lsaf_includesubfolders=,
  sas_result_dsname=work.lsafdeleterbitemsbyquery
  ) /des='Deletes the items from the recycle bin that match the given criteria.';

  %if "&sas_result_dsname" EQ "" %then %let sas_result_dsname=work.lsafdeleterbitemsbyquery;

  %****************************************;
  %* Initialize macro variables           *;
  %****************************************;
  %lsaf_getProductName();

  %global _lsafMsg_
          _lsafRC_
          _lsaf_PrintMessage_
  ;

  %let _lsafRC_ =%str(-999);
  %let _lsafMsg_=%str(The SAS Macro did not execute properly.  Unknown err%str(or).);

  %local _lsafRowCount_
         notesOption
  ;

  %let _lsafRowCount_=0;

  %**********************************************;
  %* Start the datastep to call the java code   *;
  %**********************************************;
  data &sas_result_dsname (label="Created by &SYSMACRONAME" keep=path name);
    attrib returnCode         length=8
           message            length=$200
           logMsgPrefix       length=$70
           logMsgPrimary      length=$70
           logMsg             length=$300
           rowCount           length=8
           path               length=$2048    label="Item Path"
           name               length=$256     label="Item Name"
    ;
    
    rowCount=0;

    %***********************************;
    %* Declare the java objects        *;
    %***********************************;
    declare javaobj srv("com/sas/lsaf/macro/content/repository/recyclebin/RecycleBinService") ;
    declare javaobj results("com/sas/lsaf/macro/common/result/SasPathResult");
    declare javaobj query("com/sas/lsaf/macro/content/repository/recyclebin/SasRecycleBinQuery", 
                                  "%bquote(&lsaf_location)", "%bquote(&lsaf_name)", 
                                  "%bquote(&lsaf_deletedby)", "%bquote(&lsaf_from)", "%bquote(&lsaf_to)", 
                                  "%bquote(&lsaf_includesubfolders)"); 

    %***********************************;
    %* Attempt the operation           *;
    %***********************************;
    srv.callStaticVoidMethod("deleteRecycleBinItemsByQuery", query,  results);

    %***********************************;
    %* Retrieve the results            *;
    %***********************************;
    results.callIntMethod("getReturnCode", returnCode);
    results.callStringMethod("getReturnMessage", message);
    logMsgPrefix="NOTE: &_LSAF_PRODUCT_NAME_ Macro: * ";
    if (returnCode ne 0) then
    do;
      logMsgPrimary="ERR%str(OR): &_LSAF_PRODUCT_NAME_ Macro:";
      logMsg=" Could not permanently delete items from the recycle bin." ;
    end;
    else
    do;
      logMsgPrimary=logMsgPrefix;
      logMsg=" Returned the list of paths deleted from the recycle bin.";
      results.callIntMethod("getSize", rowCount);
      do row=0 to rowCount - 1;
        results.callStringMethod("getPath", row, path);
        results.callStringMethod("getName", row, name);
        output;
      end;
    end;

    %***********************************;
    %* Print messages in log           *;
    %***********************************;
    put;
    put logMsgPrimary " " logMsg;
    put logMsgPrefix " " %sysfunc(lowcase("&SYSMACRONAME"));
    if (rowCount=0) 
      then put logMsgPrefix " No items were deleted from the recycle bin.";
      else put logMsgPrefix " " rowCount " records written to data set &sas_result_dsname";
    put logMsgPrefix " _lsafMsg_= " message ;
    put logMsgPrefix " _lsafRC_= " returnCode ;
    put;

    %***********************************;
    %* Set the macro variables         *;
    %***********************************;
    call symputx("_lsafRC_", returnCode);
    call symputx("_lsafMsg_", message);
    call symputx("_lsafRowCount_", rowCount);
  run;

  %***********************************;
  %* Sort the data set               *;
  %***********************************;
  %if (&_lsafRowCount_ > 0) %then
  %do;
    %let notesOption=%sysfunc(getoption(notes));
    options nonotes;
    proc sort data=&sas_result_dsname out=&sas_result_dsname sortseq=linguistic(strength=primary);
      by path;
    run;
    options &notesOption;
  %end;

%mend lsaf_deleterbitemsbyquery;
