%* lsaf_createworkspacejob                                                                           *;
%*                                                                                                   *;
%* Creates a job in the workspace of the current user. The SAS macro variable _lsafWorkspaceJobPath_ *;
%* is set to the full path of the created job, which includes the file extension. If the job is not  *;
%* created, _lsafWorkspaceJobPath_ is set to blank.                                                  *;
%*                                                                                                   *;
%* macrotype LSAF                                                                                    *;
%* since 1.5                                                                                         *;
%* exposure external                                                                                 *;
%*                                                                                                   *;
%* History                                                                                           *;
%* 2016-09-23  initial coding                                                                        *;
%* 2019-02-11  update to 2.1                                                                         *;
%* 2019-08-28  update file path length to $2048                                                      *;
%* 2019-12-03  update header documentation to include job manifest macros.                           *;
%* 2020-06-23  update to accept use of relative paths.                                               *;
%* 2020-08-11  update parameters and column names to reflect LSAF 5.3 compatibility.                 *;
%* 2021-04-07  update for java code refactor                                                         *;
%* 2021-08-06  Remove header documentation                                                           *;
%* 2021-09-20  Remove parameter defaults (DE16775)                                                   *;
%* 2023-03-21  DE17654 - SAS Notes option updates                                                    *;
%*                                                                                                   *;

%macro lsaf_createworkspacejob(
   lsaf_path=,
   lsaf_userelativepaths=,
   sas_dsname_info=%str(_lsaf_createworkspacejob_ds_notprovided_),
   sas_dsname_programs=%str(_lsaf_createworkspacejob_ds_notprovided_),
   sas_dsname_parameters=%str(_lsaf_createworkspacejob_ds_notprovided_),
   sas_dsname_setup=%str(_lsaf_createworkspacejob_ds_notprovided_),
   sas_dsname_checkin=%str(_lsaf_createworkspacejob_ds_notprovided_)
   ) /des='Create a job in the workspace.';

   %****************************************;
   %* Initialize macro variables           *;
   %****************************************;
  %lsaf_getProductName();

   %global _lsafMsg_
           _lsafRC_
           _lsafWorkspaceJobPath_
   ;

   %local haveData
          logMsgPrefix
          logMsgPrimary
          macroName
          notesOption
   ;

   %let haveData =%str(1);
   %let _lsafRC_ =%str(-999);
   %let _lsafMsg_=%str(The SAS Macro did not execute properly. Unknown err%str(or).);
   %let _lsafWorkspaceJobPath_=;

   %let notesOption=%sysfunc(getoption(notes));

   %*********************************************************************************;
   %* Validate all setup data sets and create master data set to set server values. *;
   %*********************************************************************************;
   %let macroName=%sysfunc(lowcase(&SYSMACRONAME));
   %lsaf_process_jobdatasets(callingMacro=&macroName,
                              sas_dsname_info=&sas_dsname_info,
                              sas_dsname_programs=&sas_dsname_programs,
                              sas_dsname_parameters=&sas_dsname_parameters,
                              sas_dsname_setup=&sas_dsname_setup,
                              sas_dsname_checkin=&sas_dsname_checkin
   );
   %let logMsgPrimary=%str(ERR%str(OR): &_LSAF_PRODUCT_NAME_ Macro: );
   %let logMsgPrefix=%str(NOTE: &_LSAF_PRODUCT_NAME_ Macro: );

   options &notesOption.;

   %if (&_lsafRC_ ne 0) %then
   %do;
      %GOTO PRINT_DSERROR_LOG_MSGS;
   %end;
   %if (&_lsafMASTERJobDATA_tempDS_ EQ %str(_null_)) %then
   %do;
      %LET haveData=0;
   %end;

   %let _lsafRC_ =%str(-999);

   %**********************************************;
   %* Process the records                        *;
   %**********************************************;
   data _null_;
      attrib
          returnCode      length=8
          message         length=$200
          jobPath         length=$2048
          logMsgPrefix    length=$70
          logMsgPrimary   length=$70
          logMsg          length=$300
      ;
      jobPath="";

      %if (&haveData EQ 1) %then
      %do;
         set &_lsafMASTERJobDATA_tempDS_ end=eof;
         name=strip(name);

      if (_n_ EQ 1) then
      do;
      %end;
         %***********************************;
         %* Declare the java objects, once  *;
         %***********************************;
         declare javaobj srv("com/sas/lsaf/macro/execution/job/JobService");
         declare javaobj results("com/sas/lsaf/macro/common/result/Result");
         declare javaobj info("com/sas/lsaf/macro/execution/job/SasJobInfo");
         declare javaobj programs("com/sas/lsaf/macro/execution/job/SasJobPrograms");
         declare javaobj params("com/sas/lsaf/macro/execution/job/SasJobParameters");
         declare javaobj setup("com/sas/lsaf/macro/execution/job/SasJobPublishSetups");
         declare javaobj checkin("com/sas/lsaf/macro/execution/job/SasJobPublishCheckinLocations");
      %if (&haveData EQ 1) %then
      %do;
      end;

      if (source EQ "&source_info") then
      do;
         name=kstrip(lowcase(name));
         select (name);
            when ("documentation") info.callVoidMethod("setDescription", kstrip(value));
            when ("logslocation") info.callVoidMethod("setLogsLocation", kstrip(value));
            when ("resultslocation") info.callVoidMethod("setResultsLocation", kstrip(value));
            when ("manifestlocation") info.callVoidMethod("setManifestLocation", kstrip(value));
            when ("newoutputs_customversion") info.callVoidMethod("setCustomVersion", kstrip(value));
            when ("newoutputs_enableversion") info.callVoidMethod("setEnableVersioning", kstrip(value));
            when ("newoutputs_versiontype") info.callVoidMethod("setVersionType", kstrip(upcase(value)));
            otherwise;
         end;
      end;
      else if (source EQ "&source_parms") then
      do;
         params.callVoidMethod("addValue", kstrip(name), kstrip(pLabel), kstrip(type), kstrip(value), kstrip(version), includeSubFolders);
      end;
      else if (source EQ "&source_programs") then
      do;
         programs.callVoidMethod("addItem", kstrip(value), kstrip(version));
      end;
      else if (source EQ "&source_setup") then
      do;
         setup.callVoidMethod("addItem", kstrip(value), kstrip(type), kstrip(version), includeSubFolders, includeFiles);
      end;
      else if (source EQ "&source_checkin") then
      do;
         checkin.callVoidMethod("addItem", kstrip(value), includeSubFolders);
      end;

      %***********************************;
      %* Attempt the operation           *;
      %***********************************;
      if (eof);
      %end;

      srv.callStaticStringMethod("createWorkspaceJob", "%bquote(&lsaf_path)", "&lsaf_userelativepaths", info, programs, params, setup, checkin, results, jobPath);

      %***********************************;
      %* Retrieve the results            *;
      %***********************************;
      results.callIntMethod("getReturnCode", returnCode);
      results.callStringMethod("getReturnMessage", message);

      %***********************************;
      %* Process the results             *;
      %***********************************;
      logMsgPrefix="NOTE: &_LSAF_PRODUCT_NAME_ Macro: * ";
      if (returnCode ne 0) then
      do;
         logMsgPrimary="ERR%str(OR): &_LSAF_PRODUCT_NAME_ Macro:";
         logMsg=" Could not create job in the workspace: &lsaf_path.";
      end;
      else
      do;
         logMsgPrimary=logMsgPrefix;
         logMsg=" Created job in workspace using path &lsaf_path.";
      end;

      %***********************************;
      %* Print messages in log           *;
      %***********************************;
      put;
      put logMsgPrimary " " logMsg;
      put logMsgPrefix " " %sysfunc(lowcase("&SYSMACRONAME"));
      put logMsgPrefix " _lsafMsg_= " message;
      put logMsgPrefix " _lsafRC_= " returnCode;

      %if (&haveData NE 1) %then
      %do;
         put logMsgPrefix " No job data was provided.";
      %end;
      %else
      %do;
         %if (&sas_dsname_info NE _lsaf_createworkspacejob_ds_notprovided_) %then
         %do;
            put logMsgPrefix " sas_dsname_info = &sas_dsname_info";
         %end;
         %if (&sas_dsname_programs NE _lsaf_createworkspacejob_ds_notprovided_) %then
         %do;
            put logMsgPrefix " sas_dsname_programs = &sas_dsname_programs";
         %end;
         %if (&sas_dsname_parameters NE _lsaf_createworkspacejob_ds_notprovided_) %then
         %do;
            put logMsgPrefix " sas_dsname_parameters = &sas_dsname_parameters";
         %end;
         %if (&sas_dsname_setup NE _lsaf_createworkspacejob_ds_notprovided_) %then
         %do;
            put logMsgPrefix " sas_dsname_setup = &sas_dsname_setup";
         %end;
         %if (&sas_dsname_checkin NE _lsaf_createworkspacejob_ds_notprovided_) %then
         %do;
            put logMsgPrefix " sas_dsname_checkin = &sas_dsname_checkin";
         %end;
      %end;

      put logMsgPrefix " _lsafWorkspaceJobPath_=" jobPath;

      %***********************************;
      %* Set the macro variables         *;
      %***********************************;
      call symputx("_lsafRC_", returnCode);
      call symputx("_lsafMsg_", message);
      call symputx("_lsafWorkspaceJobPath_", kstrip(jobPath));

   run;
   %if (%sysfunc(exist(&_lsafMASTERJobDATA_tempDS_)) > 0) %then
   %do;
     options nonotes;
     proc datasets library=work nolist;
        delete &_lsafMASTERJobDATA_tempDS_;
     quit;
     run;
     options &notesOption.;
   %end;

   %return;

   %PRINT_DSERROR_LOG_MSGS:
      options &notesOption;
      %put &logMsgPrimary  &macroName ;
      %put &logMsgPrefix _lsafRC_= &_lsafRC_ ;
      %put &logMsgPrefix _lsafMsg_= &_lsafMsg_ ;
      %put;

   options &notesOption.;

%mend lsaf_createworkspacejob;
