%* lsaf_createintervalschedule                                                                       *;
%*                                                                                                   *;
%* Creates a schedule that runs at the specified interval.                                           *;
%*                                                                                                   *;
%* macrotype LSAF                                                                                    *;
%* since 2.3                                                                                         *;
%* exposure external                                                                                 *;
%*                                                                                                   *;
%* History                                                                                           *;
%* 2019-10-01  initial code                                                                          *;
%* 2020-06-30  update service location                                                               *;
%* 2021-08-16  Remove header documentation                                                           *;
%* 2021-09-28  Update for java refactoring                                                           *;
%*                                                                                                   *;

%macro lsaf_createintervalschedule(
  lsaf_intervaltype=,
  lsaf_interval=,
  lsaf_name=,
  lsaf_description=,
  lsaf_location=,
  lsaf_path=,
  lsaf_version=,
  sas_dsname=,
  lsaf_startdate=,
  lsaf_endtype=,
  lsaf_endvalue=
  ) /des='Creates a schedule that runs based on the specified interval.';

  %****************************************;
  %* Initialize macro variables           *;
  %****************************************;
  %lsaf_getProductName();

  %global _lsafMsg_
          _lsafIntervalScheduleId_
          _lsafRC_
  ;

   %local logMsgPrefix
          logMsgPrimary
          macroName
          notesOption
          parametersProvided
          _tempRUNJOB_tempDS_
   ;

   %let _lsafIntervalScheduleId_=;
   %let _lsafRC_ =%str(-999);
   %let _lsafMsg_=%str(The SAS Macro did not execute properly.  Unknown err%str(or).);

   %let macroName=%sysfunc(lowcase(&SYSMACRONAME));
   %let logMsgPrimary=%str(ERR%str(OR): &_LSAF_PRODUCT_NAME_ Macro: );
   %let logMsgPrefix=%str(NOTE: &_LSAF_PRODUCT_NAME_ Macro: );
   %let parametersProvided=0;

   %*************************************************************;
   %* Turn off the notes generated for all the pre-processing.  *;
   %* Save original option setting so it can be reset properly. *;
   %*************************************************************;
   %let notesOption=%sysfunc(getoption(notes));

   %if ( %CMPRES(&sas_dsname) NE %str() ) %then
   %do;

      %***************************************************************************;
      %* Validate the parameters data set if supplied.                           *;
      %* sets global macro variable _lsafRC_=0 if validation successful           *;
      %***************************************************************************;
      %lsaf_validate_parameters_dataset(callingMacro=&macroName,
                                      sas_dsn=&sas_dsname);
      %if (&_lsafRC_ ne 0) %then
      %do;
         %GOTO PRINT_DSERROR_LOG_MSGS;
      %end;

      %let parametersProvided=1;
   %end;

   options &notesOption;

   %**********************************************;
   %* Process the records                        *;
   %**********************************************;
   data _null_;
      attrib returnCode      length=8
             message         length=$200
             scheduleId      length=$50
             logMsgPrefix    length=$70
             logMsgPrimary   length=$70
             logMsg          length=$300
      ;

      %if (&parametersProvided) %then
      %do;
         set &sas_dsname end=eof;
         if (_n_ = 1) then
         do;
      %end;

      %***********************************;
      %* Declare the java objects, once  *;
      %***********************************;
      declare javaobj params("com/sas/lsaf/macro/execution/job/SasParameterValues");
      declare javaobj srv("com/sas/lsaf/macro/execution/schedule/ScheduleService");
      declare javaobj results("com/sas/lsaf/macro/common/result/Result");

      %if (&parametersProvided) %then
      %do;
         end;
         params.callVoidMethod("addValue", kstrip(name), kstrip(value), kstrip(type), kstrip(fileVersion), includeSubFolders);
         if (eof);
      %end;

      %***********************************;
      %* Create the schedule             *;
      %***********************************;
      declare javaobj schedule("com/sas/lsaf/macro/execution/schedule/SasSchedule", 
            "%bquote(&lsaf_name)", "%bquote(&lsaf_description)",
            "%bquote(&lsaf_startdate)", "%bquote(&lsaf_intervaltype)", "%bquote(&lsaf_interval)", 
            "%bquote(&lsaf_endtype)", "%bquote(&lsaf_endvalue)");

      declare javaobj job("com/sas/lsaf/macro/execution/schedule/SasSchedulableJob", 
            "%bquote(&lsaf_location)", "%bquote(&lsaf_path)", "%bquote(&lsaf_version)", params);

      srv.callStaticStringMethod("createIntervalSchedule", schedule, job, results, scheduleId);

      %***********************************;
      %* Retrieve the results            *;
      %***********************************;
      results.callIntMethod("getReturnCode", returnCode);
      results.callStringMethod("getReturnMessage", message);

      %***********************************;
      %* Process the results             *;
      %***********************************;
      logMsgPrefix="NOTE: &_LSAF_PRODUCT_NAME_ Macro: * ";
      if (returnCode ne 0) then
      do;
         logMsgPrimary="ERR%str(OR): &_LSAF_PRODUCT_NAME_ Macro:";
         logMsg=" Cannot create schedule &lsaf_name.";
      end;
      else
      do;
         logMsgPrimary=logMsgPrefix;
         logMsg=" Created schedule &lsaf_name.";
      end;

      %***********************************;
      %* Print messages in log           *;
      %***********************************;
      put;
      put logMsgPrimary " " logMsg;
      put logMsgPrefix " " %sysfunc(lowcase("&SYSMACRONAME"));
      put logMsgPrefix " _lsafMsg_= " message ;
      put logMsgPrefix " _lsafRC_= " returnCode ;
      if "&sas_dsname" = "" 
        then put logMsgPrefix "No job parameters were supplied.";
        else put logMsgPrefix "sas_dsname = &sas_dsname";
      put logMsgPrefix "_lsafIntervalScheduleId_= " scheduleId;
      put;

      %***********************************;
      %* Set the macro variables         *;
      %***********************************;
      call symputx("_lsafRC_", returnCode);
      call symputx("_lsafMsg_", message);
      call symputx("_lsafIntervalScheduleId_", strip(scheduleId));

   run;

   %return;

   %PRINT_DSERROR_LOG_MSGS:
      options &notesOption;
      %put &logMsgPrimary  &macroName ;
      %put &logMsgPrefix _lsafRC_= &_lsafRC_ ;
      %put &logMsgPrefix _lsafMsg_= &_lsafMsg_ ;
      %put &logMsgPrefix Parameter data set = &sas_dsname;
      %put;

%mend lsaf_createintervalschedule;
