%* lsaf_completepfusertask                                                                       *;
%*                                                                                               *;
%* Completes a process flow task for the current user.                                           *;
%*                                                                                               *;
%* macrotype LSAF                                                                                *;
%* since 2.5                                                                                     *;
%* exposure external                                                                             *;
%*                                                                                               *;
%* History                                                                                       *;
%* 2021-03-03  initial programming                                                               *;
%* 2021-08-12  Remove header documentation                                                       *;
%* 2023-03-09  DE17654 - SAS Notes option updates                                                *;
%*                                                                                               *;

%macro lsaf_completepfusertask(
  lsaf_taskid=,
  lsaf_actualhours=,
  lsaf_comment=,
  sas_dsname=
  ) /des='Allows user to complete a process flow user task.';

  %****************************************;
  %* Initialize macro variables           *;
  %****************************************;
  %lsaf_getProductName();

  %global _lsafMsg_
          _lsafRC_
  ;

  %local _lsaf_tempdataID_
         _lsaf_tempdsobscnt_
         logMsgPrefix
         logMsgPrimary
         macroName
		 notesOption
         RC
  ;

  %let _lsafRC_ =%str(-999);
  %let _lsafMsg_=%str(The SAS Macro did not execute properly.  Unknown err%str(or).);

  %***************************************************************************;
  %* Set local macro variables for processing.                               *;
  %***************************************************************************;
  %let macroName=%sysfunc(lowcase(&SYSMACRONAME));
  %let logMsgPrimary=%str(ERR%str(OR): &_LSAF_PRODUCT_NAME_ Macro: );
  %let logMsgPrefix=%str(NOTE: &_LSAF_PRODUCT_NAME_ Macro: );

  %*************************************************************;
  %* Save original option setting so it can be reset properly. *;
  %*************************************************************;
  %let notesOption=%sysfunc(getoption(notes));

  %**********************************;
  %*  Deal with a missing data set  *;
  %**********************************;
  %if "&sas_dsname" eq "" %then
  %do;
    %let _lsaf_tempdsobscnt_=0;
  %end;
  %else
  %do;
    %*************************************************************;
    %* Turn off the notes generated for all the pre-processing.  *;
    %* Save original option setting so it can be reset properly. *;
    %*************************************************************;
    options nonotes;
    
    %***************************************************************************;
    %* Validate the input data set                                             *;
    %* sets global macro variable _lsafRC_=0 if validation successful          *;
    %***************************************************************************;
    %lsaf_validate_sasdataset(callingMacro=&macroName,
          sas_dsn=&sas_dsname,
          required_variables=%str(dataId dataValue),
          allow_empty_datasets=1
          );
    options &notesOption.;    
    %if (&_lsafRC_ ne 0) %then
    %do;
      %put &logMsgPrimary _lsafMsg_= &_lsafMsg_ ;
      %put &logMsgPrefix &macroName ;
      %put &logMsgPrefix Input data set = &sas_dsname;
      %put &logMsgPrefix _lsafRC_= &_lsafRC_ ;
      %put;
      %return;
    %end;

    %***************************************************************************;
    %* If data set is valid, process it.                                       *;
    %***************************************************************************;
    %put NOTE: &macroName:  Processing data set &sas_dsname;

    %**********************************;
    %*  Deal with a zero obs dataset  *;
    %**********************************;
    %let _lsaf_tempdataID_=%sysfunc(OPEN(&sas_dsname.,IN));
    %let _lsaf_tempdsobscnt_=%sysfunc(ATTRN(&_lsaf_tempdataID_,NOBS));
    %let RC=%sysfunc(CLOSE(&_lsaf_tempdataID_));
  %end;
  
  %**********************************************;
  %* Process the records                        *;
  %**********************************************;
  data _null_;
    attrib returnCode    length=8
           message       length=$200
    ;

    %if &_lsaf_tempdsobscnt_ ne 0 %then 
    %do;
      set &sas_dsname end=eof;
    %end;
    %else
    %do;
      _n_=1;
      eof=1;
    %end;

    %***********************************;
    %* Declare the java objects, once  *;
    %***********************************;
    if (_n_ = 1) then
    do;
      declare javaobj completetask("com/sas/lsaf/macro/common/SasProperties");
      declare javaobj srv("com/sas/lsaf/macro/workflow/task/ProcessFlowTaskService");
      declare javaobj results("com/sas/lsaf/macro/common/result/Result");
    end;

    %***************************************;
    %* Create a new completion data list   *;
    %**************************************;
    %if &_lsaf_tempdsobscnt_ ne 0 %then 
    %do;
      completetask.callVoidMethod("addProperty", kstrip(dataId), kstrip(dataValue));
    %end;
    
    if eof then
    do;
      %***********************************;
      %* Call the update                 *;
      %***********************************;
      srv.callStaticVoidMethod("completeTask", "%bquote(&lsaf_taskid)", "%bquote(&lsaf_actualhours)", completetask, "%bquote(&lsaf_comment)", results);

      %***********************************;
      %* Retrieve the results            *;
      %***********************************;
      results.callIntMethod("getReturnCode", returnCode);
      results.callStringMethod("getReturnMessage", message);

      %***********************************;
      %* Process the results             *;
      %***********************************;
      logMsgPrefix="NOTE: &_LSAF_PRODUCT_NAME_ Macro: * ";
      logMsgPrimary=logMsgPrefix;
      logMsg=" Complete process flow user task ran successfully for task &lsaf_taskid.." ;
      if (returnCode ne 0) then
      do;
        logMsgPrimary="ERR%str(OR): &_LSAF_PRODUCT_NAME_ Macro:";
        logMsg=" Cannot complete process flow user task for task &lsaf_taskid.." ;
      end;

      %***********************************;
      %* Print messages in log           *;
      %***********************************;
      put;
      put logMsgPrimary " " logMsg;
      put logMsgPrefix " " %sysfunc(lowcase("&SYSMACRONAME"));
      put logMsgPrefix " Input data set = &sas_dsname";
      put logMsgPrefix " _lsafMsg_= " message ;
      put logMsgPrefix " _lsafRC_= " returnCode ;
      put;
    
      %***********************************;
      %* Set the macro variables         *;
      %***********************************;
      call symputx("_lsafRC_", returnCode);
      call symputx("_lsafMsg_", message);
    end;
  run;

%mend lsaf_completepfusertask;
