// (c) 2016, SAS Institute Inc.
sap.ui.define([
    "./Table",
    "./TreeTable",
    "./library",
    "./WideTableRenderer",
    "./WideRow",
    "./ItemNavigation",
    "./TableUtils",
    "./TableKeyboardExtension",
    "sap/ui/Device",
    "sap/ui/model/ListBinding",
    "sap/ui/base/ManagedObjectMetadata",
    "sap/ui/core/Core"
], function(SasHcUiTable, SasHcUiTreeTable, TableLibrary, WideTableRenderer, WideRow, SasItemNavigation, TableUtils, SasKeyboardExtension, Device, ListBinding, ManagedObjectMetadata) {
    "use strict";

    var SelectionBehavior = TableLibrary.SelectionBehavior;
    var SelectionMode = TableLibrary.SelectionMode;

    /**
     * Controls how often the horizontal scroll update can happen. Updating the DOM on every scroll
     * event would cause too much DOM manipulation.
     *
     * TODO: Using a value of 0 for now to always render updated DOM. Should do performance testing
     * to determine if a different value is needed.
     *
     * TODO: Additional thought, during the scroll handler we might be able to do some runtime
     * performance monitoring. If we notice the handler takes too long to run we can dynamically
     * change this value to help improve stability.
     */
    var I_HORIZONTAL_SCROLL_UPDATE_THROTTLE = 0;

    /**
     * The number of Columns to instantiate before yielding thread control as Column instances are generated
     * from the data bound to the 'columns' aggregation.  A smaller value yields thread control more often
     * allowing the browser to be more responsive to the user.  A larger value may cause the browser to
     * warn the user a javascript task is consuming excessive resources.
     * @type {number}
     */
    var DEFAULT_ASYNC_COLGEN_BATCH_SIZE = 10;

    /**
     * A Prefix Sum is also known as cumulative sum. A Prefix Sum Table represents the cumulative
     * sum of elements in a list or array.
     *
     * For this implementation we should assume an array of monotonically non-decreasing sums.
     *
     * This is an initial, simple implementation. In future can replace with more advanced
     * underlying data structure such as Fenwick Tree. The focus on improvements should probably
     * revolve around fast updates and searches. There may also be a need for fast inserts and
     * removals.

     * Current run times are:
     *
     * Create: O(n)
     * Update: O(n)
     * Search: O(n)
     * Insert prefix: Supported only via recreating. O(n)
     * Remove prefix: Supported only via recreating. O(n)
     * Move prefix: Supported only via recreating. O(n)
     */
    var PrefixSumTable = function(prefixArray) {

        var self = this;
        this.aPrefixes = [];
        this.aSums = [];

        var iRunningTotal = 0;
        prefixArray.forEach(function (iPrefix, iIndex) {
            // Make a copy for internal use.
            self.aPrefixes.push(iPrefix);

            // Generate sums.
            iRunningTotal += iPrefix;
            self.aSums.push(iRunningTotal);
        });
    };

    PrefixSumTable.prototype.updatePrefix = function(iIndex, iNewValue) {
        var iDiff = iNewValue - this.aPrefixes[iIndex];

        if (iDiff === 0) {
            return;
        }

        this.aPrefixes[iIndex] = iNewValue;

        for(var i = iIndex, l=this.aSums.length; i < l; i++) {
            this.aSums[i] += iDiff;
        }
    };

    PrefixSumTable.prototype.getSumTo = function(iIndex) {
        if (iIndex < 0) {
            return 0;
        }
        return this.aSums[iIndex];
    };

    PrefixSumTable.prototype.getPrefixAt = function(iIndex) {
        return this.aPrefixes[iIndex];
    };

    /**
     * Assumes no negative prefix values.
     */
    PrefixSumTable.prototype.getFirstIndexWithSumGreaterThanOrEqualTo = function(iValue) {

        // For column rendering use case, will likely need to return only indices that have a non-zero prefix (width).
        // Other possible solution is to only render columns that have a non-zero width.

        for (var i = 0, l = this.aSums.length - 1; i <= l; i++) {
            if (this.aSums[i] >= iValue) {
                return i;
            }
        }
        // Return length of sums to represent the index the caller wants would be beyond the values
        // of this table.
        return i;
    };

    /**
     * Return the sum of all values.
     */
    PrefixSumTable.prototype.getTotalSum = function() {
        return this.aSums[this.aSums.length - 1];
    };

    /*************************************************************************************************************/

    /**
     * TableUtils Overrides
     */
    var fnOriginalGetFocusedItemInfo = TableUtils.getFocusedItemInfo;

    TableUtils.getFocusedItemInfo = function(oTable) {
        // If the oTable instance is a WideTable
        if (oTable._isWideTableVariant && oTable._isWideTableVariant()) {
            var oIN = oTable._getItemNavigation();
            if (!oIN) {
                return null;
            }

            var displayedColumnCount = oTable.getDisplayedColumnCount();

            if (TableUtils.hasRowHeader(oTable) === true) {
                displayedColumnCount++;
            }

            return {
                cell: oIN.getFocusedIndex(),
                columnCount: oIN.iColumns,
                cellInRow: oIN.getFocusedIndex() % displayedColumnCount,
                row: Math.floor(oIN.getFocusedIndex() / displayedColumnCount),
                cellCount: oIN.getItemDomRefs().length,
                domRef: oIN.getFocusedDomRef()
            };
        } else {
            return fnOriginalGetFocusedItemInfo(oTable);
        }
    };

    TableUtils.scrollDownRow = function(oTable) {
        TableUtils.scroll(oTable, true, false);
    };

    TableUtils.scrollDownPage = function(oTable) {
        TableUtils.scroll(oTable, true, true);
    };

    TableUtils.scrollUpRow = function(oTable) {
        TableUtils.scroll(oTable, false, false);
    };

    TableUtils.scrollUpPage = function(oTable) {
        TableUtils.scroll(oTable, false, true);
    };

    /*************************************************************************************************************/

    // Because this is the X package and we need both WideTable and WideTreeTable.
    function createWideVariant(SasParentClass) {
        var sSasParentFullName = SasParentClass.getMetadata().getElementName();
        var sSasParentShortName = sSasParentFullName.split(".").pop();

        //var logger = TableLibrary._getLogger("Wide"+sSasParentShortName);

        var WideTableRenderer = sas.hc.ui.table["Wide" + sSasParentShortName + "Renderer"]; // eslint-disable-line sashtmlcommons/missing-require

        /**
         * Constructor for a new WideTable.
         *
         * @param {string} [sId] id for the new control, generated automatically if no id is given
         * @param {object} [mSettings] initial settings for the new control
         *
         * @class
         * <p>
         *   Enhances the regular Table control to improve rendering performance when there are many columns.
         *   Columns in WideTable must be given an absolute width (px or rem).
         * </p>
         *
         *
         * @extends sas.hc.ui.table.Table
         * @version 904001.11.16.20251118090100_f0htmcm94p
         *
         * @constructor
         * @public
         * @alias sas.hc.ui.table.WideTable
         */
        var WideTable = SasParentClass.extend("sas.hc.ui.table.Wide" + sSasParentShortName, {
            metadata: {
                library: "sas.hc.ui.table",
                properties : {

                    /**
                     * Set this parameter to true to make the table handle the busy indicator by its own during
                     * specific, possibly time consuming situations.
                     * @since 9.0
                     */
                    enableBusyIndicator : {type : "boolean", group : "Behavior", defaultValue : false},

                    /**
                     * Set this parameter to true to allow the WideTable to create Columns asynchronously when
                     * columns are generated from a bound model.  Asynchronous column generation allows for
                     * a large number of columns to be created for the WideTable without monopolizing the browser's
                     * resources and degrading the user experience.
                     */
                    enableAsyncColumnGeneration: {type: "boolean", group: "Performance", defaultValue: false}

                },
                events: {

                    /**
                     * Fired after <code>columns</code> binding is updated and processed by the control.
                     */
                    columnsUpdated : {
                        parameters : {
                            /**
                             * The reason of the update, e.g. change, filter, sort, unbindAggregation.
                             */
                            reason : {type : "string"}
                        }
                    }

                }
            },
            renderer: "sas.hc.ui.table.Wide" + sSasParentShortName + "Renderer"
        });

        WideTable.prototype.init = function() {
            SasParentClass.prototype.init.apply(this, arguments);

            this._iHorizontalScrollPosition = 0;
            this._iLastHorizontalScrollUpdate = new Date().getTime();

            /**
             * The number of Columns to generate asynchronously during each pass through the creation routine.  A larger
             * number here will potentially impact the browser's responsiveness.
             */
            this._iAsyncColumnGenerationFactor = DEFAULT_ASYNC_COLGEN_BATCH_SIZE;

            // Allow some functions to cache results as needed.
            this._mFnResultsCache = {};

            // Initialize data for _getColumnRangeToRender
            this._mFnResultsCache._getColumnRangeToRender = {
                lastResult: null
            };

            this.addStyleClass("sasUiWideTable");
        };

        /**
         * @private
         */
        WideTable.prototype._isWideTableVariant = function() {
            return true;
        };

        /**
         * Called when the Table's columns aggregation is bound or the bound data has been changed.  We override what
         * ManagedObject would normally do here because we want to closely manage how Column instantiation occurs.
         * @param {string} sReason the reason the columns aggregation has been updated
         * @see sap.ui.base.ManagedObject#updateAggregation
         */
        WideTable.prototype.updateColumns = function(sReason) {
            if (this._bExitCalled) {
                return;
            }

            //maybe call this._updateNoData() if the binding (of columns) length has changed

            if (this.getEnableAsyncColumnGeneration() !== true) {
                //simply call the Parent version
                SasParentClass.prototype.updateAggregation.call(this, "columns");

                //fire new event
                this.fireColumnsUpdated({
                    reason : sReason
                });

                return;
            }

            var iBatchSize = this._iAsyncColumnGenerationFactor;

            //copied from ManagedObject.updateAggregation and modified
            //======================================================================== ManagedObject.updateAggregation
            var oBindingInfo = this.getBindingInfo("columns"),
                oBinding = oBindingInfo.binding,
                fnFactory = oBindingInfo.factory,
                iBindingLength = oBindingInfo.length,
                oAggregationInfo = this.getMetadata().getAggregation("columns"),
                bGrouped,
                sGroup,
                sGroupFunction = oAggregationInfo._sMutator + "Group",
                self = this,
                iBusyIndicatorDelay = self.getBusyIndicatorDelay();

            // utility function to support cloning
            function getIdSuffix(oControl, iIndex) {
                if (self.bUseExtendedChangeDetection) {
                    return ManagedObjectMetadata.uid('clone');
                } else {
                    return oControl.getId() + "-" + iIndex;
                }
            }

            // called at the start of column generation
            function updateStarted() {
                // update busy indicator state
                if (self.getEnableBusyIndicator()) {
                    self.setBusyIndicatorDelay(0);
                    self.setBusy(true);
                }
                // set a flag to let the table know we'll be doing a lot with columns
                self._bGeneratingColumns = true;
            }

            // called when the async operation completes to ready the table for user interaction
            function updateFinished() {
                // turn off the generating columns flag
                self._bGeneratingColumns = false;
                // reset the row template
                self._resetRowTemplate();
                // invalidate the table
                self.invalidate();
                // yield thread control once more to allow invalidate/render cycle
                setTimeout(function() {
                    self.fireColumnsUpdated({
                        reason : sReason
                    });

                    // update busy indicator state
                    if (self.getEnableBusyIndicator()) {
                        self.setBusy(false);
                        self.setBusyIndicatorDelay(iBusyIndicatorDelay);
                    }
                }, 10);
            }

            // Update a single aggregation with the array of contexts. Reuse existing children
            // and just append or remove at the end, if some are missing or too many.
            function update(oControl, iStartIndex, fnBefore, fnAfter) {
                var aContexts = oBinding.getContexts(iStartIndex, iBatchSize),
                    aChildren = oControl[oAggregationInfo._sGetter]() || [],
                    oModel = oBinding.getModel(),
                    oContext,
                    oClone, i, iChildIndex;

                // because this operation will be asynchronous, consider that our control and/or model may be destroyed
                if (self.bDestroyed) {
                    return;
                }
                if (oModel.bDestroyed) {
                    updateFinished();   // restore table state before abandoning
                    return;
                }

                //remove excess children from aggregation and destroy them
                if (aChildren.length > iBindingLength) {
                    for (i = iBindingLength; i < aChildren.length; i++) {
                        oClone = aChildren[i];
                        oControl[oAggregationInfo._sRemoveMutator](oClone);
                        oClone.destroy("KeepDom");
                    }
                }
                //create new children and add them to the aggregation
                iChildIndex = iStartIndex;
                for (i = 0; i < aContexts.length; i++, iChildIndex++) {
                    oContext = aContexts[i];
                    if (!oContext) {
                        continue;
                    }

                    oClone = aChildren[iChildIndex];

                    if (fnBefore) {
                        fnBefore(oContext);
                    }
                    if (oClone) {
                        oClone.setBindingContext(oContext, oBindingInfo.model);
                    } else {
                        oClone = fnFactory(getIdSuffix(oControl, iChildIndex), oContext);
                        oClone.setBindingContext(oContext, oBindingInfo.model);
                        oControl[oAggregationInfo._sMutator](oClone);
                    }
                    if (fnAfter) {
                        fnAfter(oContext, oClone);
                    }
                }

                //check if we need to keep more
                if (iChildIndex < iBindingLength) {
                    setTimeout(function() {
                        update(oControl, iChildIndex, fnBefore, fnAfter);
                    }, 10);
                } else {
                    updateFinished();
                }
            }

            //TODO: updateDiff has not been copied from ManagedObject and reworked to be async yet (no known consumers)
            //TODO: updateGroup has been copied, but not reworked (no known consumers)

            // Check the current context for its group. If the group key changes, call the
            // group function on the control.
            function updateGroup(oContext) {
                var oNewGroup = oBinding.getGroup(oContext);
                if (oNewGroup.key !== sGroup) {
                    var oGroupHeader;
                    //If factory is defined use it
                    if (oBindingInfo.groupHeaderFactory) {
                        oGroupHeader = oBindingInfo.groupHeaderFactory(oNewGroup);
                    }
                    self[sGroupFunction](oNewGroup, oGroupHeader);
                    sGroup = oNewGroup.key;
                }
            }

            // check the binding and begin to process ListBinding contexts
            if (oBinding instanceof ListBinding) {
                iBindingLength = oBindingInfo.length || oBinding.getLength();
                bGrouped = oBinding.isGrouped() && self[sGroupFunction];

                // call our updateStarted helper function to update busy state and column generation flag
                updateStarted();

                if (bGrouped || oBinding.bWasGrouped) {
                    // If grouping is enabled, destroy aggregation and use updateGroup as fnBefore to create groups
                    this[oAggregationInfo._sDestructor]();
                    update(this, isNaN(oBindingInfo.startIndex) ? 0 : oBindingInfo.startIndex, bGrouped ? updateGroup : undefined);
                } else {
                    // If factory function is used without extended change detection, destroy aggregation
                    if (!oBindingInfo.template) {
                        this[oAggregationInfo._sDestructor]();
                    }
                    update(this, isNaN(oBindingInfo.startIndex) ? 0 : oBindingInfo.startIndex);
                }
                oBinding.bWasGrouped = bGrouped;
            }
            //======================================================================== ManagedObject.updateAggregation
        };

        /**
         * Override invalidation to improve performance during asynchronous column generation.
         * @param {sap.ui.base.ManagedObject} [oOrigin] Child control for which the method was called
         * @returns {void|*|any}
         */
        WideTable.prototype.invalidate = function(oOrigin) {
            // ignore invalidation if we're running through the async column generation routine
            if (!this._bGeneratingColumns) {
                return SasParentClass.prototype.invalidate.call(this, oOrigin);
            }
        };

        /**
         * Computes the number of Columns currently rendered.
         * @public
         * @returns {Number} The number of Columns currently rendered.
         */
        WideTable.prototype.getDisplayedColumnCount = function() {
            var oRenderRange = this._getColumnRangeToRender();
            return this.getTotalFrozenColumnCount() + oRenderRange.visibleColumnsCount;
        };

        /**
         * @private
         */
        WideTable.prototype._attachExtensions = function() {
            SasParentClass.prototype._attachExtensions.apply(this, arguments);
        };

        /**
         * @private
         */
        WideTable.prototype._onTableResize = function() {
            SasParentClass.prototype._onTableResize.apply(this, arguments);
            if (this._bInvalid || !this.getDomRef()) {
                return;
            }
            if (this._hasColumnRenderRangeChanged()) {
                this.refreshScrollableContent(true);
            } else {
                this._stretchTableAndColumns();
            }
        };

        WideTable.prototype.onBeforeRendering = function() {
            this._createColumnWidthsTable();
            this._getColumnRangeToRender(true);
            this._updateBindingContexts(false);
            SasParentClass.prototype.onBeforeRendering.apply(this, arguments);
        };

        WideTable.prototype.getCellDomRef = function(iRowIndex, iColIndex) {
            if (iRowIndex === -1) {
                return this.getColumns()[iColIndex].getDomRef();
            }
            return this.getDomRef("rows-row" + iRowIndex + "-col" + iColIndex);
        };

        WideTable.prototype.onAfterRendering = function(oEvent) {
            SasParentClass.prototype.onAfterRendering.apply(this, arguments);

            // cancel if the pseudo onAfterRendering event from refreshScrollableContent was fired
            if (oEvent && oEvent.isMarked("refreshScrollableContent")) {
                return;
            }

            var $hsb = this._getHsbRef();
            if ($hsb) {
                $hsb.scrollLeft(this._iHorizontalScrollPosition);
            }

            this.refreshScrollableContent();
        };

        /**
         * @private
         */
        WideTable.prototype._stretchTableAndColumns = function() {

            // Do not run stretching before first rendering phase has finished.
            if (this.$().hasClass("sapUiTableNoOpacity")) {
                return;
            }

            // Stretch the scrollable <table> to fill available space when the render range
            // is at the beginning and there aren't enough columns to fill it normally.
            var oColumnRange = this._getColumnRangeToRender();
            var iAvailableWidth = this.$().find(".sapUiTableCtrlCnt").width();

            if (oColumnRange.start === this.getTotalFrozenColumnCount()
                    && oColumnRange.end >= (this._getVisibleColumns().length - 1)) {
                this.$().find(".sapUiTableCtrl.sapUiTableCtrlScroll").css({
                    "width": iAvailableWidth + "px",
                    "min-width": iAvailableWidth + "px"
                });
            }

        };

        /**
         * Sets and renders the first visible column (left-most or right-most in RTL mode).
         * @param {int | sas.hc.ui.table.Column} oColumn The column that should be the first rendered column.
         * @param {boolean} bRetainFocus Indicate if cell focus should be retained when the columns are rendered.
         * @public
         */
        WideTable.prototype.makeFirstVisibleColumn = function(oColumn, bRetainFocus) {
            var iColumnIndex = null;
            var columns = this._getVisibleColumns();
            if (typeof oColumn === "object") {
                iColumnIndex = jQuery.inArray(oColumn, columns);
            } else {
                // Assume passed column variable is the index.
                iColumnIndex = oColumn;
            }

            if (iColumnIndex > columns.length) {
                iColumnIndex = columns.length - 1;
            }

            var widthsSumTable = this._getColumnWidthsSumTable();
            // The +1 helps make sure things scroll as expected.
            var newHScrollPos = (widthsSumTable.getSumTo(iColumnIndex) - widthsSumTable.getPrefixAt(iColumnIndex)) + 1;

            this._iHorizontalScrollPosition = newHScrollPos;
            this.refreshScrollableContent(bRetainFocus);

            var $hsb = this._getHsbRef();
            if ($hsb) {
                $hsb.scrollLeft(this._iHorizontalScrollPosition);
            }
        };

        /**
         * Tells the WideTable to rerender the content of the horiztonally scrollable section.
         *
         * @param {boolean} bRetainFocus If true, attempt to restore focus to the currently focused item after refresh. Defaults to true.
         * @public
         */
        WideTable.prototype.refreshScrollableContent = function(bRetainFocus) {
            // TODO: Test for what columns are already in the DOM.
            // We may not need to need to rerender anything here.

            // Default value of bRetainFocus should be true.
            bRetainFocus = (bRetainFocus === undefined || bRetainFocus === true) ? true : false;

            this._detachEvents();

            var oColumnRange = this._getColumnRangeToRender(true);
            this._updateBindingContexts(false);

            var aColumns = this._getVisibleColumns();

            var oItemNav = this._getItemNavigation();
            var iFocusedRowIndex = null;
            var iFocusedColumnIndex;
            if (bRetainFocus && oItemNav) {
                // Preserve focus info
                var currentFocusedDomRef = oItemNav.getFocusedDomRef();
                var focusedRow = this._rowFromRef(currentFocusedDomRef);
                if (focusedRow) {
                    iFocusedRowIndex = focusedRow.$().attr("data-sap-ui-rowindex");
                } else if (this._isColumnHeaderNavigation()) {
                    iFocusedRowIndex = -1;
                }

                var focusedColumn = this._colFromRef(currentFocusedDomRef);
                iFocusedColumnIndex = aColumns.indexOf(focusedColumn);
            }

            this._getPointerExtension().cleanupColumnResizeEvents();

            // Update header content
            var oRm = sap.ui.getCore().createRenderManager();
            WideTableRenderer.renderColHdr(oRm, this, true); // eslint-disable-line sashtmlcommons/missing-require
            oRm.flush(this.$().find('.sasUiTableColHdrCntShell')[0], true, false);
            this.$().find(".sapUiTableColHdr.sapUiTableNoOpacity").removeClass("sapUiTableNoOpacity");
            this.$().find(".sapUiTableColHdrFixed.sapUiTableNoOpacity").removeClass("sapUiTableNoOpacity");

            // Capture px scrolling
            var $scrollableTableBody = this.$().find(".sapUiTableCtrlCnt .sapUiTableCtrlScroll");
            var sPxScrollingStyle = $scrollableTableBody[0].style["transform"];

            // Update body content
            WideTableRenderer.renderTableControl(oRm, this); // eslint-disable-line sashtmlcommons/missing-require
            oRm.flush(this.$().find(".sapUiTableCtrlCnt")[0], true, false);

            // Restore px scrolling
            this.$().find(".sapUiTableCtrlCnt .sapUiTableCtrlScroll").css({
                "transform": sPxScrollingStyle
            });

            // Refresh ItemNavigation DOM refs
            var oKeyboardExt = this._getKeyboardExtension();
            oKeyboardExt.invalidateItemNavigation();
            oKeyboardExt.initItemNavigation();

            if (bRetainFocus
                && oItemNav
                && (jQuery.isNumeric(iFocusedRowIndex) && iFocusedRowIndex >= -1)
                && (jQuery.isNumeric(iFocusedColumnIndex) && iFocusedColumnIndex >= 0)
                // Dont restore focus if the column is clipped. Focusing a clipped column causes
                // one of the parent div elements to uncontrollably get a scrollLeft value set
                // by the browser that causes visual problems.
                && ((iFocusedColumnIndex !== oColumnRange.end)
                    || (iFocusedColumnIndex === oColumnRange.end && !oColumnRange.lastColIsClipped))) {
                // Restore focus info

                var oDomRefToFocus = this.getCellDomRef(iFocusedRowIndex, iFocusedColumnIndex);
                this._focusItemByDomRef(oDomRefToFocus);
            }

            var oTableSizes = this._collectTableSizes();
            this._syncColumnHeaders(oTableSizes);

            // Restore Column icons
            aColumns.slice(oColumnRange.start, oColumnRange.end)
                .forEach(function(oCol) {
                    oCol._updateIcons();
                });

            // Sync cell heights.
            this._updateRowHeader(oTableSizes.tableRowHeights);

            // TreeTable requires this call to indent first column to look like a tree.
            this._updateTableContent();

            // Need to fire a pseudo onAfterRendering event, (in the same spirit as insertTableRows)
            // This will inform delegates (such as the DraggableRowDelegate) that the DOM has been updated
            var oAfterRenderingEvent = jQuery.Event("AfterRendering");
            oAfterRenderingEvent.setMarked("refreshScrollableContent");
            oAfterRenderingEvent.srcControl = this;
            this._handleEvent(oAfterRenderingEvent);

            this._stretchTableAndColumns();

            this._attachEvents();
            TableUtils.registerResizeHandler(this, "", this._onTableResize.bind(this), true);
        };

        /**
         * @override
         */
        WideTable.prototype.onhscroll = function(oEvent) {
            // Update internal state
            var $hsb = this._getHsbRef();
            if ($hsb) {
                this._iHorizontalScrollPosition = this._getNormalizedScrollLeft();
            }

            // Throttle the update to the DOM
            var iNow = new Date().getTime();
            if ((iNow - this._iLastHorizontalScrollUpdate) > I_HORIZONTAL_SCROLL_UPDATE_THROTTLE) {
                this.refreshScrollableContent();
                this._iLastHorizontalScrollUpdate = new Date().getTime();
            }

            // Fire scroll event
            jQuery.sap.interaction.notifyScrollEvent && jQuery.sap.interaction.notifyScrollEvent(oEvent);
        };

        /**
         * The scrollLeft property of Elements is inconsistent in RTL scenarios. This method
         * will attempt to normalize these to not count left/right but begin/end. This is so
         * that calculating render ranges can be simplified and consistent. Using the result
         * of this method to then manipulate the position of the scrollbar has not been considered.
         *
         * @private
         */
        WideTable.prototype._getNormalizedScrollLeft = function() {
            var $hsb = this._getHsbRef();
            var reportedScrollLeft = $hsb.scrollLeft() || 0;

            // Browsers generally work the same in LTR mode. No need to normalize.
            if (!this._bRtlMode) {
                return reportedScrollLeft;
            }

            if ($hsb.length > 0 && Device.browser.chrome) {
                // return reportedScrollLeft;
                return $hsb[0].scrollWidth - $hsb.width() - $hsb.scrollLeft();
            }

            if (Device.browser.firefox || Device.browser.safari) {
                return -1 * reportedScrollLeft;
            }

            return reportedScrollLeft;
        };

        /**
         * @private
         */
        WideTable.prototype._getColumnWidthsSumTable = function() {
            if (!this.oColumnWidthsSumTable) {
                this._createColumnWidthsTable();
            }
            return this.oColumnWidthsSumTable;
        };

        /**
         * @private
         */
        WideTable.prototype._createColumnWidthsTable = function() {
            var self = this;
            var iFrozenColCount = this.getTotalFrozenColumnCount();
            this.oColumnWidthsSumTable = new PrefixSumTable(
                this.retrieveLeafColumns()
                    .map(function (col, index) {
                    // Fixed column takes no space in the scrollable area.
                        if (index < iFrozenColCount) {
                            return 0;
                        }
                        if (!col.getVisible()) {
                            return 0;
                        }

                        return self._CSSSizeToPixel(col.getWidth());
                    })
            );
        };

        /**
         * Computes the width, in pixels, used by the frozen columns.
         * @public
         * @returns {Number}
         */
        WideTable.prototype.getFixedAreaWidth = function() {
            var self = this;
            return this.getColumns()
                // Get fixed columns
                .slice(0, this.getTotalFrozenColumnCount())
                // Sum their widths
                .reduce(function(iSum, oCol) {
                    return iSum + self._CSSSizeToPixel(oCol.getWidth());
                }, 0);
        };

        /**
         * Computes the width, in pixels, used by the row selector.
         * @public
         * @returns {Number}
         */
        WideTable.prototype.getSelectorAreaWidth = function() {
            var bSelectorIsVisible = (this.getSelectionMode() !== SelectionMode.None && this.getSelectionBehavior() !== SelectionBehavior.RowOnly);
            if (bSelectorIsVisible) {
                var $selectorArea = this.$().find(".sapUiTableRowHdrScr");
                // Default width for the selector area should be 32.
                return $selectorArea ? $selectorArea.width() : 32;
            }
            return 0;
        };

        /**
         * Computes the width, in pixels, that the entire control has available.
         * @public
         * @returns {Number}
         */
        WideTable.prototype.getAvailableTableWidth = function() {
            var sWidth = this.getWidth();
            if (sWidth && /px$/.test(sWidth)) {
                return parseInt(sWidth, 10);
            }
            return this.$().width();
        };

        /**
         * Computes the width, in pixels, that is available for scrollable section.
         * @public
         * @returns {Number}
         */
        WideTable.prototype.getScrollableAreaWidth = function() {
            var iScrollableAreaWidth = this.getAvailableTableWidth() - this.getFixedAreaWidth() - this.getSelectorAreaWidth();
            if (this._isVSbRequired()) {
                iScrollableAreaWidth = iScrollableAreaWidth - TableUtils._determineVerticalScrollBarWidth();
            }
            return iScrollableAreaWidth;
        };

        /**
         * @private
         */
        WideTable.prototype._getHorizontalScrollPosition = function() {
            return this._iHorizontalScrollPosition;
        };

        /**
         * @private
         */
        WideTable.prototype._hasColumnRenderRangeChanged = function() {
            var bHasCachedResult = this._hasFnCacheResult("_getColumnRangeToRender");
            // If there is no cached result, we've probably never called it and we default to it has changed.
            if (!bHasCachedResult) {
                return true;
            }

            var oCurrentRange = this._getColumnRangeToRender(false);
            var oComputedRange = this._calculateColumnRangeToRender();

            return oCurrentRange.start !== oComputedRange.start
                    || oCurrentRange.end !== oComputedRange.end;
        };

        /**
         * @private
         */
        WideTable.prototype._hasFnCacheResult = function(sFnName) {
            return this._mFnResultsCache
                && this._mFnResultsCache[sFnName]
                && this._mFnResultsCache[sFnName].lastResult;
        };

        /**
         * @private
         */
        WideTable.prototype._getColumnRangeToRender = function(bForceRecalc) {

            var oFnCache = this._mFnResultsCache._getColumnRangeToRender;
            if (!bForceRecalc && oFnCache.lastResult) {
                // Return a copy of the cached result.
                return jQuery.extend({}, oFnCache.lastResult);
            }

            var oResult = this._calculateColumnRangeToRender();

            // Use jQuery.extend to create a clone for the cache. The
            // calling function can then manipulate the returned object
            // without affecting the cached result.
            oFnCache.lastResult = jQuery.extend({}, oResult);
            return oResult;
        };

        /**
         * Computes the column range to render given the current column configuration, scroll position and available width.
         * Can be expensive to compute at times. Please prefer to _getColumnRangeToRender over this function. Only a few scenarios
         * should ever prefer to use this function directly.
         *
         * @private
         */
        WideTable.prototype._calculateColumnRangeToRender = function() {
            var iScrollAreaWidth = this.getScrollableAreaWidth();

            var iViewStart = this._getHorizontalScrollPosition();
            var iViewEnd = iViewStart + iScrollAreaWidth;
            var iVSBWidth = TableUtils._determineVerticalScrollBarWidth();
            if (this._hasVSB()) {
                // Since the HSB is slighly wider than the actual scrollable content, when there is a VSB,
                // including VSB width helps make sure we can scroll to the end properly.
                iViewEnd += iVSBWidth;
            }

            var iStartColIndex;
            var iEndColIndex;

            var oColumnWidthsSumTable = this._getColumnWidthsSumTable();

            if (iViewEnd < oColumnWidthsSumTable.getTotalSum() || iViewStart === 0) {
                iStartColIndex = oColumnWidthsSumTable.getFirstIndexWithSumGreaterThanOrEqualTo(iViewStart);

                // Reset iViewStart to be the beginning of our first rendered column.
                // If this isn't done, there are edge cases where regular iViewEnd
                // calculation can result in rendering more columns than should be
                // causing visual rendering errors.
                iViewStart = oColumnWidthsSumTable.getSumTo(iStartColIndex-1);
                // Update iViewEnd to accomodate changes to iViewStart.
                iViewEnd = iViewStart + iScrollAreaWidth;

                iEndColIndex = oColumnWidthsSumTable.getFirstIndexWithSumGreaterThanOrEqualTo(iViewEnd);

                // If the end col index is beyond the number of columns we have, make it the last col.
                if (iEndColIndex >= oColumnWidthsSumTable.aPrefixes.length) {
                    iEndColIndex = oColumnWidthsSumTable.aPrefixes.length-1;
                }

                // Go to last column with a width. Hidden columns are given a width of 0.
                while (oColumnWidthsSumTable.aPrefixes[iEndColIndex] === 0 && iEndColIndex > 0) {
                    iEndColIndex--;
                }
            } else {
                // When trying to render the end of the table we need to work backwards instead of
                // forwards to determine which columns to render. This is to prevent the last column
                // from clipping in certain scenarios.
                iEndColIndex = this.retrieveLeafColumns().length-1; // end index should be last column

                // Go to last column with a width. Hidden columns are given a width of 0.
                while (oColumnWidthsSumTable.aPrefixes[iEndColIndex] === 0 && iEndColIndex > 0) {
                    iEndColIndex--;
                }

                // The logic can be viewed as going through a PrefixSumTable of the columns but ordered
                // in reverse and not allowing a column to clip.
                // TODO: This logic would appear to be best suited to reside in PrefixSumTable however,
                // I'm not sure how this can easily translate into an abstraction that fits. - niroth.
                var aWidths = oColumnWidthsSumTable.aPrefixes;
                var i = iEndColIndex;
                iStartColIndex = iEndColIndex;
                var iAvailableWidth = iScrollAreaWidth + iVSBWidth;
                for (; i >= 0; i--) {
                    if (aWidths[i] <= iAvailableWidth) {
                        iAvailableWidth -= aWidths[i];
                        iStartColIndex = i;
                    } else {
                        break;
                    }
                }
            }

            // Fix some issues where 0 width columns cause visual problems.
            while (oColumnWidthsSumTable.aPrefixes[iStartColIndex] === 0) {
                iStartColIndex++;
            }

            // Must use sum for the column before startIndex to include the starting column's width.
            var iViewWidth = oColumnWidthsSumTable.getSumTo(iEndColIndex) - oColumnWidthsSumTable.getSumTo(iStartColIndex-1);

            // Compute the number of visible columns
            var iVisibleColumns = 0;
            for (var i = iStartColIndex; i <= iEndColIndex ; i++) {
                if (oColumnWidthsSumTable.aPrefixes[i] > 0) {
                    iVisibleColumns++;
                }
            }

            var oResult = {
                start: iStartColIndex,
                end: iEndColIndex,
                width: iViewWidth,
                lastColIsClipped: (iViewEnd < iViewStart + iViewWidth),
                visibleColumnsCount: iVisibleColumns,
                contains: function(iIndex) {
                    return this.start <= iIndex && iIndex <= this.end;
                }
            };

            return oResult;
        };

        /**
         * @private
         */
        WideTable.prototype._createRowTemplate = function(sId) {
            //default value is based on Table's ID
            sId = sId || (this.getId() + "-rows");

            var oTemplate = new WideRow(sId);
            //assign a template selection control (if desired)
            oTemplate.setSelectionControl(this._createSelectionControlTemplate(sId));
            oTemplate.setCacheDomRefs(false);

            return oTemplate;
        };

        /**
         * @override
         */
        WideTable.prototype.onfocusin = function(oEvent) {
            if (!this._getKeyboardExtension()._keyboardActionInProcess) {
                SasParentClass.prototype.onfocusin.apply(this, arguments);
            }
            this.$().addClass("sasContainsFocusedItem");
        };

        /**
         * @private
         * @override
         */
        WideTable.prototype._attachMouseMoveResizerEvents = function() {
            this.$().find(".sapUiTableCtrlScr, .sapUiTableCtrlScrFixed, .sasUiTableColHdrCntShell").mousemove(jQuery.proxy(this._onScrPointerMove, this));
        };

        /**
         * @private
         * @override
         */
        WideTable.prototype._detachMouseMoveResizerEvents = function() {
            this.$().find(".sapUiTableCtrlScr, .sapUiTableCtrlScrFixed, .sasUiTableColHdrCntShell").unbind();
        };

        /**
         * @private
         * @override
         */
        WideTable.prototype._onScrPointerMove = function(oEvent) {
            // Find the table headers again in case a scroll happened.
            // TODO Decide if this is better suited to be in the onhscroll
            this._aTableHeaders = this.$().find(".sapUiTableCtrlFirstCol > th:not(.sapUiTableColSel)");

            SasParentClass.prototype._onScrPointerMove.apply(this, arguments);
        };

        /**
         * @private
         * @override
         */
        WideTable.prototype._collectTableSizes = function(aTableRowHeights) {
            var oTableSizes = SasParentClass.prototype._collectTableSizes.apply(this, arguments);
            oTableSizes.tableCtrlScrollWidth = this._getColumnsWidth(this.getTotalFrozenColumnCount());
            // oTableSizes.tableCtrlScrWidth = oTableSizes.tableCtrlScrollWidth;

            return oTableSizes;
        };

        /**
         * @private
         */
        WideTable.prototype._getWidthSumForScrollableSection = function() {
            var oRenderRange = this._getColumnRangeToRender();
            var iNewWidth = this._getColumnsWidth(oRenderRange.start, oRenderRange.end+1);
            return iNewWidth;
        };

        /**
         * @private
         * @override
         */
        WideTable.prototype._includeColumnInRowTemplate = function(oColumn) {
            // Include all columns in the row template for WideTables
            return true;
        };

        /**
         * @private
         * @override
         */
        WideTable.prototype._shouldHaveHSB = function() {
            return this._getColumnsWidth(this.getTotalFrozenColumnCount()) > this.getScrollableAreaWidth();
        };

        /**
         * @private
         * @override
         */
        WideTable.prototype._determineCellDownAndRightTillPage = function() {
            var oLastFullyVisibleRow = this._determineLastFullyVisibleRow();
            var aCells = oLastFullyVisibleRow.getCells();
            var oColumnWidthsSumTable = this._getColumnWidthsSumTable();
            var oTargetCell;
            for (var i = aCells.length-1; i >= 0; i--) {
                if (aCells[i].getVisible() && oColumnWidthsSumTable.getPrefixAt(i) > 0) {
                    oTargetCell = aCells[i];
                    break;
                }
            }
            if (oTargetCell) {
                return oTargetCell.$().closest('td.sapUiTableTd').get(0);
            }
            return undefined;
        };

        /**
         * Copied and slightly modified from the Table version.
         * @private
         * @override
         */
        WideTable.prototype._oncntscroll = function() {
            if (this._bRtlMode && jQuery.sap.touchEventMode === "ON") {
                return;
            }

            if (!this._bSyncScrollLeft) {
                var $hsb = this._getHsbRef();
                if ($hsb) {
                    var oColHdrScr = this.getDomRef().querySelector(".sapUiTableCtrlScr");
                    // In WideTable, the table body doesn't have the full width like the HSB does.
                    // Thus instead, we have to take the body's scroll as an offset of the HSB's
                    // current position.

                    var iNewScrollLeft = oColHdrScr.scrollLeft + this._getHorizontalScrollPosition();
                    $hsb.scrollLeft(iNewScrollLeft);

                    oColHdrScr.scrollLeft = 0;
                }
            }
        };

        /**
         * @private
         */
        WideTable.prototype._calculateHeaderWidth = function(oHeaderElement, oColumn, iIndex) {
            // Use actual count instead of from regular getter because the getter is affected by
            // the _bIgnoreFixedColumnCount property.
            var iActualFrozenColumnCount = this._getActualFrozenColumnCount();
            var iHeaderWidth = iIndex < iActualFrozenColumnCount
                ? this._CSSSizeToPixel(oColumn.getWidth())
                : SasParentClass.prototype._calculateHeaderWidth.apply(this, arguments);

            return iHeaderWidth;
        };

        // Make PrefixSumTable available for unit testing
        // TODO consider refactoring PrefixSumTable to its own file.
        WideTable.__PrefixSumTable = PrefixSumTable;

        return WideTable;
    }

    var SasHcUiTableWideTable = createWideVariant(SasHcUiTable);
    createWideVariant(SasHcUiTreeTable);

    return SasHcUiTableWideTable;
}, /* bExport= */ true);
