/*!
 * (c) 2014-2018, SAS Institute Inc.
 */

// Provides helper sas.hc.ui.table.TableUtils.
sap.ui.define(['jquery.sap.global', 'sap/ui/core/Control', 'sap/ui/core/ResizeHandler', './TableGrouping', './library'],
    function(jQuery, Control, ResizeHandler, TableGrouping, library) {
        "use strict";

        // shortcuts
        var SelectionBehavior = library.SelectionBehavior,
            SelectionMode = library.SelectionMode;

        /**
         * Static collection of utility functions related to the sas.hc.ui.table.Table, ...
         *
         * @author SAP SE
         * @version 904001.11.16.20251118090100_f0htmcm94p
         * @namespace
         * @name sas.hc.ui.table.TableUtils
         * @private
         */
        var TableUtils = {

            Grouping: TableGrouping, //Make grouping utils available here

            /*
             * Known basic cell types in the table
             */
            CELLTYPES : {
                DATACELL : "DATACELL", // standard data cell (standard, group or sum)
                COLUMNHEADER : "COLUMNHEADER", // column header
                ROWHEADER : "ROWHEADER", // row header (standard, group or sum)
                COLUMNROWHEADER : "COLUMNROWHEADER" // select all row selector (top left cell)
            },

            /**
             * Returns whether the table has a row header or not
             * @param {sas.hc.ui.table.Table} oTable Instance of the table
             * @return {boolean}
             * @private
             */
            hasRowHeader : function(oTable) {
                return (oTable.getSelectionMode() !== SelectionMode.None
                    && oTable.getSelectionBehavior() !== SelectionBehavior.RowOnly)
                    || TableGrouping.isGroupMode(oTable);
            },

            /**
             * Returns whether selection is allowed on the cells of a row (not row selector).
             * @param {sas.hc.ui.table.Table} oTable Instance of the table
             * @return {boolean}
             * @private
             */
            isRowSelectionAllowed : function(oTable) {
                return oTable.getSelectionMode() !== SelectionMode.None &&
                (oTable.getSelectionBehavior() === SelectionBehavior.Row || oTable.getSelectionBehavior() === SelectionBehavior.RowOnly);
            },

            /**
             * Returns whether selection is allowed via the row selector.
             * @param {sas.hc.ui.table.Table} oTable Instance of the table
             * @return {boolean}
             * @private
             */
            isRowSelectorSelectionAllowed : function(oTable) {
            // Incl. that RowOnly works like Row
                return oTable.getSelectionMode() !== SelectionMode.None && TableUtils.hasRowHeader(oTable);
            },

            /**
             * Returns whether the no data text is currently shown or not
             * If true, also CSS class sapUiTableEmpty is set on the table root element.
             * @param {sas.hc.ui.table.Table} oTable Instance of the table
             * @return {boolean}
             * @private
             */
            isNoDataVisible : function(oTable) {
                if (!oTable.getShowNoData()) {
                    return false;
                }

                var oBinding = oTable.getBinding("rows"),
                    iBindingLength = oTable._getRowCount(),
                    bHasData = oBinding ? !!iBindingLength : false;

                return !bHasData || (TableUtils.getVisibleColumnCount(oTable) === 0);
            },

            /**
             * Checks whether the given object is of the given type (given in AMD module syntax)
             * without the need of loading the types module.
             * @param {sap.ui.base.ManagedObject} oObject The object to check
             * @param {string} sType The type given in AMD module syntax
             * @return {boolean}
             * @private
             */
            isInstanceOf : function(oObject, sType) {
                if (!oObject || !sType) {
                    return false;
                }
                var oType = sap.ui.require(sType);
                return !!(oType && (oObject instanceof oType));
            },

            /**
             * Toggles the expand / collapse state of the group which contains the given Dom element.
             * @param {sas.hc.ui.table.Table} oTable Instance of the table
             * @param {Object} oRef DOM reference of an element within the table group header
             * @param {boolean} [bExpand] If defined instead of toggling the desired state is set.
             * @return {boolean} <code>true</code> when the operation was performed, <code>false</code> otherwise.
             * @private
             */
            toggleGroupHeader : function(oTable, oRef, bExpand) {
                var $Ref = jQuery(oRef),
                    $GroupRef;

                if ($Ref.hasClass("sapUiTableTreeIcon") || $Ref.hasClass("sasUiTreeNodeBullet")) {
                    $GroupRef = $Ref.closest("tr");
                } else {
                    $GroupRef = $Ref.closest(".sapUiTableGroupHeader");
                }

                var oBinding = oTable.getBinding("rows");
                if ($GroupRef.length > 0 && oBinding) {
                    var iFirstVisibleRow = oTable.getFirstVisibleRow();
                    var iVisibleRowIndex = parseInt($GroupRef.attr("data-sap-ui-rowindex"), 10);
                    var iRowIndex = iFirstVisibleRow + iVisibleRowIndex;
                    var bIsExpanded = TableGrouping.toggleGroupHeader(oTable, iRowIndex, bExpand);
                    var bChanged = bIsExpanded === true || bIsExpanded === false;
                    if (bChanged && oTable._onGroupHeaderChanged) {
                        oTable._onGroupHeaderChanged(iRowIndex, bIsExpanded);
                        // S1295909 - when collapsing a node, ensure that the node retains focus
                        if (bIsExpanded === false) {
                            // setTimeout is necessary to ensure the firstVisibleRow is updated
                            // expanding the tree node (at least via keyboard) does not go through
                            // setFirstVisibleNode
                            setTimeout(function() {
                                var iNewFirstVisibleRow = oTable.getFirstVisibleRow();
                                var iNewVisibleRowIndex = iVisibleRowIndex + (iFirstVisibleRow - iNewFirstVisibleRow);
                                var aRows = oTable.getRows();
                                if (iNewVisibleRowIndex < 0) {
                                    iNewVisibleRowIndex = 0;
                                } else if (iNewVisibleRowIndex >= aRows.length) {
                                    iNewVisibleRowIndex = aRows.length -1;
                                }
                                var iFocusIndex = oTable._getItemNavigation().getIndexFromDomRef(aRows[iNewVisibleRowIndex].getCells()[0].$());
                                oTable._getItemNavigation().setFocusedIndex(iFocusIndex);
                                var $treeIcon;
                                if (oTable.getFixedColumnCount() > 0) {
                                    $treeIcon = aRows[iNewVisibleRowIndex].getDomRefs(true).rowFixedPart.find(".sapUiTableTreeIcon");
                                } else {
                                    $treeIcon = aRows[iNewVisibleRowIndex].getDomRefs(true).rowScrollPart.find(".sapUiTableTreeIcon");
                                }
                                if ($treeIcon) {
                                    $treeIcon.attr("tabindex", 0).focus();
                                }
                            }, 0);
                        }
                    }
                    return bChanged;
                }
                return false;
            },

            /**
             * Returns the text to be displayed as no data message.
             * If a custom noData control is set null is returned.
             * @param {sas.hc.ui.table.Table} oTable Instance of the table
             * @return {String|string|null}
             * @private
             */
            getNoDataText : function(oTable) {
                var oNoData = oTable.getNoData();
                if (oNoData instanceof Control) {
                    return null;
                } else {
                    if (typeof oNoData === "string" || oTable.getNoData() instanceof String) {
                        return oNoData;
                    } else {
                        return oTable._oResBundle.getText("TBL_NO_DATA");
                    }
                }
            },

            /**
             * Returns the number of currently visible columns
             * @param {sas.hc.ui.table.Table} oTable Instance of the table
             * @return {int}
             * @private
             */
            getVisibleColumnCount : function(oTable) {
                return oTable._getVisibleColumns().length;
            },

            /**
             * Returns the number of header rows
             * @param {sas.hc.ui.table.Table} oTable Instance of the table
             * @return {int}
             * @private
             */
            getHeaderRowCount : function(oTable) {
                if (!oTable.getColumnHeaderVisible()) {
                    return 0;
                }
                if (oTable._aRenderArray) {
                    return oTable._aRenderArray.length;
                }
                var iHeaderRows = 0;
                // if we have column groups, do not limit the check to visible columns
                // because we do not currently support hidden columns in column groups
                // TODO - in the future if we support hidden columns in column groups,
                // we will need to make this function only account for visible columns
                oTable.getColumns().forEach(function(oColumnGroup, iIndex) {
                    iHeaderRows = Math.max(iHeaderRows,  oColumnGroup.calculateRowSpan());
                });
                // if we don't have column groups, then we only have one header row
                return iHeaderRows > 0 ? iHeaderRows : 1;
            },

            /**
             * Returns the height of the defined row, identified by its row index.
             * @param {Object} oTable current table object
             * @param {int} iRowIndex the index of the row which height is needed
             * @private
             */
            getRowHeightByIndex : function(oTable, iRowIndex) {
                var iRowHeight = 0;

                if (oTable) {
                    var aRows = oTable.getRows();
                    if (aRows && aRows.length && iRowIndex > -1 && iRowIndex < aRows.length) {
                        var oDomRefs = aRows[iRowIndex].getDomRefs();
                        if (oDomRefs) {
                            if (oDomRefs.rowScrollPart && oDomRefs.rowFixedPart) {
                                iRowHeight = Math.max(oDomRefs.rowScrollPart.clientHeight, oDomRefs.rowFixedPart.clientHeight);
                            } else if (oDomRefs.rowScrollPart) {
                                iRowHeight = oDomRefs.rowScrollPart.clientHeight;
                            }
                        }
                    }
                }

                return iRowHeight;
            },

            /**
             * Checks whether all conditions for pixel-based scrolling (Variable Row Height) are fulfilled.
             * @param {Object} oTable current table object
             * @returns {Boolean} true/false if fulfilled
             * @private
             */
            isVariableRowHeightEnabled : function(oTable) {
                return oTable.getEnablePixelScrolling() === true
                && oTable.getFixedRowCount() <= 0
                && oTable.getFixedBottomRowCount() <= 0;
            },

            /**
             * Returns the logical number of rows
             * Optionally empty visible rows are added (in case that the number of data
             * rows is smaller than the number of visible rows)
             * @param {sas.hc.ui.table.Table} oTable Instance of the table
             * @param {boolean} bIncludeEmptyRows
             * @return {int}
             * @private
             */
            getTotalRowCount : function(oTable, bIncludeEmptyRows) {
                var iRowCount = oTable._getRowCount();
                if (bIncludeEmptyRows) {
                    iRowCount = Math.max(iRowCount, oTable.getVisibleRowCount());
                }
                return iRowCount;
            },

            /**
             * Returns the number of visible rows that are not empty.
             * If the number of visible rows is smaller than the number of data rows,
             * the number of visible rows is returned, otherwise the number of data rows.
             * @param {sas.hc.ui.table.Table} oTable Instance of the table
             * @returns {int}
             * @private
             */
            getNonEmptyVisibleRowCount : function(oTable) {
                return Math.min(oTable.getVisibleRowCount(), oTable._getRowCount());
            },

            /**
             * Returns a combined info about the currently focused item (based on the item navigation)
             * @param {sas.hc.ui.table.Table} oTable Instance of the table
             * @return {Object|null}
             * @type {Object}
             * @property {int} cell Index of focused cell in ItemNavigation
             * @property {int} columnCount Number of columns in ItemNavigation
             * @property {int} cellInRow Index of the cell in row
             * @property {int} row Index of row in ItemNavigation
             * @property {int} cellCount Number of cells in ItemNavigation
             * @property {Object|undefined} domRef Focused DOM reference of undefined
             * @private
             */
            getFocusedItemInfo : function(oTable) {
                var oIN = oTable._getItemNavigation();
                if (!oIN) {
                    return null;
                }
                return {
                    cell: oIN.getFocusedIndex(),
                    columnCount: oIN.iColumns,
                    cellInRow: oIN.getFocusedIndex() % oIN.iColumns,
                    row: Math.floor(oIN.getFocusedIndex() / oIN.iColumns),
                    cellCount: oIN.getItemDomRefs().length,
                    domRef: oIN.getFocusedDomRef()
                };
            },

            /**
             * Returns an object describing the context of the cell in focus including the Column object and logical row index.
             * @param {sas.hc.ui.table.Table} oTable Instance of the table
             * @param {Object} oFocusInfo an object returned by getFocusedItemInfo
             * @return {Object|null}
             * @type {Object}
             * @property {int} rowIndex The logical row index
             * @property {int} colIndex The visible column index
             * @property {Column|undefined} column The visible Column or undefined
             * @property {Control|undefined} cell Cell Control or Column Label
             */
            getTablePositionFromFocusInfo: function(oTable, oFocusInfo) {
                var iRowIndex, iColIndex, oRow, oColumn, oCell;

                if (oTable && oFocusInfo) {
                    //find column
                    iColIndex = oFocusInfo.cellInRow - (TableUtils.hasRowHeader(oTable) ? 1 : 0);
                    oColumn = oTable._getVisibleColumns()[iColIndex];

                    iRowIndex = oFocusInfo.row - TableUtils.getHeaderRowCount(oTable);
                    if (oTable.shouldShowSummaryHeader()) {
                        //summary header is 2 rows above header, but not counted as header rows
                        iRowIndex -= 2;
                    }

                    if (iRowIndex >= 0) {
                        //focused row is part of the data cells
                        oRow = oTable.getRows()[iRowIndex];
                        iRowIndex = oRow.getIndex();
                        //find cell for column in row
                        oCell = oRow.getCells()[iColIndex];
                    } else if (oColumn) {
                        // return the label as the cell
                        oCell = oColumn.getLabel();
                    }

                    return {
                        rowIndex: iRowIndex,
                        colIndex: oColumn ? oColumn.getIndex() : -1,
                        column: oColumn,
                        cell: oCell
                    };
                }

                // default response is null to indicate no usable information can be provided
                return null;
            },

            /**
             * Returns the index of the column (in the array of visible columns (see Table._getVisibleColumns())) of the current focused cell
             * @param {sas.hc.ui.table.Table} oTable Instance of the table
             * @return {int}
             * @private
             */
            getColumnIndexOfFocusedCell : function(oTable) {
                var oInfo = TableUtils.getFocusedItemInfo(oTable);
                return oInfo.cellInRow - (TableUtils.hasRowHeader(oTable) ? 1 : 0);
            },

            /**
             * Returns the index of the row (in the rows aggregation) of the current focused cell
             * @param {sas.hc.ui.table.Table} oTable Instance of the table
             * @return {int}
             * @private
             *
             */
            getRowIndexOfFocusedCell : function(oTable) {
                var oInfo = TableUtils.getFocusedItemInfo(oTable);
                return oInfo.row - TableUtils.getHeaderRowCount(oTable);
            },

            /**
             * Returns whether the given cell is located in a group header.
             * @param {Object} oCellRef DOM reference of table cell
             * @return {boolean}
             * @private
             */
            isInGroupingRow : function(oCellRef) {
                var oInfo = TableUtils.getCellInfo(oCellRef);
                if (oInfo && oInfo.type === TableUtils.CELLTYPES.DATACELL) {
                    return oInfo.cell.parent().hasClass("sapUiTableGroupHeader");
                } else if (oInfo && oInfo.type === TableUtils.CELLTYPES.ROWHEADER) {
                    return oInfo.cell.hasClass("sapUiTableGroupHeader");
                }
                return false;
            },

            /**
             * Returns whether the given cell is located in a analytical summary row.
             * @param {Object} oCellRef DOM reference of table cell
             * @return {boolean}
             * @private
             */
            isInSumRow : function(oCellRef) {
                var oInfo = TableUtils.getCellInfo(oCellRef);
                if (oInfo && oInfo.type === TableUtils.CELLTYPES.DATACELL) {
                    return oInfo.cell.parent().hasClass("sapUiAnalyticalTableSum");
                } else if (oInfo && oInfo.type === TableUtils.CELLTYPES.ROWHEADER) {
                    return oInfo.cell.hasClass("sapUiAnalyticalTableSum");
                }
                return false;
            },

            /**
             * Returns whether column with the given index (in the array of visible columns (see Table._getVisibleColumns()))
             * is a fixed column.
             * @param {sas.hc.ui.table.Table} oTable Instance of the table
             * @param {int} iColIdx Index of column in the tables column aggregation
             * @return {boolean}
             * @private
             */
            isFixedColumn : function(oTable, iColIdx) {
                return iColIdx < oTable.getTotalFrozenColumnCount();
            },

            /**
             * Returns whether the table has fixed columns.
             * @param {sas.hc.ui.table.Table} oTable Instance of the table
             * @return {boolean}
             * @private
             */
            hasFixedColumns : function(oTable) {
                return oTable.getTotalFrozenColumnCount() > 0;
            },

            /**
             * Focus the item with the given index in the item navigation
             * @param {sas.hc.ui.table.Table} oTable Instance of the table
             * @param {int} iIndex Index of item in ItemNavigation which shall get the focus
             * @param {Object} oEvent
             * @private
             */
            focusItem : function(oTable, iIndex, oEvent) {
                if (isNaN(iIndex)) {
                    return;
                }

                var oIN = oTable._getItemNavigation();
                if (oIN) {
                    oIN.focusItem(iIndex, oEvent);
                }
            },

            /**
             * Returns the cell type and the jQuery wrapper object of the given cell dom ref or
             * null if the given dom element is not a table cell.
             * {type: <TYPE>, cell: <$CELL>}
             * @param {Object} oCellRef DOM reference of table cell
             * @return {Object}
             * @type {Object}
             * @property {sas.hc.ui.table.TableUtils.CELLTYPES} type
             * @property {Object} cell jQuery object of the cell
             * @see TableUtils.CELLTYPES
             * @private
             */
            getCellInfo : function(oCellRef) {
                if (!oCellRef) {
                    return null;
                }
                var $Cell = jQuery(oCellRef);
                if ($Cell.hasClass("sapUiTableTd")) {
                    return {type: TableUtils.CELLTYPES.DATACELL, cell: $Cell};
                } else if ($Cell.hasClass("sapUiTableCol")) {
                    return {type: TableUtils.CELLTYPES.COLUMNHEADER, cell: $Cell};
                } else if ($Cell.hasClass("sapUiTableRowHdr") || /-selectionControl-/.test($Cell.attr('data-sap-ui')) === true) {
                    return {type: TableUtils.CELLTYPES.ROWHEADER, cell: $Cell};
                } else if ($Cell.hasClass("sapUiTableColRowHdr")) {
                    return {type: TableUtils.CELLTYPES.COLUMNROWHEADER, cell: $Cell};
                }
                return null;
            },

            /**
             * Returns the Row, Column and Cell instances for the given row index (in the rows aggregation)
             * and column index (in the array of visible columns (see Table._getVisibleColumns()).
             * @param {sas.hc.ui.table.Table} oTable Instance of the table
             * @param {int} iRowIdx Index of row in the tables rows aggregation
             * @param {int} iColIdx Index of column in the tables columns aggregation
             * @return {Object}
             * @type {Object}
             * @property {sas.hc.ui.table.Row} row Row of the table
             * @property {sas.hc.ui.table.Column} column Column of the table
             * @property {sap.ui.core.Control} cell Cell control of row/column
             * @private
             */
            getRowColCell : function(oTable, iRowIdx, iColIdx) {
                var oRow = oTable.getRows()[iRowIdx];
                var oColumn = oTable._getVisibleColumns()[iColIdx];
                var oCell = oRow && oRow.getCells()[iColIdx];

                //TBD: Clarify why this is needed!
                if (oCell && oCell.data("sap-ui-colid") !== oColumn.getId()) {
                    var aCells = oRow.getCells();
                    for (var i = 0; i < aCells.length; i++) {
                        if (aCells[i].data("sap-ui-colid") === oColumn.getId()) {
                            oCell = aCells[i];
                            break;
                        }
                    }
                }

                return {row: oRow, column: oColumn, cell: oCell};
            },

            /**
             * Returns the table cell which is either the parent of the specified element, or returns the specified element itself if it is a table cell.
             *
             * @param {sap.ui.table.Table} oTable Instance of the table used as the context within which to search for the parent.
             * @param {jQuery|HTMLElement} oElement An element inside a table cell. Can be a jQuery object or a DOM Element.
             * @returns {jQuery|null} Returns null if the passed element is not inside a table cell or a table cell itself.
             * @private
             */
            getCell: function(oTable, oElement) {
                if (oTable === null || oElement === null) {
                    return null;
                }

                var $Element = jQuery(oElement);
                var $Cell;
                var oTableElement = oTable.getDomRef();

                $Cell = $Element.closest(".sapUiTableTd", oTableElement);
                if ($Cell.length > 0) {
                    return $Cell;
                }

                $Cell = $Element.closest(".sapUiTableCol", oTableElement);
                if ($Cell.length > 0) {
                    return $Cell;
                }

                $Cell = $Element.closest(".sapUiTableRowHdr", oTableElement);
                if ($Cell.length > 0) {
                    return $Cell;
                }

                $Cell = $Element.closest(".sapUiTableRowAction", oTableElement);
                if ($Cell.length > 0) {
                    return $Cell;
                }

                $Cell = $Element.closest(".sapUiTableColRowHdr", oTableElement);
                if ($Cell.length > 0) {
                    return $Cell;
                }

                return null;
            },

            /**
             * Registers a ResizeHandler for a DOM reference identified by its ID suffix. The ResizeHandler ID is tracked
             * in _mResizeHandlerIds of the table instance. The sIdSuffix is used as key.
             * Existing ResizeHandlers will be de-registered before the new one is registered.
             *
             * @param {sas.hc.ui.table.Table} oTable Instance of the table
             * @param {string} sIdSuffix ID suffix to identify the DOM element for which to register the ResizeHandler
             * @param {Function} fnHandler Function to handle the resize event
             * @param {boolean}[bRegisterParent] Flag to register the ResizeHandler for the parent DOM element of the one identified by sIdSuffix
             *
             * @return {int|undefined} ResizeHandler ID or undefined if the DOM element could not be found
             * @private
             */
            registerResizeHandler : function(oTable, sIdSuffix, fnHandler, bRegisterParent) {
                var oDomRef;
                if (typeof sIdSuffix == "string") {
                    oDomRef = oTable.getDomRef(sIdSuffix);
                } else {
                    jQuery.sap.log.error("sIdSuffix must be a string", oTable);
                    return;
                }

                if (typeof fnHandler !== "function") {
                    jQuery.sap.log.error("fnHandler must be a function", oTable);
                    return;
                }

                // make sure that each DOM element of the table can only have one resize handler in order to avoid memory leaks
                this.deregisterResizeHandler(oTable, sIdSuffix);

                if (!oTable._mResizeHandlerIds) {
                    oTable._mResizeHandlerIds = {};
                }

                if (bRegisterParent && oDomRef) {
                    oDomRef = oDomRef.parentNode;
                }

                if (oDomRef) {
                    oTable._mResizeHandlerIds[sIdSuffix] = ResizeHandler.register(oDomRef, fnHandler);
                }

                return oTable._mResizeHandlerIds[sIdSuffix];
            },

            /**
             * De-register ResizeHandler identified by sIdSuffix. If sIdSuffix is undefined, all know ResizeHandlers will be de-registered
             * @param {sas.hc.ui.table.Table} oTable Instance of the table
             * @param {string|Array.<string>} [vIdSuffix] ID suffix to identify the ResizeHandler to de-register. If undefined, all will be de-registered
             * @private
             */
            deregisterResizeHandler : function(oTable, vIdSuffix) {
                var aIdSuffix;
                if (!oTable._mResizeHandlerIds) {
                    // no resize handler registered so far
                    return;
                }

                if (typeof vIdSuffix == "string") {
                    aIdSuffix = [vIdSuffix];
                } else if (vIdSuffix === undefined) {
                    aIdSuffix = [];
                    // de-register all resize handlers if no specific is named
                    for (var sKey in oTable._mResizeHandlerIds) {
                        if (typeof sKey == "string" && oTable._mResizeHandlerIds.hasOwnProperty(sKey)) {
                            aIdSuffix.push(sKey);
                        }
                    }
                } else if (jQuery.isArray(vIdSuffix)) {
                    aIdSuffix = vIdSuffix;
                }

                for (var i = 0; i < aIdSuffix.length; i++) {
                    var sIdSuffix = aIdSuffix[i];
                    if (oTable._mResizeHandlerIds[sIdSuffix]) {
                        ResizeHandler.deregister(oTable._mResizeHandlerIds[sIdSuffix]);
                        oTable._mResizeHandlerIds[sIdSuffix] = undefined;
                    }
                }
            },

            /**
             * Scrolls the data in the table forward or backward by manipulating the property <code>firstVisibleRow</code>.
             * @param {sas.hc.ui.table.Table} oTable Instance of the table
             * @param {boolean} bDown Whether to scroll down or up
             * @param {boolean} bPage Whether scrolling should be page wise or a single step (only possibe with navigation mode <code>Scrollbar</code>)
             * @private
             */
            scroll : function(oTable, bDown, bPage) {
                var bScrolled = false;
                var iRowCount = oTable._getRowCount();
                var iVisibleRowCount = oTable.getVisibleRowCount();
                var iScrollableRowCount = iVisibleRowCount - oTable.getFixedRowCount() - oTable.getFixedBottomRowCount();
                var iFirstVisibleScrollableRow = oTable._getSanitizedFirstVisibleRow();
                var iSize = bPage ? iScrollableRowCount : 1;

                if (bDown) {
                    if (iFirstVisibleScrollableRow + iVisibleRowCount < iRowCount) {
                        oTable.setFirstVisibleRow(Math.min(iFirstVisibleScrollableRow + iSize, iRowCount - iVisibleRowCount));
                        bScrolled = true;
                    }
                } else {
                    if (iFirstVisibleScrollableRow > 0) {
                        oTable.setFirstVisibleRow(Math.max(iFirstVisibleScrollableRow - iSize, 0));
                        bScrolled = true;
                    }
                }

                return bScrolled;
            },

            /**
             * Scrolls the data in the table to the end or to the beginning by manipulating the property <code>firstVisibleRow</code>.
             * @param {sas.hc.ui.table.Table} oTable Instance of the table
             * @param {boolean} bDown Whether to scroll down or up
             * @returns {boolean} True if scrolling was actually performed
             * @private
             */
            scrollMax : function(oTable, bDown) {
                var bScrolled = false;
                var iFirstVisibleScrollableRow = oTable._getSanitizedFirstVisibleRow();

                if (bDown) {
                    var iFirstVisibleRow = oTable._getRowCount() - this.getNonEmptyVisibleRowCount(oTable);
                    if (iFirstVisibleScrollableRow < iFirstVisibleRow) {
                        oTable.setFirstVisibleRow(iFirstVisibleRow);
                        bScrolled = true;
                    }
                } else {
                    if (iFirstVisibleScrollableRow > 0) {
                        oTable.setFirstVisibleRow(0);
                        bScrolled = true;
                    }
                }

                return bScrolled;
            },

            /**
             * Checks whether the cell of the given DOM reference is in the first row (from DOM point of view) of the scrollable area.
             * @param {sas.hc.ui.table.Table} oTable Instance of the table
             * @param {Object} oRef Cell DOM Reference
             * @private
             */
            isFirstScrollableRow : function(oTable, oRef) {

                if (TableUtils.isSASTable(oTable) === true) {
                    var oRow = oTable._rowFromRef(oRef),
                        oFirstVisibleRow = oTable._determineFirstFullyVisibleRow();
                    if (!oRow) {
                        return false;
                    }
                    return oFirstVisibleRow.getIndex() >= oRow.getIndex();
                }

                var $Ref = jQuery(oRef);
                var iRowIndex = parseInt($Ref.add($Ref.parent()).filter("[data-sap-ui-rowindex]").attr("data-sap-ui-rowindex"), 10);
                var iFixed = oTable.getFixedRowCount() || 0;
                return iRowIndex === iFixed;
            },

            /**
             * Checks whether the cell of the given DOM reference is in the last row (from DOM point of view) of the scrollable area.
             * @param {sas.hc.ui.table.Table} oTable Instance of the table
             * @param {Object} oRef Cell DOM Reference
             * @private
             */
            isLastScrollableRow : function(oTable, oRef) {

                if (TableUtils.isSASTable(oTable) === true) {
                    var oRow = oTable._rowFromRef(oRef),
                        oLastVisibleRow = oTable._determineLastFullyVisibleRow();
                    if (!oRow) {
                        return false;
                    }
                    return oLastVisibleRow.getIndex() <= oRow.getIndex();
                }

                var $Ref = jQuery(oRef);
                var iRowIndex = parseInt($Ref.add($Ref.parent()).filter("[data-sap-ui-rowindex]").attr("data-sap-ui-rowindex"), 10);
                var iFixed = oTable.getFixedBottomRowCount() || 0;
                return iRowIndex === oTable.getVisibleRowCount() - iFixed - 1;
            },

            /**
             * Returns the content density style class which is relevant for the given control. First it tries to find the
             * definition via the control API. While traversing the controls parents, it's tried to find the closest DOM
             * reference. If that is found, the check will use the DOM reference to find the closest content density style class
             * in the parent chain. This approach caters both use cases: content density defined at DOM and/or control level.
             *
             * If at the same level, several style classes are defined, this is the priority:
             * sapUiSizeCompact, sapUiSizeCondensed, sapUiSizeCozy
             *
             * @param {sas.hc.ui.table.Table} oControl Instance of the table
             * @returns {String|undefined} name of the content density stlye class or undefined if none was found
             * @private
             */
            getContentDensity : function(oControl) {
                var sContentDensity;
                var aContentDensityStyleClasses = ["sapUiSizeCompact", "sapUiSizeCondensed", "sapUiSizeCozy"];

                var fnGetContentDensity = function (sFnName, oObject) {
                    if (!oObject[sFnName]) {
                        return;
                    }

                    for (var i = 0; i < aContentDensityStyleClasses.length; i++) {
                        if (oObject[sFnName](aContentDensityStyleClasses[i])) {
                            return aContentDensityStyleClasses[i];
                        }
                    }
                };

                var $DomRef = oControl.$();
                if ($DomRef.length > 0) {
                    // table was already rendered, check by DOM and return content density class
                    sContentDensity = fnGetContentDensity("hasClass", $DomRef);
                } else {
                    sContentDensity = fnGetContentDensity("hasStyleClass", oControl);
                }

                if (sContentDensity) {
                    return sContentDensity;
                }

                // since the table was not yet rendered, traverse its parents:
                //   - to find a content density defined at control level
                //   - to find the first DOM reference and then check on DOM level
                var oParentDomRef = null;
                var oParent = oControl.getParent();
                // the table might not have a parent at all.
                if (oParent) {
                    // try to get the DOM Ref of the parent. It might be required to traverse the complete parent
                    // chain to find one parent which has DOM rendered, as it may happen that an element does not have
                    // a corresponding DOM Ref
                    do {
                        // if the content density is defined at control level, we can return it, no matter the control was already
                        // rendered. By the time it will be rendered, it will have that style class
                        sContentDensity = fnGetContentDensity("hasStyleClass", oParent);
                        if (sContentDensity) {
                            return sContentDensity;
                        }

                        // if there was no style class set at control level, we try to find the DOM reference. Using that
                        // DOM reference, we can easily check for the content density style class via the DOM. This allows us
                        // to include e.g. the body tag as well.
                        if (oParent.getDomRef) {
                            // for Controls and elements
                            oParentDomRef = oParent.getDomRef();
                        } else if (oParent.getRootNode) {
                            // for UIArea
                            oParentDomRef = oParent.getRootNode();
                        }

                        if (!oParentDomRef && oParent.getParent) {
                            oParent = oParent.getParent();
                        } else {
                            // make sure there is not endless loop if oParent has no getParent function
                            oParent = null;
                        }
                    } while (oParent && !oParentDomRef);
                }

                // if we found a DOM reference, check for content density
                $DomRef = jQuery(oParentDomRef || document.body);
                sContentDensity = fnGetContentDensity("hasClass", $DomRef.closest("." + aContentDensityStyleClasses.join(",.")));

                return sContentDensity;
            },


            isSASTable : function(oTable) {
                return oTable.getMetadata().getName().indexOf("sas\.") === 0;
            },

            /**
             * Determine whether the passed-in Table is a WideTable
             * @param {object} Table instance
             * @return {boolean} wide or not
             * @private
             */
            isWideTable : function(oTable) {
                return jQuery.isFunction(oTable._isWideTableVariant) && oTable._isWideTableVariant();
            },

            /**
             * Determine the height of a horizontal scrollbar since it can vary between browsers
             * @return {int} height (in pixes) of a horizontal scrollbar
             * @private
             */
            _determineHorizontalScrollBarHeight: function() {
                var div = document.createElement("div"), iHeight;
                div.style.visibility = "hidden";
                div.style.overflow = "scroll";
                document.body.appendChild(div);
                iHeight = div.offsetHeight - div.clientHeight;
                document.body.removeChild(div);
                return iHeight;
            },

            /**
             * Determine the width of a vertical scrollbar since it can vary between browsers
             * @return {int} width (in pixes) of a vertical scrollbar
             * @private
             */
            _determineVerticalScrollBarWidth: function() {

                // Use a cache of the function result if available.
                // This shouldn't change at all between between calls. This
                // function can be called frequently by WideTable has shown
                // to be taking too much time without the cache.
                if (TableUtils._determineVerticalScrollBarWidth._result) {
                    return TableUtils._determineVerticalScrollBarWidth._result;
                }

                // https://stackoverflow.com/questions/13382516/getting-scroll-bar-width-using-javascript
                var outer = document.createElement("div");
                outer.style.visibility = "hidden";
                outer.style.width = "100px";
                outer.style.msOverflowStyle = "scrollbar"; // needed for WinJS apps

                document.body.appendChild(outer);

                var widthNoScroll = outer.offsetWidth;
                // force scrollbars
                outer.style.overflow = "scroll";

                // add innerdiv
                var inner = document.createElement("div");
                inner.style.width = "100%";
                outer.appendChild(inner);

                var widthWithScroll = inner.offsetWidth;

                // remove divs
                outer.parentNode.removeChild(outer);

                var iResult = widthNoScroll - widthWithScroll;
                TableUtils._determineVerticalScrollBarWidth._result = iResult;

                return iResult;
            },

            /**
             * Returns true if the passed width string represents an
             * absolutely sized width. Note: It is assumed that
             * font-sizes for a page are not dynamic. Thus rem and em
             * are considered absolutely sized.
             *
             * @param A string representing a Column width
             * @return True if the input is an absolute width, false otherwise
             */
            isAbsoluteWidth: function(sWidth) {
                var endsWith = jQuery.sap.endsWith;
                return ["px", "rem", "em"].some(function(sType) { return endsWith(sWidth, sType); });
            },

            /**
             * Invoke a callback function upon Table's next rerender.
             * @param {sas.hc.ui.table.Table} oTable Instance of the table
             * @param {function} fn callback function
             */
            _attachAfterInvalidationOnce: function(oTable, fn) {
                var oDelegate = {
                    onAfterRendering: function() {
                        oTable.removeDelegate(oDelegate);
                        fn.call(oTable);
                    }
                };
                oTable.addDelegate(oDelegate);
            }
        };

        TableGrouping.TableUtils = TableUtils; // Avoid cyclic dependency

        return TableUtils;

    }, /* bExport= */ true);
