/*!
 * (c) 2014-2018, SAS Institute Inc.
 */

//Provides default renderer for control sas.hc.ui.table.Table
sap.ui.define([
    'jquery.sap.global',
    'sap/ui/core/Control',
    'sap/ui/core/library', // DataTypes: [TextAlign]
    'sap/ui/core/theming/Parameters',
    'sap/ui/Device',
    './library',
    './TableUtils',
    './Column',
    './ColumnGroup',
    'sap/ui/core/Renderer',
    'sap/ui/core/IconPool'
],
function(jQuery, Control, coreLibrary, Parameters, Device, library, TableUtils, Column, ColumnGroup, Renderer, IconPool) {
    "use strict";


    // shortcuts
    var SelectionMode = library.SelectionMode,
        VisibleRowCountMode = library.VisibleRowCountMode,
        TextAlign = coreLibrary.TextAlign;

    //var logger = library._getLogger("TableRenderer");

    // helper function to decide when to render a partial group (WideTable support)
    function shouldRenderPartialGroup(oTable, iHeaderIdx, iColIdx) {
        if (!oTable || !oTable._aRenderArray) {
            return false;
        }
        var oRow = oTable._aRenderArray[iHeaderIdx];
        if (!!oRow[iColIdx]) {
            return false;
        }
        // then from [h][c], look above for Column instance
        while (iHeaderIdx > 0) {
            iHeaderIdx--;
            oRow = oTable._aRenderArray[iHeaderIdx];
            if (oRow[iColIdx] instanceof Column) {
                return false;
            }
        }
        return true;
    }

    /**
     * Table renderer.
     * @namespace
     */
    var TableRenderer = {};

    /**
     * Renders the HTML for the given control, using the provided {@link sap.ui.core.RenderManager}.
     *
     * @param {sap.ui.core.RenderManager} rm the RenderManager that can be used for writing to the Render-Output-Buffer
     * @param {sap.ui.core.Control} oTable an object representation of the control that should be rendered
     */
    TableRenderer.render = function(rm, oTable) {
        // basic table div
        rm.write("<div");
        oTable._getAccRenderExtension().writeAriaAttributesFor(rm, oTable, "ROOT");
        rm.writeControlData(oTable);
        rm.addClass("sapUiTable");
        if ('ontouchstart' in document) {
            rm.addClass("sapUiTableTouch");
        }
        rm.addClass("sapUiTableSelMode" + oTable.getSelectionMode());

        if (oTable.getColumnHeaderVisible()) {
            rm.addClass("sapUiTableCHdr"); // show column headers
        }
        if (TableUtils.hasRowHeader(oTable)) {
            rm.addClass("sapUiTableRSel"); // show row selector
        }

        // This class flags whether the sap.m. library is loaded or not.
        var sSapMTableClass = library.TableHelper.addTableClass();
        if (sSapMTableClass) {
            rm.addClass(sSapMTableClass);
        }

        if (oTable._isVSbRequired()) {
            rm.addClass("sapUiTableVScr"); // show vertical scrollbar
        }

        var oSelMode = oTable.getSelectionMode();
        if (oTable.getEditable() || oSelMode !== SelectionMode.None) {
            rm.addClass("sapUiTableEdt"); // editable (background color)
        }

        if (TableUtils.isNoDataVisible(oTable)) {
            rm.addClass("sapUiTableEmpty"); // no data!
        }

        if (oTable.getShowOverlay()) {
            rm.addClass("sapUiTableOverlay");
        }

        if (oTable.getTotalFrozenColumnCount() > 0) {
            rm.addClass("sasUiTableFrozenColumns");
        }

        var sModeClass = TableUtils.Grouping.getModeCssClass(oTable);
        if (sModeClass) {
            rm.addClass(sModeClass);
        }

        if (oTable.getWidth()) {
            rm.addStyle("width", oTable.getWidth());
        }

        /* called before writing classes and styles to the main table element to allow the summary header to appear
           independently of the column header's visibility */
        if (oTable.getShowSummaryHeader()) {
            rm.addClass("sasUiTableCSHdr");
        }

        if (oTable.getVisibleRowCountMode() === VisibleRowCountMode.Auto) {
            rm.addStyle("height", "0px");
            if (oTable._bFirstRendering) {
                rm.addClass("sapUiTableNoOpacity");
            }
        }

        if (oTable.getShowAlternateRowShading()) {
            rm.addClass("sasUiTableRowShading");
        }

        if (oTable.getShowOuterBorder()) {
            rm.addClass("sasUiTableOuterBorder");
        }

        if (!oTable.getEnablePixelScrolling()) {
            rm.addClass("sasUiTableNoPixelScrolling");
        }

        rm.writeClasses();
        rm.writeStyles();
        rm.write(">");

        this.renderTabElement(rm, "sapUiTableOuterBefore");

        if (oTable.getTitle()) {
            this.renderHeader(rm, oTable, oTable.getTitle());
        }

        if (oTable.getToolbar()) {
            this.renderToolbar(rm, oTable, oTable.getToolbar());
        }

        if (oTable.getExtension() && oTable.getExtension().length > 0) {
            this.renderExtensions(rm, oTable, oTable.getExtension());
        }
        rm.write("<div");
        rm.writeAttribute("id", oTable.getId() + "-sapUiTableCnt");
        rm.addClass("sapUiTableCnt");

        this.tableCntHook(rm, oTable);

        rm.writeClasses();

        oTable._getAccRenderExtension().writeAriaAttributesFor(rm, oTable, "CONTENT");

        // Define group for F6 handling
        rm.writeAttribute("data-sap-ui-fastnavgroup", "true");
        rm.write(">");

        this.renderColRsz(rm, oTable);

        this.preRenderColHdrHook(rm, oTable);

        this.renderColHdr(rm, oTable);
        this.renderTable(rm, oTable);

        this.postRenderTableA11yHook(rm, oTable);

        oTable._getAccRenderExtension().writeHiddenAccTexts(rm, oTable);

        this.renderOptionsMenuButton(rm, oTable);

        rm.write("<div");
        rm.addClass("sapUiTableOverlayArea");
        rm.writeClasses();
        rm.writeAttribute("tabindex", "0");
        rm.writeAttribute("id", oTable.getId() + "-overlay");
        oTable._getAccRenderExtension().writeAriaAttributesFor(rm, oTable, "OVERLAY");
        rm.write("></div>");

        rm.write("</div>");

        if (oTable.getFooter()) {
            this.renderFooter(rm, oTable, oTable.getFooter());
        }

        this.renderTabElement(rm, "sapUiTableOuterAfter");

        rm.write("</div>");
    };

    // =============================================================================
    // BASIC AREAS OF THE TABLE
    // =============================================================================

    TableRenderer.renderHeader = function(rm, oTable, oTitle) {
        rm.write("<div");
        rm.addClass("sapUiTableHdr");
        rm.writeClasses();
        oTable._getAccRenderExtension().writeAriaAttributesFor(rm, oTable, "TABLEHEADER");
        rm.write(">");

        rm.renderControl(oTitle);

        rm.write("</div>");
    };

    TableRenderer.renderToolbar = function(rm, oTable, oToolbar) {
        rm.write("<div");
        rm.addClass("sapUiTableTbr");
        if (typeof oToolbar.getStandalone !== "function") {
            // for the mobile toolbar we add another class
            rm.addClass("sapUiTableMTbr");
        }
        rm.writeClasses();
        oTable._getAccRenderExtension().writeAriaAttributesFor(rm, oTable, "TABLESUBHEADER");
        rm.write(">");

        // toolbar has to be embedded (not standalone)!
        if (typeof oToolbar.getStandalone === "function" && oToolbar.getStandalone()) {
            oToolbar.setStandalone(false);
        }

        // set the default design of the toolbar
        var Toolbar = sap.ui.require("sap/m/Toolbar");
        if ( Toolbar && oToolbar instanceof Toolbar ) {
            oToolbar.setDesign(Parameters.get("sapUiTableToolbarDesign"), true);
        }

        rm.renderControl(oToolbar);

        rm.write("</div>");
    };

    TableRenderer.renderExtensions = function(rm, oTable, aExtensions) {
        for (var i = 0, l = aExtensions.length; i < l; i++) {
            this.renderExtension(rm, oTable, aExtensions[i]);
        }
    };

    TableRenderer.renderExtension = function(rm, oTable, oExtension) {
        rm.write("<div");
        rm.addClass("sapUiTableExt");
        rm.writeClasses();
        oTable._getAccRenderExtension().writeAriaAttributesFor(rm, oTable, "TABLESUBHEADER");
        rm.write(">");

        rm.renderControl(oExtension);

        rm.write("</div>");
    };

    TableRenderer.renderTable = function(rm, oTable) {
        rm.write("<div");
        rm.writeAttribute("id", oTable.getId() + "-tableCCnt");
        rm.writeAttribute("role", "presentation");
        rm.addClass("sapUiTableCCnt");
        rm.writeClasses();
        rm.write(">");

        this.renderTableCCnt(rm, oTable);
        rm.write("</div>");
        this.renderHSb(rm, oTable);
    };

    TableRenderer.renderTableCCnt = function(rm, oTable) {
        this.renderTabElement(rm, "sapUiTableCtrlBefore");
        this.renderTableCtrl(rm, oTable);
        this.renderRowHdr(rm, oTable);
        this.renderVSb(rm, oTable);
        this.renderTabElement(rm, "sapUiTableCtrlAfter");

        rm.write("<div");
        rm.addClass("sapUiTableCtrlEmpty");
        rm.writeClasses();
        rm.writeAttribute("tabindex", "0");
        rm.writeAttribute("id", oTable.getId() + "-noDataCnt");
        oTable._getAccRenderExtension().writeAriaAttributesFor(rm, oTable, "NODATA");
        rm.write(">");

        this.renderNoData(rm, oTable);

        rm.write("</div>");
    };

    TableRenderer.renderFooter = function(rm, oTable, oFooter) {
        rm.write("<div");
        rm.addClass("sapUiTableFtr");
        rm.writeClasses();
        oTable._getAccRenderExtension().writeAriaAttributesFor(rm, oTable, "TABLEFOOTER");
        rm.write(">");

        rm.renderControl(oFooter);

        rm.write("</div>");
    };

    // =============================================================================
    // COLUMN HEADER OF THE TABLE
    // =============================================================================

    TableRenderer.renderColHdr = function(rm, oTable) {
        var iHeaderColumnCount = oTable._calculateHeaderColumnCount(),
            iHeaderRowCount = TableUtils.getHeaderRowCount(oTable);

        rm.write("<div");
        rm.addClass("sapUiTableColHdrCnt");
        rm.writeClasses();
        if (oTable.getColumnHeaderHeight() > 0) {
            rm.addStyle("height", this.determineColHdrCntHeight(oTable));
        }
        oTable._getAccRenderExtension().writeAriaAttributesFor(rm, oTable, "COLUMNHEADER_ROW");
        rm.writeStyles();
        rm.write(">");

        this.renderColRowHdr(rm, oTable);

        var iFrozenColumnCount = oTable.getTotalFrozenColumnCount();

        if (iFrozenColumnCount > 0) {
            this.renderColHdrSection(rm, oTable, 0, iFrozenColumnCount, iHeaderRowCount, iHeaderColumnCount);
        }

        var rangeToRender = this.getColumnRenderRange(oTable);
        this.renderColHdrSection(rm, oTable, rangeToRender.start, rangeToRender.end, iHeaderRowCount, iHeaderColumnCount);

        rm.write("</div>");

    };

    TableRenderer.renderColHdrSection = function(rm, oTable, iStart, iEnd, iHeaderRowCount, iHeaderColumnCount) {

        var iFrozenColumnCount = oTable.getTotalFrozenColumnCount();
        var bIsFixedSection = iStart < iFrozenColumnCount && iEnd === iFrozenColumnCount; // Not sure what to do if range covers both fixed and scrollable.
        var aCols = oTable.retrieveLeafColumns();
        var aVisibleColumns = oTable._getVisibleColumns();

        var sId = oTable.getId() + (bIsFixedSection ? "-sapUiTableColHdrFixed" : "-sapUiTableColHdrScr");
        var sMainClass = bIsFixedSection ? "sapUiTableColHdrFixed" :"sapUiTableColHdrScr";

        rm.write("<div");
        rm.writeAttribute("id", sId);
        rm.writeAttribute("role", "presentation");
        rm.addClass(sMainClass);
        if (aCols.length === 0 && !oTable.hasColumnGroups()) {
            rm.addClass("sapUiTableHasNoColumns");
        }
        rm.writeClasses();
        if (!bIsFixedSection && iFrozenColumnCount > 0) {
            if (oTable._bRtlMode) {
                rm.addStyle("margin-right", "0");
            } else {
                rm.addStyle("margin-left", "0");
            }
            rm.writeStyles();
        }
        rm.write(">");

        rm.write("<div");
        rm.addClass("sapUiTableColHdr");
        rm.addClass("sapUiTableNoOpacity");
        rm.writeClasses();

        if (!bIsFixedSection && !oTable._hasSomePercentageWidthColumn()) {
            var iScrollableColumnWidthTotal = oTable._getColumnsWidth(iStart, iEnd);
            // TODO Can we remove this style?
            rm.addStyle('min-width', iScrollableColumnWidthTotal + 'px');
        }

        rm.writeStyles();
        rm.writeAttribute("role", "presentation");
        rm.write(">");

        rm.write("<table");
        rm.addClass("sapUiTableCtrl");
        if (oTable.hasColumnGroups()) {
            rm.addClass("sasUiTableMultiLineHeader");
        }
        if (bIsFixedSection) {
            rm.addClass("sapUiTableCtrlFixed");
        } else {
            rm.addClass("sapUiTableCtrlScroll");
        }
        rm.writeClasses();

        var iSectionWidth = oTable._getColumnsWidth(iStart, iEnd);
        rm.addStyle("min-width", iSectionWidth + "px");

        //S1519975: fixed sections will have known discrete widths (and for MS Edge v44 need to have it explicitly defined)
        //Firefox, chrome, and safari need a defined width for the fixed table
        if ( (bIsFixedSection && !!Device.browser.edge) || ( (oTable.getTotalFrozenColumnCount() < aVisibleColumns.length) && bIsFixedSection && (!!Device.browser.firefox || !!Device.browser.chrome || !!Device.browser.safari) ) ) {
            rm.addStyle("width", iSectionWidth + "px");
        }
        this.addCustomColHeaderSectionAttributes(rm, oTable, iStart, iEnd);
        oTable._getAccRenderExtension().writeAriaAttributesFor(rm, oTable, "TABLE");
        rm.writeStyles();

        rm.write("><thead><tr");
        rm.addClass("sapUiTableCtrlCol");
        rm.writeClasses();
        rm.write(">");
        var sWidth;
        jQuery.each(aVisibleColumns, function (iIndex, oColumn) {
            if (iIndex >= iStart && iIndex < iEnd) {
                rm.write("<th");
                rm.writeAttribute("data-sap-ui-headcolindex", iIndex);
                rm.writeAttribute("data-sap-ui-colid", oColumn.getId());
                if (oColumn.getWidth) {
                    sWidth = oColumn.getWidth();
                    if (bIsFixedSection) {
                        sWidth = sWidth || "160px";
                    }
                    rm.addStyle("width", sWidth);
                    rm.writeStyles();
                }
                rm.write("></th>");
            }
        });
        rm.write("</tr></thead>");

        rm.write("<tbody>");
        if (oTable._aRenderArray) {
            for (var h = 0; h < oTable._aRenderArray.length; h++) {
                this.renderTableRowStart(rm, oTable);
                var rowRenderArray = oTable._aRenderArray[h];
                for (var i = iStart; i < iEnd; i++) {
                    // render undefined columns because those are placeholders with multi labels
                    var bIsPlaceholder = !rowRenderArray[i];
                    if (i < rowRenderArray.length && i === iStart && shouldRenderPartialGroup(oTable, h, i)) {
                        //0. find column group
                        var iAdjustedIdx = i;
                        while (!rowRenderArray[iAdjustedIdx] && iAdjustedIdx > 0) {
                            iAdjustedIdx--;
                        }
                        var oGroupColumn = rowRenderArray[iAdjustedIdx];
                        if (!oGroupColumn) {
                            continue;
                        }
                        //1. calculate out how many empty spaces are here to fill
                        var iColSpan = oGroupColumn.calculateColumnSpan();
                        var iRemainder = iColSpan - (iStart - iAdjustedIdx);
                        //2. render portion that fits into visible region
                        var oTempCol = oGroupColumn.clone();
                        oTempCol.calculateColumnSpan = function () {
                            return iRemainder;
                        };
                        this.renderCol(rm, oTable, oTempCol, iAdjustedIdx, h, false, iHeaderRowCount, iHeaderColumnCount);
                        //3. update i to be the last index of leaves in group
                        var iNext = iAdjustedIdx + (iColSpan - 1);
                        if (iNext > i) {
                            i = iNext;
                        }
                    } else if (
                        i < rowRenderArray.length &&
                        (bIsPlaceholder || this.shouldRenderColumnAtIndex(oTable, rowRenderArray[i], i, bIsFixedSection))
                    ) {
                        this.renderCol(rm, oTable, rowRenderArray[i], i, h, bIsPlaceholder, iHeaderRowCount, iHeaderColumnCount);
                    }
                }
                this.renderTableRowEnd(rm);
            }
        }
        rm.write("</tbody></table></div></div>");
    };

    TableRenderer.renderTableSectionStart = function(rm, oTable, bIsFixedSection, iStart, iEnd) {
        var aCols = oTable.retrieveLeafColumns();
        rm.write("<table");
        rm.addClass("sapUiTableCtrl");
        if (bIsFixedSection) {
            rm.addClass("sapUiTableCtrlFixed");
        } else {
            rm.addClass("sapUiTableCtrlScroll");
        }
        rm.writeClasses();

        var iSectionWidth = oTable._getColumnsWidth(iStart, iEnd);
        rm.addStyle("min-width", iSectionWidth + "px");
        //Firefox and chrome and safari need a defined width for the fixed table
        if (bIsFixedSection && (!!Device.browser.firefox || !!Device.browser.chrome || !!Device.browser.safari)) {
            rm.addStyle("width", iSectionWidth + "px");
        }
        this.addCustomColHeaderSectionAttributes(rm, oTable, iStart, iEnd);
        oTable._getAccRenderExtension().writeAriaAttributesFor(rm, oTable, "TABLE");
        rm.writeStyles();
        rm.write("><thead><tr");
        rm.writeAttribute("aria-owns", oTable.getId()+"-selall");
        rm.addClass("sapUiTableCtrlCol");
        rm.writeClasses();
        rm.write(">");

        if (TableUtils.hasRowHeader(oTable)) {
            rm.write("<th");
            rm.addStyle("width", "0px");
            rm.writeStyles();
            oTable._getAccRenderExtension().writeAriaAttributesFor(rm, oTable, "SELECTALL_COLHEADER");
            rm.writeAttribute("id", oTable.getId() + "-hdrcolsel");
            rm.addClass("sapUiTableColSel");
            rm.writeClasses();
            rm.write("></th>");
        } else if (aCols.length === 0) {
            // no cols => render th => avoids rendering issue in firefox
            rm.write("<th></th>");
        }
    };

    TableRenderer.renderTableRowStart = function(rm, oTable) {
        var aCols = oTable.retrieveLeafColumns();

        rm.write("<tr");
        rm.writeAttribute("aria-owns", oTable.getId()+"-selall");
        rm.write(">");

        if (TableUtils.hasRowHeader(oTable)) {
            rm.write("<th");
            rm.addStyle("width", "0px");
            rm.writeStyles();
            oTable._getAccRenderExtension().writeAriaAttributesFor(rm, oTable, "SELECTALL_COLHEADER");
            rm.writeAttribute("id", oTable.getId() + "-hdrcolsel");
            rm.addClass("sapUiTableColSel");
            rm.writeClasses();
            rm.write("></th>");
        } else if (aCols.length === 0) {
            // no cols => render th => avoids rendering issue in firefox
            rm.write("<th></th>");
        }
    };

    TableRenderer.renderTableRowEnd = function(rm) {
        rm.write("</tr>");
    };

    TableRenderer.renderTableSectionEnd = function(rm) {
        rm.write("</tr></tbody></table>");
    };

    TableRenderer.addCustomColHeaderSectionAttributes = function(rm, oTable, iStart, iEnd) {
        // Hook function mostly for WideTable.
    };

    TableRenderer.renderColRowHdr = function(rm, oTable) {
        var oSelMode = oTable.getSelectionMode();
        var bEnabled = false;
        if (!((oSelMode === "MultiToggle" || oSelMode === "ParentChild") && oTable.getEnableSelectAll())) {
            //do not render any part of the 'select all' area if it is not needed
            return;
        }

        rm.write("<div");
        rm.writeAttribute("id", oTable.getId() + "-selall");
        if ((oSelMode === "MultiToggle" || oSelMode === "ParentChild") && oTable.getEnableSelectAll()) {
            rm.writeAttributeEscaped("title", oTable._oResBundle.getText("TBL_SELECT_ALL.txt"));
            //TODO: remove second _getSelectableRowCount Call!
            if (oTable._getSelectableRowCount() === 0 || oTable._getSelectableRowCount() !== oTable.getSelectedIndices().length) {
                rm.addClass("sapUiTableSelAll");
            }
            rm.addClass("sapUiTableSelAllEnabled");
            bEnabled = true;
        } else {
            rm.addClass("sapUiTableSelAllDisabled");
        }
        rm.addClass("sapUiTableColRowHdr");
        rm.writeClasses();

        rm.writeAttribute("tabindex", "-1");

        oTable._getAccRenderExtension().writeAriaAttributesFor(rm, oTable, "COLUMNROWHEADER", {enabled: bEnabled});

        rm.write(">");
        if (oTable.getSelectionMode() !== SelectionMode.Single) {

            if (oTable.getEnableSelectAll() && jQuery.isFunction(oTable.getSelectAllControl) === true && oTable.getSelectAllControl() !== undefined) {
                this.renderSelectAllControl(rm, oTable);
            } else {
                rm.write("<div");
                rm.addClass("sapUiTableColRowHdrIco");
                rm.writeClasses();
                if (oTable.getColumnHeaderHeight() > 0) {
                    rm.addStyle("height", oTable.getColumnHeaderHeight() + "px");
                }
                rm.write(">");
                rm.write("</div>");
            }
        }
        rm.write("</div>");
    };

    TableRenderer.renderColStart = function(rm, oTable, oColumn, iIndex, sHeaderId, oLabel, iColSpan, iRowSpan, bInvisible, iHeaderColumnCount) {
        var sId = iIndex;

        if (oColumn) {
            sId = oColumn.getId();
        }

        rm.write("<td");
        if (iColSpan && iColSpan > 1) {
            rm.writeAttribute("colspan", iColSpan);
        }
        if (iRowSpan && iRowSpan > 1) {
            rm.writeAttribute("rowspan", iRowSpan);
        }
        if (bInvisible || !oColumn) {
            rm.addStyle("display", "none");
            rm.writeStyles();
        }

        oTable._getAccRenderExtension().writeAriaAttributesFor(rm, oTable, "COLUMNHEADER", {
            column: oColumn,
            headerId: sHeaderId,
            index: iIndex,
            label: oLabel
            // NOTE: In the future if we change the way A11Y works, we may
            // need to add rowspan and colspan for tables with column groups
        });

        rm.writeAttribute("id", sId + "-td");
        rm.addClass("sapUiTableColHeader");

        var bIsLastColumn = (iIndex + iColSpan) >= iHeaderColumnCount;
        if (bIsLastColumn && iColSpan > 1) {
            rm.addClass("sasUiTableColHeaderLastVisible");
        }

        rm.writeClasses();
        rm.write(">");
    };

    TableRenderer.renderColEnd = function(rm) {
        rm.write("</td>");
    };

    TableRenderer.renderCol = function(rm, oTable, oColumn, iIndex, iHeader, bInvisible, iHeaderRowCount, iHeaderColumnCount) {
        var sHeaderId,
            oLabel,
            iColSpan = 1,
            iRowSpan = 1;

        if (iHeader === -1 && oColumn && jQuery.isFunction(oColumn.getSummaryLabel) === true)
        {
            oLabel = oColumn.getSummaryLabel();
        } else if (oColumn) {
            oLabel = oColumn.getLabel();
            sHeaderId = oColumn.getId();
        }

        if (oColumn && oColumn.calculateColumnSpan) {
            iColSpan = oColumn.calculateColumnSpan();
        }
        if (oColumn && oColumn.calculateRowSpan) {
            iRowSpan = oColumn.calculateRowSpan();
        }
        // consider iRowSpan to equate max height of column/group
        if (oColumn instanceof Column && (iRowSpan<=iHeader+1)) {
            iRowSpan = iHeaderRowCount - iRowSpan - iHeader + 1;
        } else {
            iRowSpan = 1;
        }

        this.renderColStart(rm, oTable, oColumn, iIndex, sHeaderId, oLabel, iColSpan, iRowSpan, bInvisible, iHeaderColumnCount);

        rm.write("<div");
        if (oColumn) {
            rm.writeElementData(oColumn);
        }

        if (sHeaderId && iColSpan <= 1) {
            rm.writeAttribute('data-sap-ui-colid', sHeaderId);
        }
        if (iColSpan <= 1) {
            rm.writeAttribute("data-sap-ui-colindex", iIndex);
        }

        rm.writeAttribute("tabindex", "-1");

        rm.addClass("sapUiTableCol");
        if (oColumn instanceof ColumnGroup) {
            rm.addClass("sapUiTableColGroup");
        }

        // if column is in keyboard resizing mode, add style class so it's visually apparent
        if (oColumn && oColumn._bKeyboardResizing === true) {
            rm.addClass("sapUiTableColKeyboardResizeInProcess");
        }

        if (oTable.getTotalFrozenColumnCount() === iIndex + 1) {
            rm.addClass("sapUiTableColLastFixed");
        }

        // S1309081
        if (oColumn && oColumn.getStyleClass && oColumn.getStyleClass()) {
            rm.addClass(oColumn.getStyleClass());
        }

        rm.writeClasses();

        if (oTable.getColumnHeaderHeight() > 0) {
            rm.addStyle("height", oTable.getColumnHeaderHeight() + "px");
        }
        rm.writeStyles();
        var sTooltip;
        if (oColumn) {
            sTooltip = oColumn.getTooltip_AsString();
        }
        if (sTooltip) {
            rm.writeAttributeEscaped("title", sTooltip);
        }
        rm.write("><div");
        rm.addClass("sapUiTableColCell");

        // HTMLCOMMONS-4582
        if (oColumn) {
            this.addCustomTableColCellCssClass(rm, oColumn);
        }

        rm.writeClasses();
        var sColHAlign;
        if (oColumn && oColumn.getHAlign) {
            sColHAlign = oColumn.getHAlign();
        }
        var sHAlign = Renderer.getTextAlign(sColHAlign, oLabel && oLabel.getTextDirection && oLabel.getTextDirection());
        if (sHAlign) {
            rm.addStyle("text-align", sHAlign);
        }
        rm.writeStyles();
        rm.write(">");

        // HTMLCOMMONS-4582
        if (oColumn && iColSpan <= 1) {
            this.renderColPreControlRenderHook(rm, oColumn);    //render icon container
        }

        if (oLabel) {
            if (iColSpan > 1) {
                oLabel.setTextAlign(TextAlign.Center);
                oLabel.addStyleClass("sasUiTableMultiColumnHeader");
            }
            rm.renderControl(oLabel);
        }
        rm.write("</div></div>");

        this.renderColEnd(rm);
    };

    TableRenderer.renderColRsz = function(rm, oTable) {
        rm.write("<div");
        rm.writeAttribute("id", oTable.getId() + "-rsz");
        rm.writeAttribute("tabindex", "-1");
        rm.addClass("sapUiTableColRsz");
        rm.writeClasses();
        rm.write("></div>");
    };

    // =============================================================================
    // CONTENT AREA OF THE TABLE
    // =============================================================================

    TableRenderer.renderRowHdr = function(rm, oTable) {
        rm.write("<div");
        rm.writeAttribute("id", oTable.getId() + "-sapUiTableRowHdrScr");
        rm.addClass("sapUiTableRowHdrScr");
        rm.addClass("sapUiTableNoOpacity");
        rm.writeClasses();
        oTable._getAccRenderExtension().writeAriaAttributesFor(rm, oTable, "ROWHEADER_COL");
        rm.write(">");

        // start with the first current top visible row
        for (var row = 0, count = oTable.getRows().length; row < count; row++) {
            this.renderRowHdrRow(rm, oTable, oTable.getRows()[row], row);
        }

        rm.write("</div>");
    };

    TableRenderer._addFixedRowCSSClasses = function(rm, oTable, iIndex) {
        var iFixedRowCount = oTable.getFixedRowCount();
        var iFixedBottomRowCount = oTable.getFixedBottomRowCount();
        var iVisibleRowCount = oTable.getVisibleRowCount();
        var iFirstVisibleRow = oTable.getFirstVisibleRow();

        if (iFixedRowCount > 0) {
            if (iIndex < iFixedRowCount) {
                rm.addClass("sapUiTableFixedTopRow");
            }

            if (iIndex === iFixedRowCount - 1) {
                rm.addClass("sapUiTableFixedLastTopRow");
            }
        }

        if (iFixedBottomRowCount > 0) {
            var bIsPreBottomRow = false;
            var oBinding = oTable.getBinding("rows");
            if (oBinding) {
                if (oTable._iBindingLength >= iVisibleRowCount) {
                    bIsPreBottomRow = (iIndex === iVisibleRowCount - iFixedBottomRowCount - 1);
                } else {
                    bIsPreBottomRow = (iFirstVisibleRow + iIndex) === (oTable._iBindingLength - iFixedBottomRowCount - 1) && (oTable.getFirstVisibleRow() + iIndex) < oTable._iBindingLength;
                }
            }

            if (bIsPreBottomRow) {
                rm.addClass("sapUiTableFixedPreBottomRow");
            }
        }
    };

    TableRenderer.renderRowHdrRow = function(rm, oTable, oRow, iRowIndex) {
        rm.write("<div");
        rm.writeAttribute("id", oTable.getId() + "-rowsel" + iRowIndex);
        rm.writeAttribute("data-sap-ui-rowindex", iRowIndex);
        rm.addClass("sapUiTableRowHdr");
        this._addFixedRowCSSClasses(rm, oTable, iRowIndex);
        var bRowSelected = false;
        if (oRow._bHidden) {
            rm.addClass("sapUiTableRowHidden");
        } else {
            if (oTable.isIndexSelected(oTable._getAbsoluteRowIndex(iRowIndex))) {
                rm.addClass("sapUiTableRowSel");
                bRowSelected = true;
            }
        }

        rm.writeClasses();
        if (oTable.getRowHeight() > 0) {
            rm.addStyle("height", oTable.getRowHeight() + "px");
        }

        rm.writeAttribute("tabindex", "-1");

        oTable._getAccRenderExtension().writeAriaAttributesFor(rm, oTable, "ROWHEADER", {
            index: oRow.getIndex(),     //using logical index
            rowSelected: bRowSelected
        });

        var aCellIds = [];
        jQuery.each(oRow.getCells(), function(iIndex, oCell) {
            aCellIds.push(oRow.getId() + "-col" + iIndex);
        });

        rm.writeStyles();
        rm.write(">");
        this.writeRowSelectorContent(rm, oTable, oRow, iRowIndex);

        this.renderRowHdrRowSelection(rm, oTable, oRow, iRowIndex);

        rm.write("</div>");
    };

    TableRenderer.renderTableCtrl = function(rm, oTable) {

        if (oTable.getTotalFrozenColumnCount() > 0) {
            rm.write("<div");
            rm.writeAttribute("id", oTable.getId() + "-sapUiTableCtrlScrFixed");
            rm.addClass("sapUiTableCtrlScrFixed");
            rm.writeClasses();
            rm.write(">");

            this.renderTableControl(rm, oTable, true);

            rm.write("</div>");
        }

        rm.write("<div");
        rm.writeAttribute("id", oTable.getId() + "-sapUiTableCtrlScr");
        rm.writeAttribute("role", "presentation");
        rm.addClass("sapUiTableCtrlScr");
        rm.writeClasses();
        if (oTable.getTotalFrozenColumnCount() > 0) {
            if (oTable._bRtlMode) {
                rm.addStyle("margin-right", "0");
            } else {
                rm.addStyle("margin-left", "0");
            }
            rm.writeStyles();
        }
        rm.write(">");

        rm.write("<div");
        rm.writeAttribute("id", oTable.getId() + "-tableCtrlCnt");
        rm.writeAttribute("role", "presentation");
        rm.addClass("sapUiTableCtrlCnt");
        rm.writeClasses();
        var sVisibleRowCountMode = oTable.getVisibleRowCountMode();
        if (oTable._iTableRowContentHeight && sVisibleRowCountMode === VisibleRowCountMode.Fixed) {
            var sStyle = "min-height";
            rm.addStyle(sStyle, oTable._iTableRowContentHeight + "px");
            rm.writeStyles();
        }
        rm.write(">");

        this.renderTableControl(rm, oTable, false);

        rm.write("</div></div>");
    };


    TableRenderer.renderTableControl = function(rm, oTable, bFixedTable) {

        //HTMLCOMMONS-1559
        var oRenderRange = this.getColumnRenderRange(oTable, bFixedTable);
        var iStartColumn = oRenderRange.start;
        var iEndColumn = oRenderRange.end;

        var iFixedRows = oTable.getFixedRowCount();
        var iFixedBottomRows = oTable.getFixedBottomRowCount();
        var aRows = oTable.getRows();

        if (iFixedRows > 0) {
            this.renderTableControlCnt(rm, oTable, bFixedTable, iStartColumn, iEndColumn, true, false, 0, iFixedRows);
        }
        this.renderTableControlCnt(rm, oTable, bFixedTable, iStartColumn, iEndColumn, false, false, iFixedRows, aRows.length - iFixedBottomRows);
        if (iFixedBottomRows > 0 && aRows.length > 0) {
            this.renderTableControlCnt(rm, oTable, bFixedTable, iStartColumn, iEndColumn, false, true, aRows.length - iFixedBottomRows, aRows.length);
        }
    };

    TableRenderer.renderTableControlCnt = function(rm, oTable, bFixedTable, iStartColumn, iEndColumn, bFixedRow, bFixedBottomRow, iStartRow, iEndRow) {
        rm.write("<table");
        var sId = oTable.getId() + "-table";

        if (bFixedTable) {
            sId += "-fixed";
            rm.addClass("sapUiTableCtrlFixed");
        } else {
            rm.addClass("sapUiTableCtrlScroll");
        }
        if (bFixedRow) {
            sId += "-fixrow";
            rm.addClass("sapUiTableCtrlRowFixed");
        } else if (bFixedBottomRow) {
            sId += "-fixrow-bottom";
            rm.addClass("sapUiTableCtrlRowFixedBottom");
        } else {
            rm.addClass("sapUiTableCtrlRowScroll");
        }
        rm.writeAttribute("id", sId);

        oTable._getAccRenderExtension().writeAriaAttributesFor(rm, oTable, "TABLE");

        rm.addClass("sapUiTableCtrl");

        this.addCustomControlCntAttributes(rm, oTable, bFixedTable, iStartColumn, iEndColumn);

        rm.writeClasses();
        rm.addStyle("min-width", oTable._getColumnsWidth(iStartColumn, iEndColumn) + "px");
        //Firefox and chrome and safari need a defined width for the fixed table (S1519975 edge too! is that everyone?)
        if (bFixedTable && (!!Device.browser.firefox || !!Device.browser.chrome || !!Device.browser.safari || !!Device.browser.edge)) {
            rm.addStyle("width", oTable._getColumnsWidth(iStartColumn, iEndColumn) + "px");
        }
        rm.writeStyles();
        rm.write(">");

        rm.write("<thead>");

        rm.write("<tr");
        rm.addClass("sapUiTableCtrlCol");
        if (iStartRow === 0) {
            rm.addClass("sapUiTableCtrlFirstCol");
        }
        rm.writeClasses();
        oTable._getAccRenderExtension().writeAriaAttributesFor(rm, oTable, "PRESENTATION");
        rm.write(">");

        var aCols = oTable.retrieveLeafColumns();
        if (TableUtils.hasRowHeader(oTable)) {
            rm.write("<th");
            rm.addStyle("width", "0px");
            rm.writeStyles();
            //if there is a fixedRowCount, we need to make sure the scrollable table needs to have id, and style class so the correct
            //column width can be computed.
            var aFirstRow = (oTable.getFixedRowCount() > 0 && iStartRow > 0) ? (iStartRow - oTable.getFixedRowCount()) : iStartRow;
            if (aFirstRow === 0) {
                oTable._getAccRenderExtension().writeAriaAttributesFor(rm, oTable, "TH");
                rm.writeAttribute("id", oTable.getId() + "-colsel");
                rm.addClass("sapUiTableColSel");
                rm.writeClasses();
            }
            rm.write("></th>");
        } else {
            if (aCols.length === 0) {
                // no cols => render th => avoids rendering issue in firefox
                rm.write("<th></th>");
            }
        }

        for (var col = iStartColumn, count = iEndColumn; col < count; col++) {
            var oColumn = aCols[col], sWidth;

            //HTMLCOMMONS-1559 - Support for Wide Table to conditionally not render certain columns
            if (this.shouldRenderColumnAtIndex(oTable, oColumn, col, bFixedTable)) {
                if (oColumn.getWidth) {
                    sWidth = oColumn.getWidth();
                }
                if (bFixedTable) {
                    sWidth = sWidth || "160px";
                }
                rm.write("<th");
                rm.addStyle("width", sWidth);
                rm.writeStyles();
                if (iStartRow === 0) {
                    oTable._getAccRenderExtension().writeAriaAttributesFor(rm, oTable, "TH", {column: oColumn});
                    rm.writeAttribute("id", oTable.getId() + "_col" + col);
                }
                rm.writeAttribute("data-sap-ui-headcolindex", col);
                rm.writeAttribute("data-sap-ui-colid", oColumn.getId());
                rm.write(">");
                if (iStartRow === 0 && TableUtils.getHeaderRowCount(oTable) === 0) {
                    if (oColumn.getMultiLabels().length > 0) {
                        rm.renderControl(oColumn.getMultiLabels()[0]);
                    } else {
                        rm.renderControl(oColumn.getLabel());
                    }
                }
                rm.write("</th>");
            }
        }

        // dummy column to fill the table width
        if (!bFixedTable && oTable._hasOnlyFixColumnWidths() && aCols.length > 0) {
            rm.write("<th></th>");
        }

        rm.write("</tr>");
        rm.write("</thead>");

        rm.write("<tbody>");

        var aVisibleColumns = oTable._getVisibleColumns();
        var bHasOnlyFixedColumns = oTable._hasOnlyFixColumnWidths();

        // render the table rows
        var aRows = oTable.getRows();
        if (aRows.length === 0) {
            // For the very first rendering in visibleRowCountMode Auto, there are no rows which can be rendered but
            // it's required to have a dummy row rendered to determine the default/expected row height since the controls
            // of the column template may expand the rowHeight.
            aRows = [oTable._getDummyRow()];
            iEndRow = 1;
        }
        // retrieve tooltip and aria texts only once and pass them to the rows _updateSelection function
        var mTooltipTexts = oTable._getAccExtension().getAriaTextsForSelectionMode(true);

        // check whether the row can be clicked to change the selection
        var bSelectOnCellsAllowed = TableUtils.isRowSelectionAllowed(oTable);
        for (var row = iStartRow, count = iEndRow; row < count; row++) {
            this.renderTableRow(rm, oTable, aRows[row], row, bFixedTable, iStartColumn, iEndColumn, false, aVisibleColumns, bHasOnlyFixedColumns, mTooltipTexts, bSelectOnCellsAllowed);
        }

        rm.write("</tbody>");
        rm.write("</table>");
    };

    TableRenderer.addTrClasses = function(rm, oTable, oRow, iRowIndex) {
        return;
    };

    TableRenderer.writeRowSelectorContent = function(rm, oTable, oRow, iRowIndex) {
        oTable._getAccRenderExtension().writeAccRowSelectorText(rm, oTable, oRow, iRowIndex);

        if (TableUtils.Grouping.isGroupMode(oTable)) {
            rm.write("<div");
            rm.writeAttribute("id", oRow.getId() + "-groupHeader");
            rm.writeAttribute("class", "sapUiTableGroupIcon");
            rm.write("></div>");

            if (TableUtils.Grouping.showGroupMenuButton(oTable)) {
                var oIconInfo = IconPool.getIconInfo("sap-icon://drop-down-list");
                rm.write("<div class='sapUiTableGroupMenuButton'>");
                rm.writeEscaped(oIconInfo.content);
                rm.write("</div>");
            }
        }
    };

    TableRenderer.renderTableRow = function(rm, oTable, oRow, iRowIndex, bFixedTable, iStartColumn, iEndColumn, bFixedRow, aVisibleColumns, bHasOnlyFixedColumns, mTooltipTexts, bSelectOnCellsAllowed) {
        rm.write("<tr");
        if (oRow._bDummyRow) {
            rm.addStyle("opacity", "0");
        }
        rm.addClass("sapUiTableTr");
        if (bFixedTable) {
            rm.writeAttribute("id", oRow.getId() + "-fixed");
        } else {
            rm.writeElementData(oRow);
        }
        if (oRow._bHidden) {
            rm.addClass("sapUiTableRowHidden");
        } else {
            if (oTable.isIndexSelected(oTable._getAbsoluteRowIndex(iRowIndex))) {
                rm.addClass("sapUiTableRowSel");
            }

            this.addTrClasses(rm, oTable, oRow, iRowIndex);
        }

        if (oTable.getShowAlternateRowShading()) {
            if (iRowIndex % 2 === 0) {
                rm.addClass("sapUiTableRowEven");
            } else {
                rm.addClass("sapUiTableRowOdd");
            }
        }

        this._addFixedRowCSSClasses(rm, oTable, iRowIndex);

        rm.writeClasses();
        rm.writeAttribute("data-sap-ui-rowindex", iRowIndex);
        var iTableRowHeight = oTable.getRowHeight();
        if (iTableRowHeight > 0) {
            rm.addStyle("height", iTableRowHeight + "px");
        }
        rm.writeStyles();

        oTable._getAccRenderExtension().writeAriaAttributesFor(rm, oTable, "TR", {index: iRowIndex});

        rm.write(">");
        var aCells = oRow.getCells();
        // render the row headers
        if (TableUtils.hasRowHeader(oTable) || aCells.length === 0) {
            rm.write("<td");
            oTable._getAccRenderExtension().writeAriaAttributesFor(rm, oTable, "ROWHEADER_TD", {
                rowSelected: !oRow._bHidden && oTable.isIndexSelected(oTable._getAbsoluteRowIndex(iRowIndex)), //see TableRenderer.renderRowHdrRow
                index: iRowIndex
            });
            rm.write("></td>");
        }

        for (var cell = 0, count = aCells.length; cell < count; cell++) {
            this.renderTableCell(rm, oTable, oRow, aCells[cell], cell, bFixedTable, iStartColumn, iEndColumn, aVisibleColumns);
        }
        if (!bFixedTable && bHasOnlyFixedColumns && aCells.length > 0) {
            rm.write("<td");
            rm.addClass("sapUiTableTDDummy");
            rm.writeClasses();
            rm.write(">");
            rm.write("</td>");
        }
        rm.write("</tr>");
    };

    TableRenderer.renderTableCell = function(rm, oTable, oRow, oCell, iCellIndex, bFixedTable, iStartColumn, iEndColumn, aVisibleColumns) {
        var iColIndex = oCell.data("sap-ui-colindex");
        var oColumn = oTable.retrieveLeafColumns()[iColIndex];

        //HTMLCOMMONS-1559
        if (this.shouldRenderColumnAtIndex(oTable, oColumn, iColIndex, bFixedTable) && iStartColumn <= iColIndex && iEndColumn > iColIndex) {
            rm.write("<td");
            var sId = oRow.getId() + "-col" + iCellIndex;
            rm.writeAttribute("id", sId);
            rm.writeAttribute("tabindex", "-1");

            var sColumnId = oColumn.getId();
            if (sColumnId) {
                rm.writeAttribute('data-sap-ui-colid', sColumnId);
            }

            var bIsFirstColumn = aVisibleColumns.length > 0 && aVisibleColumns[0] === oColumn;

            oTable._getAccRenderExtension().writeAriaAttributesFor(rm, oTable, "DATACELL", {
                index: iColIndex,
                column: oColumn,
                row: oRow,
                fixed: bFixedTable,
                firstCol: bIsFirstColumn
            });

            var sColHAlign;
            if (oColumn && oColumn.getHAlign) {
                sColHAlign = oColumn.getHAlign();
            }
            var sHAlign = Renderer.getTextAlign(sColHAlign, oCell && oCell.getTextDirection && oCell.getTextDirection());
            if (sHAlign) {
                rm.addStyle("text-align", sHAlign);
            }
            rm.writeStyles();
            rm.addClass("sapUiTableTd");
            if (bIsFirstColumn) {
                rm.addClass("sapUiTableTdFirst");
            }
            // grouping support to show/hide values of grouped columns
            if (oColumn.getGrouped && oColumn.getGrouped()) {
                rm.addClass("sapUiTableTdGroup");
            }

            var oBinding = oTable.getBinding("rows");
            if (oBinding && oColumn.getLeadingProperty && oBinding.isMeasure(oColumn.getLeadingProperty())) {
                // for AnalyticalTable
                rm.addClass("sapUiTableMeasureCell");
            }

            rm.writeClasses();
            rm.write("><div");
            rm.addClass("sapUiTableCell");

            rm.writeClasses();

            if (oTable.bSetMaxHeightForTableCellInVisibleRowCountAutoMode !== false) {
                if (oTable.getRowHeight() && oTable.getVisibleRowCountMode() === VisibleRowCountMode.Auto) {
                    rm.addStyle("max-height", oTable.getRowHeight() + "px");
                }
            }

            rm.writeStyles();

            rm.write(">");
            this.renderTableCellControl(rm, oTable, oCell, bIsFirstColumn);
            rm.write("</div></td>");
        }
    };

    TableRenderer.renderTableCellControl = function(rm, oTable, oCell, bIsFirstColumn) {
        if (TableUtils.Grouping.isTreeMode(oTable) && bIsFirstColumn) {
            rm.write("<span class='sapUiTableTreeIcon' tabindex='-1'");
            oTable._getAccRenderExtension().writeAriaAttributesFor(rm, oTable, "TREEICON", {row: oCell.getParent()});
            rm.write(">&nbsp;</span>");
        }
        rm.renderControl(oCell);
    };

    TableRenderer.renderVSb = function(rm, oTable) {
        rm.write("<div");
        rm.addClass("sapUiTableVSb");
        rm.writeClasses();
        rm.writeAttribute("id", oTable.getId() + "-vsb");
        rm.write(">");

        rm.write("<div");
        rm.writeAttribute("id", oTable.getId() + "-vsb-content");
        rm.addClass("sapUiTableVSbContent");
        rm.writeClasses();
        rm.write(">");
        rm.write("</div>");
        rm.write("</div>");
    };

    TableRenderer.renderHSb = function(rm, oTable) {
        rm.write("<div");
        rm.addClass("sapUiTableHSb");
        rm.writeClasses();
        rm.writeAttribute("id", oTable.getId() + "-hsb");
        rm.writeAttribute("tabindex", "-1"); // Avoid focusing in Firefox
        rm.write(">");
        rm.write("<div");
        rm.writeAttribute("id", oTable.getId() + "-hsb-content");
        rm.addClass("sapUiTableHSbContent");
        rm.writeClasses();
        rm.write(">");
        rm.write("</div>");
        rm.write("</div>");
    };


    // =============================================================================
    // HELPER FUNCTIONALITY
    // =============================================================================

    /**
     * Renders an empty area with tabindex=0 and the given class and id.
     * @private
     */
    TableRenderer.renderTabElement = function(rm, sClass) {
        rm.write("<div");
        if (sClass) {
            rm.addClass(sClass);
            rm.writeClasses();
        }
        rm.writeAttribute("tabindex", "0");
        rm.write("></div>");
    };


    // hook function for render
    // called while rendering div with .sapUiTableCnt
    TableRenderer.tableCntHook = function(rm, oTable) {
        // S1164865: to help add padding on the right when there's a table options
        // without the vertical scrollbar
        if (oTable.getShowTableOptions()){
            rm.addClass("sasUiTableOptions");
        }
    };

    // hook function for render
    // called before rendering the Column Header
    TableRenderer.preRenderColHdrHook = function(rm, oTable) {
        //HTMLCOMMONS-4581: New header feature allowing secondary label and control space per column
        if (oTable.shouldShowSummaryHeader()) {
            this.renderSummaryColHdr(rm, oTable);    //summary header is not included in ColHdr for more clean management
        }
    };

    // hook function for render
    // called after rendering Table during a11y span section
    TableRenderer.postRenderTableA11yHook = function(rm, oTable) {
        if (oTable._getAccExtension().getAccMode()) {
            //S1144475: adding description for table to announce visible rows vs total
            rm.write("<span");
            rm.writeAttribute("id", oTable.getId() + "-ariasummary");
            rm.addClass("sasInvisibleAriaText");
            rm.writeClasses();
            rm.write(">");
            rm.write(oTable._getARIADescription());
            rm.write("</span>");
        }
    };

    // hook function for renderTableCCnt
    TableRenderer.renderNoData = function(rm, oTable) {
        //FIXID S1141541: Need to show prompt message in Empty Table situation, also remove vertical scrollbar
        var oMsg = (TableUtils.getVisibleColumnCount(oTable) === 0) ? oTable.getAllColumnsHidden() : oTable.getNoData();
        if (oMsg && oMsg instanceof Control) {
            rm.renderControl(oMsg);
        } else {
            rm.write("<span");
            rm.addClass("sapUiTableCtrlEmptyMsg");
            rm.writeClasses();
            rm.write(">");

            var sText = null;
            if (TableUtils.getVisibleColumnCount(oTable) === 0) {
                if (typeof oMsg === "string" || oMsg instanceof String) {
                    sText = oMsg;
                } else {
                    sText = oTable.oResBundle.getText("TBL_ALL_COLUMNS_HIDDEN.txt");
                }
            } else {
                if (typeof oMsg === "string" || oMsg instanceof String) {
                    sText = oMsg;
                } else {
                    sText = oTable.oResBundle.getText("TBL_NO_DATA.txt");
                }
            }

            rm.writeEscaped(sText);
            rm.write("</span>");
        }
    };

    // hook function for renderColHdr
    // determine inline style height for .sapUiTableColHdrCnt
    TableRenderer.determineColHdrCntHeight = function(oTable) {
        return (oTable.getColumnHeaderHeight() * TableUtils.getHeaderRowCount(oTable)) + "px";
    };

    // called before rendering .sapUiTableColRowHdr
    // hook function for renderColHdr
    TableRenderer.renderOptionsMenuButton = function(rm, oTable) {
        if (oTable.getShowTableOptions()){
            var optionsButton = oTable.getAggregation("tableOptionsMenuButton");
            if (optionsButton) {
                rm.renderControl(optionsButton);
            }
        }
    };

    // Hook function for rendering custom styles for scrollable column header container
    TableRenderer.writeCustomStylesForScrollableColHdr = function(rm, oTable, renderState) {
        var iFixedColumnCount = renderState.iFixedColumnCount;
        var aCols = renderState.aCols;

        if (!oTable._hasSomePercentageWidthColumn()) {
            var iScrollableColumnWidthTotal = oTable._getColumnsWidth(iFixedColumnCount, aCols.length);
            rm.addStyle('min-width', iScrollableColumnWidthTotal + 'px');
        }
    };

    // Hook function to render only the columns needed.
    // HTMLCOMMONS-1559
    TableRenderer.getColumnRenderRange = function(oTable, bFixedTable) {
        var iStartColumn, iEndColumn;
        if (bFixedTable) {
            iStartColumn = 0;
            iEndColumn = oTable.getTotalFrozenColumnCount();
        } else {
            iStartColumn = oTable.getTotalFrozenColumnCount();
            iEndColumn = oTable.retrieveLeafColumns().length;
        }
        return {
            start: iStartColumn,
            end: iEndColumn
        };
    };

    TableRenderer.shouldRenderColumnAtIndex = function(oTable, oColumn, iIndex) {
        return oColumn && oColumn.shouldRender();
    };

    // whether to set an absolute min-width when there is a fixed column count
    // hook function for renderColHdr
    TableRenderer.shouldSetMinWidthOnColHdrFixed = function() {
        //Jira story, HTMLCOMMONS-4427 substask of HTMLCOMMONS-4016, resizing column animation story
        //col-resize change, cannot set min width on fixed column or resizing does not work when animation is on
        return false;
    };

    // add css class for each .sapUiTableColCell
    // HTMLCOMMONS-4582
    TableRenderer.addCustomTableColCellCssClass = function(rm, oColumn) {
        // HTMLCOMMONS-4582
        if (jQuery.isFunction(oColumn.getHeaderEditable) && oColumn.getHeaderEditable()) {
            rm.addClass("sasUiTableColEdited");
        }
    };

    // add any custom bits to the column header cell prior to rendering the header control itself
    // HTMLCOMMONS-6551
    TableRenderer.renderColPreControlRenderHook = function(rm, oColumn) {
        rm.write("<div id=\"" + oColumn.getId() + "-icons\" class=\"sapUiTableColIcons\"></div>");
    };

    // hook function for renderRowHdrRow
    TableRenderer.renderRowHdrRowSelection = function(rm, oTable, oRow, iRowIndex) {
        if (oTable._useSelectionControls() === true) {
            var oSelectionControl = jQuery.isFunction(oRow.getSelectionControl) && oRow.getSelectionControl();
            if (!!oSelectionControl && TableUtils.getVisibleColumnCount(oTable) > 0) {
                rm.renderControl(oSelectionControl);
            }
        }
    };

    // hook function for renderTableControlCnt
    // add custom dom attributes to .sapUiTableCtrl
    TableRenderer.addCustomControlCntAttributes = function(rm, oTable, bFixedTable, iStartColumn, iEndColumn) {
        //S1144475: adding aria described-by attribute
        rm.writeAttribute("aria-describedby", oTable.getId() + "-ariasummary");
    };

    // hook function for renderTableControlCnt
    // return width attribute for .sapUiTableCtrl
    // TODO: is this getting invoked?
    TableRenderer.determineControlCntWidth = function(oTable, iStartColumn, iEndColumn) {
        //modified the way the width is calculated to be very small for
        //resizing columns animation Jira story, HTMLCOMMONS-4427 substask of HTMLCOMMONS-4016
        return ((iEndColumn - iStartColumn)*oTable._iColMinWidth) + "px";
    };

    /**
     * Hook-point provided at openui-sdk layer
     * for rendering a custom select-all control.
     * Render the "selectAllControl" aggregation
     * @param {object} rm RenderManager being used
     * @param {oTable} oTable The Table control being rendered
     */
    TableRenderer.renderSelectAllControl = function(rm, oTable) {
        rm.renderControl(oTable.getSelectAllControl());
    };

    // hook function for renderColHdr
    // determine inline style height for .sapUiTableColHdrCnt
    TableRenderer.determineColHdrCntHeight = function(oTable) {
        var iColumnHeaderCount = TableUtils.getHeaderRowCount(oTable);
        return (oTable.getColumnHeaderHeight() * iColumnHeaderCount) + "px";
    };

    // hook function for renderColHdr
    // whether to render a column when it has a header span greater than 1
    // TODO: is this getting invoked?
    TableRenderer.shouldRenderColumnWithSpanGT1 = function() {
        return false;
    };

    TableRenderer.renderSummaryColHdr = function(rm, oTable) {
        var bInvisible = !oTable.getShowSummaryHeader();

        rm.write("<div");
        rm.addClass("sapUiTableColHdrCnt");       //TODO this makes header sizing calculations go bad
        rm.addClass("sasUiTableSumColHdrCnt");
        rm.writeClasses();
        if (oTable.getSummaryHeaderHeight() > 0) {
            rm.addStyle("height", oTable.getSummaryHeaderHeight()*2 + "px");    //label row + control row
        }

        if (bInvisible) {
            rm.addStyle("display", "none");
        }

        rm.writeStyles();
        rm.write(">");

        var aCols = oTable.retrieveLeafColumns();

        if (oTable.getTotalFrozenColumnCount() > 0) {
            rm.write("<div");
            rm.addClass("sapUiTableColHdrFixed");
            rm.writeClasses();
            rm.write(">");

            this._renderSummaryHeader(rm, oTable, aCols, 0, oTable.getTotalFrozenColumnCount(), true);

            rm.write("</div>");
        }

        rm.write("<div");
        rm.addClass("sapUiTableColHdrScr");
        rm.writeClasses();
        if (oTable.getTotalFrozenColumnCount() > 0) {
            if (oTable._bRtlMode) {
                rm.addStyle("margin-right", "0");
            } else {
                rm.addStyle("margin-left", "0");
            }
            rm.writeStyles();
        }
        rm.write(">");

        this._renderSummaryHeader(rm, oTable, aCols, oTable.getTotalFrozenColumnCount(), aCols.length, false);

        rm.write("</div>");

        rm.write("</div>");

    };

    TableRenderer._renderSummaryHeader = function(rm, oTable, aCols, iStart, iLength, bFixed) {
        var aCols = oTable.retrieveLeafColumns();

        // SUMMARY ROW HEADER LABELS
        rm.write("<div");
        rm.addClass("sapUiTableColHdr");
        rm.writeClasses();
        rm.write(">");

        this.renderTableSectionStart(rm, oTable, bFixed, iStart, iLength);

        var aVisibleColumns = oTable._getVisibleColumns();
        var sWidth;
        jQuery.each(aVisibleColumns, function(iIndex, oColumn) {
            if (iIndex >= iStart && iIndex < iLength) {
                rm.write("<th");
                rm.writeAttribute("data-sap-ui-headcolindex", iIndex);
                rm.writeAttribute("data-sap-ui-colid", oColumn.getId());
                if (oColumn.getWidth) {
                    sWidth = oColumn.getWidth();
                    if (bFixed) {
                        sWidth = sWidth || "160px";
                    }
                    rm.addStyle("width", sWidth);
                    rm.writeStyles();
                }
                rm.write("></th>");
            }
        });

        rm.write("</tr></thead>");
        rm.write("<tbody>");

        var iSpan = 1;
        var aHeaderSpan = null;
        for (var i = iStart; i < iLength; i++) {
            if (aCols[i] && aCols[i].shouldRender()) {
                if (iSpan <= 1) {
                    this.renderSummaryCol(rm, oTable, aCols[i], i, false, bFixed);
                    if (aCols[i].getHeaderSpan) {
                        aHeaderSpan = aCols[i].getHeaderSpan();
                    }
                    iSpan = 1 + (jQuery.isArray(aHeaderSpan) ? aHeaderSpan[0] : aHeaderSpan);
                }
                iSpan--;
            }
        }

        this.renderTableSectionEnd(rm);
        rm.write("</div>");

        // SUMMARY ROW HEADER CONTROLS
        rm.write("<div");
        rm.addClass("sapUiTableColHdr sasUiTableColHdr");
        rm.writeClasses();
        if (bFixed){
            rm.addStyle("min-width", oTable._getColumnsWidth(0, oTable.getTotalFrozenColumnCount()) + "px");
        }
        rm.writeStyles();
        rm.write(">");

        this.renderTableSectionStart(rm, oTable, bFixed, iStart, iLength);

        jQuery.each(aVisibleColumns, function(iIndex, oColumn) {
            if (iIndex >= iStart && iIndex < iLength) {
                rm.write("<th");
                rm.writeAttribute("data-sap-ui-headcolindex", iIndex);
                rm.writeAttribute("data-sap-ui-colid", oColumn.getId());
                if (oColumn.getWidth) {
                    sWidth = oColumn.getWidth();
                    if (bFixed) {
                        sWidth = sWidth || "160px";
                    }
                    rm.addStyle("width", sWidth);
                    rm.writeStyles();
                }
                rm.write("></th>");
            }
        });

        rm.write("</tr></thead>");
        rm.write("<tbody>");

        iSpan = 1;
        for (i = iStart; i < iLength; i++) {
            if (aCols[i] && aCols[i].shouldRender()) {
                if (iSpan <= 1) {
                    this.renderSummaryControl(rm, oTable, aCols[i], i);
                    if (aCols[i].getHeaderSpan) {
                        aHeaderSpan = aCols[i].getHeaderSpan();
                    }
                    iSpan = 1 + (jQuery.isArray(aHeaderSpan) ? aHeaderSpan[0] : aHeaderSpan);
                }
                iSpan--;
            }
        }

        this.renderTableSectionEnd(rm);
        rm.write("</div>");
    };

    TableRenderer.renderSummaryCol = function(rm, oTable, oColumn, iIndex, bInvisible, bFixed) {
        var sHeaderId = oColumn.getId(),
            oLabel;
        if (!!oColumn && jQuery.isFunction(oColumn.getSummaryLabel)) {
            oLabel = oColumn.getSummaryLabel();
        }
        if (!oLabel) {
            if (oColumn.getMultiLabels().length > 0) {
                oLabel = oColumn.getMultiLabels()[0];
            } else {
                oLabel = oColumn.getLabel();
            }
        }

        this.renderColStart(rm, oTable, oColumn, iIndex, sHeaderId, oLabel);

        rm.write("<div");
        // TODO: we need a writeElementData with suffix - it is another HTML element
        //       which belongs to the same column but it is not in one structure!
        rm.writeAttribute('id', oColumn.getId() + "-shead");
        rm.writeAttribute('data-sap-ui-colid', oColumn.getId());
        rm.writeAttribute("data-sap-ui-colindex", iIndex);
        if (oTable._getAccExtension().getAccMode()) {
            if (!!Device.browser.internet_explorer) {
                rm.writeAttribute("role", "columnheader");
            }
            // TODO: determine if the column has a column menu
            rm.writeAttribute("aria-haspopup", "true");
            rm.writeAttribute("tabindex", "-1");
        }

        rm.addClass("sapUiTableCol");
        rm.addClass("sasUiTableColSumHead");
        if (oTable.getTotalFrozenColumnCount() === (iIndex + 1)) {
            rm.addClass("sapUiTableColLastFixed");
        }
        rm.writeClasses();
        // rm.addStyle("width", oColumn.getWidth());
        if (oTable.getSummaryHeaderHeight() > 0) {
            rm.addStyle("height", oTable.getSummaryHeaderHeight() + "px");
        }
        if (bInvisible) {
            rm.addStyle("display", "none");
        }
        rm.writeStyles();
        var sTooltip = oColumn.getTooltip_AsString();
        if (sTooltip) {
            rm.writeAttributeEscaped("title", sTooltip);
        }
        rm.write("><div");
        rm.addClass("sapUiTableColCell");
        rm.writeClasses();
        var sColHAlign;
        if (oColumn && oColumn.getHAlign) {
            sColHAlign = oColumn.getHAlign();
        }
        var sHAlign = Renderer.getTextAlign(sColHAlign, oLabel && oLabel.getTextDirection && oLabel.getTextDirection());
        if (sHAlign) {
            rm.addStyle("text-align", sHAlign);
        }
        rm.writeStyles();
        rm.write(">");

        if (oLabel) {
            rm.renderControl(oLabel);
        }

        rm.write("</div></div>");
        this.renderColEnd(rm);
    };

    TableRenderer.renderSummaryControl = function(rm, oTable, oColumn, iIndex, bInvisible) {
        var sHeaderId = oColumn.getId(),
            oLabel;

        if (!!oColumn && jQuery.isFunction(oColumn.getSummaryControl)) {
            oLabel = oColumn.getSummaryControl();
        }

        this.renderColStart(rm, oTable, oColumn, iIndex, sHeaderId, oLabel);

        rm.write("<div");
        // TODO: we need a writeElementData with suffix - it is another HTML element
        //       which belongs to the same column but it is not in one structure!
        rm.writeAttribute('id', oColumn.getId() + "_scon");
        rm.writeAttribute('data-sap-ui-colid', oColumn.getId());
        rm.writeAttribute("data-sap-ui-colindex", iIndex);
        if (oTable._getAccExtension().getAccMode()) {
            if (!!Device.browser.internet_explorer) {
                rm.writeAttribute("role", "columnheader");
            }
            // TODO: determine if the column has a column menu
            rm.writeAttribute("aria-haspopup", "true");
            rm.writeAttribute("tabindex", "-1");
        }
        rm.addClass("sapUiTableCol");
        rm.addClass("sasUiTableColSumControl");
        if (oTable.getTotalFrozenColumnCount() === (iIndex + 1)) {
            rm.addClass("sapUiTableColLastFixed");
        }
        rm.writeClasses();
        if (bInvisible) {
            rm.addStyle("display", "none");
        }
        rm.writeStyles();
        var sTooltip = oColumn.getTooltip_AsString();
        if (sTooltip) {
            rm.writeAttributeEscaped("title", sTooltip);
        }
        rm.write("><div");
        rm.addClass("sapUiTableColCell");
        rm.writeClasses();
        var sColHAlign;
        if (oColumn && oColumn.getHAlign) {
            sColHAlign = oColumn.getHAlign();
        }
        var sHAlign = Renderer.getTextAlign(sColHAlign, oLabel && oLabel.getTextDirection && oLabel.getTextDirection());
        if (sHAlign) {
            rm.addStyle("text-align", sHAlign);
        }
        rm.writeStyles();
        rm.write(">");

        if (oLabel) {
            rm.renderControl(oLabel);
        } else {
            rm.write("<div");   //w/o a control here, we need a placeholder to take up room
            rm.addStyle("height", "100%");
            rm.writeStyles();
            rm.write(">&nbsp;</div>");
        }

        rm.write("</div></div>");
        this.renderColEnd(rm);
    };

    return TableRenderer;

}, /* bExport= */ true);
