/*!
 * (c) 2014-2018, SAS Institute Inc.
 */

// Provides helper sas.hc.ui.table.TableKeyboardExtension.
sap.ui.define(['jquery.sap.global', './TableExtension', './ItemNavigation', './TableUtils', './TableKeyboardDelegate', './library', './Row'],
    function(jQuery, TableExtension, ItemNavigation, TableUtils, TableKeyboardDelegate, library, Row) {
        "use strict";

        // shortcuts
        var SelectionBehavior = library.SelectionBehavior,
            SharedDomRef = library.SharedDomRef;


        function keyboardHandlers(oTable) {
            var
            // these events will be directly handled by ItemNavigation
                aHandledByItemNav = [
                    'onfocusin',
                    'onsapfocusleave',
                    'onmousedown',
                    'onsapnextmodifiers',
                    'onsappreviousmodifiers',
                    'onsapkeyup'
                ],

                // define callback functions for sap keyboard events
                oHandlers = {
                    onsapselect: function() {
                        return new Promise(function(resolve) {
                            oTable._bEventSapSelect = true;
                            resolve();
                        });
                    },

                    onsapselectmodifiers: function() {
                        return new Promise(function(resolve) {
                            oTable._bEventSapSelect = true;
                            resolve();
                        });
                    },

                    onsapspace: function(oEvent) {
                        return new Promise(function(resolve) {
                            resolve();
                        });
                    },

                    onsapescape: function(oEvent) {
                        return new Promise(function(resolve) {
                            resolve();
                        });
                    },

                    // Navigation Mode:
                    //   If focus is on header: jump to the next focusable control before the table
                    //   If focus in on content: jump to header for the current column
                    // Action Mode: switch back to navigation mode
                    onsaptabprevious: function(oEvent) {
                        return new Promise(function(resolve) {
                            var $table = oTable.$();
                            // if the target was the overlay or no content element, then...
                            if (oEvent.target === oTable.getDomRef("overlay") || oEvent.target === oTable.getDomRef("noDataCnt")) {
                                // move focus to the 'outer before' tab stop prior to browser's tab handling
                                oTable._getKeyboardExtension()._setSilentFocus($table.find(".sapUiTableOuterBefore"));
                                resolve();
                                return;
                            }

                            // if the target was a selection checkbox and there is no select all checkbox...
                            if (oEvent.target.getAttribute('role') === "checkbox" && oEvent.target.parentElement.classList.contains('sapUiTableRowHdr') && !oTable.getEnableSelectAll()) {
                                // move focus to the 'ctrl before' element and skip the div before the table control
                                oTable._getKeyboardExtension()._setSilentFocus($table.find(".sapUiTableCtrlBefore"));
                                resolve();
                                return;
                            }

                            var oInfo = TableUtils.getFocusedItemInfo(oTable);
                            var bNoData = TableUtils.isNoDataVisible(oTable);
                            var oSapUiTableCCnt = $table.find('.sapUiTableCCnt')[0];
                            var bFocusFromTableContent = jQuery.contains(oSapUiTableCCnt, oEvent.target);

                            if (bFocusFromTableContent && oTable.getColumnHeaderVisible()) {
                            // Focus comes from table content. Focus the column header which corresponds to the
                            // selected column (column index)
                                oTable.focusPreviousFocusableHeader(oEvent, oInfo.cellInRow, true);
                            } else if (oInfo.domRef === oEvent.target && jQuery.sap.containsOrEquals(oSapUiTableCCnt, oEvent.target) ||
                            (!oTable.getColumnHeaderVisible() && bNoData && bFocusFromTableContent)) {
                            // in case of having the focus in the row or column header we do not need to
                            // place the focus to the div before the table control because there we do
                            // not need to skip the table controls anymore.
                                oTable._getKeyboardExtension()._setSilentFocus($table.find(".sapUiTableCtrlBefore"));
                            }
                            resolve();
                        });
                    },

                    // Navigation Mode:
                    //   If focus is on header: jump to the first data column of the focused column header
                    //   If focus in on content: jump to the next focusable control after the table
                    // Action Mode: switch back to navigation mode
                    onsaptabnext: function(oEvent) {
                        return new Promise(function(resolve) {
                            var $table = oTable.$();
                            if (oEvent.target === oTable.getDomRef("overlay")) {
                                oTable._getKeyboardExtension()._setSilentFocus($table.find(".sapUiTableOuterAfter"));
                                resolve();
                                return;
                            }

                            var oInfo = TableUtils.getFocusedItemInfo(oTable);
                            var bContainsColHdrCnt = jQuery.contains($table.find('.sapUiTableColHdrCnt')[0], oEvent.target);
                            var bNoData = TableUtils.isNoDataVisible(oTable);

                            if (bContainsColHdrCnt && !bNoData) {
                                oTable._getKeyboardExtension()._restoreFocusOnLastFocusedDataCell(oTable, oEvent);

                                // if focus is on the select all checkbox and tab next was pressed
                                // check to see if focus is still on the select all checkbox after
                                // calling _restoreFocusOnLastFocusedDataCell
                                // if the focused domRef is still the select all checkbox
                                // that means the last focused row is not focusable, so set focus to the
                                // first row of the table - S1458376
                                oInfo = TableUtils.getFocusedItemInfo(oTable);
                                var bIsSelectAll = jQuery(oEvent.target).hasClass("sasSelectAllControl");
                                if (bIsSelectAll && oEvent.target === oInfo.domRef) {
                                    TableUtils.focusItem(oTable, oInfo.columnCount, oEvent);
                                }

                                // prevent focus from additionally going to tabbable controls within cells
                                oEvent.stopImmediatePropagation(true);
                                oEvent.preventDefault();
                            } else if (oInfo.domRef === oEvent.target || (bNoData && bContainsColHdrCnt)) {
                                oTable._getKeyboardExtension()._setSilentFocus($table.find(".sapUiTableCtrlAfter"));
                            }
                            resolve();
                        });
                    },

                    onsappagedown: function(oEvent) {
                        return new Promise(function(resolve) {
                        // gather information about current state
                            var oActiveRow = oTable._rowFromRef(oEvent.target),
                                bIsBottomRowOfCurrentPage = oActiveRow && oActiveRow.isLastFullyVisible() === true;

                            // if active cell is last visible row
                            // load next page of rows (async), and put focus on last visible row
                            if (bIsBottomRowOfCurrentPage === true) {
                                oTable.displayNextPage(function() {
                                    oTable._getKeyboardExtension()._putFocusAtBottomOfCurrentPage(oTable, oEvent).then(function() {
                                        resolve();
                                    });
                                });
                                return;
                            }

                            // otherwise, just put focus on last visible row
                            // this path includes row-headers, col-headers, select-all
                            oTable._getKeyboardExtension()._putFocusAtBottomOfCurrentPage(oTable, oEvent).then(function() {
                                resolve();
                            });
                        });
                    },

                    onsappagedownmodifiers: function(oEvent) {
                        return new Promise(function(resolve) {
                            if (oTable._getKeyboardExtension().isInActionMode() && oEvent.altKey) {
                                resolve();
                                return;
                            }

                            var oInfo = TableUtils.getFocusedItemInfo(oTable);
                            var bRowHeader = (oTable.getSelectionBehavior() !== SelectionBehavior.RowOnly);

                            var iCol = oInfo.columnCount;
                            var iNewCol;
                            if (iCol === 0 && bRowHeader) {
                                iNewCol = 1;
                            } else {
                                var iVisibleColumns = oTable._aVisibleColumns.length;
                                var iMaxIndex = oTable._getVisibleColumns().length;
                                if (!bRowHeader) {
                                    iMaxIndex--;
                                }
                                if (iVisibleColumns === 0) {
                                    iNewCol = iMaxIndex;
                                } else {
                                    iNewCol = Math.min(iMaxIndex, iCol + iVisibleColumns);
                                }
                            }
                            oTable.focusNextFocusableHeader(oEvent, oInfo.cell - (iCol - iNewCol), true);
                            resolve();
                        });
                    },

                    onsappageup: function(oEvent) {
                        return new Promise(function(resolve) {
                        // gather information about current state
                            var oActiveRow = oTable._rowFromRef(oEvent.target),
                                bIsVeryTopRow = oActiveRow && oActiveRow.getIndex() === 0,
                                bIsTopRowOfCurrentPage = oActiveRow && (oActiveRow.isFirstVisible() === true || oActiveRow.isFirstFullyVisible() === true),
                                bIsColHeader = jQuery.contains(oTable.$().find(".sapUiTableColHdrCnt")[0], oEvent.target),
                                bIsSelectAll = jQuery(oEvent.target).hasClass("sasSelectAllControl");

                            // if col header or select-all, do nothing
                            if (bIsColHeader === true || bIsSelectAll === true) {
                                resolve();
                                return;
                            }

                            // if active cell is very top row, put focus on col header
                            if (bIsVeryTopRow === true) {
                                oTable.focusPreviousFocusableHeader(oEvent, TableUtils.getFocusedItemInfo(oTable).cellInRow, true);
                                resolve();
                                return;
                            }

                            // if active cell is first visible row in current page,
                            // load previous page of rows (async), and put focus on first visible row
                            if (bIsTopRowOfCurrentPage === true) {
                                oTable.displayPreviousPage(function() {
                                    oTable._getKeyboardExtension()._putFocusAtTopOfCurrentPage(oTable, oEvent).then(function() {
                                        resolve();
                                    });
                                });
                                return;
                            }

                            // otherwise, just put focus on first visible row
                            oTable._getKeyboardExtension()._putFocusAtTopOfCurrentPage(oTable, oEvent).then(function() {
                                resolve();
                            });
                        });
                    },

                    onsappageupmodifiers: function(oEvent) {
                        return new Promise(function(resolve) {
                            if (!oTable._getKeyboardExtension().isInActionMode() && oEvent.altKey) {
                                var oInfo = TableUtils.getFocusedItemInfo(oTable);
                                var bRowHeader = (oTable.getSelectionBehavior() !== SelectionBehavior.RowOnly);

                                var iCol = oInfo.columnCount;
                                if (iCol > 0) {
                                    var iNewCol;
                                    if (iCol === 1 && bRowHeader) {
                                        iNewCol = 0;
                                    } else {
                                        var iVisibleColumns = oTable._aVisibleColumns.length;
                                        if (iVisibleColumns === 0) {
                                            if (bRowHeader) {
                                                iNewCol = 1;
                                            } else {
                                                iNewCol = 0;
                                            }
                                        } else {
                                            var iMin = 1;
                                            if (!bRowHeader) {
                                                iMin = 0;
                                            }
                                            iNewCol = Math.max(iMin, iCol - iVisibleColumns);
                                        }
                                    }
                                    oTable.focusPreviousFocusableHeader(oEvent, oInfo.cell - (iCol - iNewCol), true);
                                }
                            }
                            resolve();
                        });
                    },

                    onsaphome: function(oEvent) {
                        return new Promise(function(resolve) {
                            var bIsRowOnly = (oTable.getSelectionBehavior() === SelectionBehavior.RowOnly);

                            // If focus is on a group header, do nothing.
                            var bIsGroupCell = jQuery(oEvent.target).parents(".sapUiTableGroupHeader").length > 0;
                            if (bIsGroupCell) {
                                resolve();
                                return;
                            }

                            var oInfo = TableUtils.getFocusedItemInfo(oTable);
                            var iFocusedIndex = oInfo.cell;
                            var iSelectedCellInRow = oInfo.cellInRow;
                            var bIsWideTable = TableUtils.isWideTable(oTable);

                            var offset = 0;
                            var iIndex;
                            var iFirstFocusableCellInRow = 1;
                            var iFrozenColumnCount = oTable.getTotalFrozenColumnCount();

                            if (TableUtils.hasRowHeader(oTable) === true) {
                                offset = 1;
                            }

                            if (iSelectedCellInRow > iFrozenColumnCount + offset) {
                            // If there is a frozen column, stop right of it.
                                iIndex = iFocusedIndex - iSelectedCellInRow + iFrozenColumnCount + offset;
                                oTable.focusPreviousFocusableHeader(oEvent, iIndex, true);
                            } else if (!bIsRowOnly) {
                                if (iSelectedCellInRow > iFirstFocusableCellInRow) {
                                // if focus is anywhere in the row, move focus to the first column cell.
                                    iIndex = iFocusedIndex - iSelectedCellInRow + offset;
                                    oTable.focusPreviousFocusableHeader(oEvent, iIndex, true);
                                } else if (iSelectedCellInRow === iFirstFocusableCellInRow && !bIsWideTable) {
                                // if focus is on first cell, move focus to row header.
                                    iIndex = iFocusedIndex - iFirstFocusableCellInRow;
                                    oTable.focusPreviousFocusableHeader(oEvent, iIndex, true);
                                    resolve();
                                    return;
                                } else if (!bIsWideTable) {
                                // If focus is on selection cell, do nothing.
                                    resolve();
                                    return;
                                }
                            }

                            if (bIsWideTable) {
                            // wide-table handling
                                oWideTableHandlers._home(oTable, oEvent).then(function() {
                                    resolve();
                                });
                                return;
                            }

                            resolve();
                        });
                    },

                    onsaphomemodifiers: function(oEvent) {
                        return new Promise(function(resolve) {
                        // return if ctrl not pressed
                            if (!oEvent.metaKey && !oEvent.ctrlKey) {
                                resolve();
                                return;
                            }

                            var oActiveRow = oTable._rowFromRef(oEvent.target),
                                oColumn = oTable._colFromRef(TableUtils.getFocusedItemInfo(oTable).domRef),
                                bIsColHeader = jQuery.contains(oTable.$().find(".sapUiTableColHdrCnt")[0], oEvent.target),
                                bIsRowHeader = jQuery(oEvent.target).closest(".sapUiTableRowHdr").length === 1,
                                bIsSelectAll = jQuery(oEvent.target).hasClass("sasSelectAllControl"),
                                bHasSelectAll = TableUtils.hasRowHeader(oTable) && oTable.getSelectAllControl() && oTable.getSelectAllControl().$().is(":visible") === true,
                                bCell00 = oActiveRow && oActiveRow.getIndex() === 0 && oColumn && oColumn.getIndex() === 0;

                            // no-op if on select-all
                            if (bIsSelectAll === true) {
                                resolve();
                                return;
                            }

                            // if focus is on a col header
                            // then put focus on select-all if present, or first col header
                            if (bIsColHeader === true) {
                                if (bHasSelectAll === true) {
                                    oTable._getKeyboardExtension()._putFocusOnSelectAll(oTable, oEvent).then(function() {
                                        resolve();
                                    });
                                } else {
                                    if (oTable.retrieveLeafColumns().length > 0) {

                                        if (TableUtils.isWideTable(oTable)) {
                                            oTable.makeFirstVisibleColumn(0);
                                        }

                                        oTable._getKeyboardExtension()._safeFocus(oTable, oTable.retrieveLeafColumns()[0].getDomRef(), oEvent).then(function() {
                                            resolve();
                                        });
                                    } else {
                                        resolve();
                                    }
                                }
                                return;
                            }

                            // if on row header, scroll up to first page,
                            // then put focus on select-all if present, or first row header
                            if (bIsRowHeader === true) {
                                if (bHasSelectAll === true) {
                                    oTable._showFirstPage(function() {
                                        oTable._getKeyboardExtension()._putFocusOnSelectAll(oTable, oEvent).then(function() {
                                            resolve();
                                        });
                                    });
                                } else {
                                    oTable._showFirstPage(function() {
                                        oTable._getKeyboardExtension()._putFocusAtTopOfCurrentPage(oTable, oEvent).then(function() {
                                            resolve();
                                        });
                                    });
                                }
                                return;
                            }

                            // if on very first cell on very first row, put focus on either select-all (if present) or col header
                            if (bCell00 === true) {
                                if (bHasSelectAll === true) {
                                    oTable._getKeyboardExtension()._putFocusOnSelectAll(oTable, oEvent).then(function() {
                                        resolve();
                                    });
                                    return;
                                } else {
                                    oTable.focusPreviousFocusableHeader(oEvent, TableUtils.getFocusedItemInfo(oTable).cellInRow, true);
                                    resolve();
                                }
                                return;
                            }

                            // otherwise, put focus on first row
                            // first, jump there and then put focus on correct cell
                            oTable._showFirstPage(function() {
                            // wide-table handling
                                if (TableUtils.isWideTable(oTable) === true) {
                                    oWideTableHandlers._homemodifiers(oTable, oEvent).then(function() {
                                        oTable._getKeyboardExtension()._putFocusAtTopLeftOfCurrentPage(oTable, oEvent).then(function() {
                                            resolve();
                                        });
                                    });
                                } else {
                                    oTable._getKeyboardExtension()._putFocusAtTopLeftOfCurrentPage(oTable, oEvent).then(function() {
                                        resolve();
                                    });
                                }
                            });
                        });
                    },

                    onsapend: function(oEvent) {
                        return new Promise(function(resolve) {
                        // If focus is on a group header, do nothing.
                            var bIsGroupCell = jQuery(oEvent.target).parents(".sapUiTableGroupHeader").length > 0;
                            if (bIsGroupCell) {
                                resolve();
                                return;
                            }

                            // If focus is on a selection cell, move focus to the first cell of the same row.
                            var oInfo = TableUtils.getFocusedItemInfo(oTable);
                            var iFocusedIndex = oInfo.cell;
                            var iSelectedCellInRow = oInfo.cellInRow;

                            var bIsRowOnly = TableUtils.hasRowHeader(oTable) === true;

                            var offset = 0;
                            if (!bIsRowOnly) {
                                offset = 1;
                            }

                            // Account for the situation where there are less visible columns than the given totalFrozenColumnCount.
                            // This can occur when hiding columns with the dual-selector after previously freezing columns
                            var iFrozenColumnCountAdjusted = Math.min(TableUtils.getVisibleColumnCount(oTable), oTable.getTotalFrozenColumnCount());

                            if (iSelectedCellInRow < oTable.getTotalFrozenColumnCount() - offset) {
                            // if there is a frozen column, stop left of it
                                TableUtils.focusItem(oTable, iFocusedIndex - iSelectedCellInRow + iFrozenColumnCountAdjusted - offset, null);
                                // TODO - currently, multi header tables do not support frozen columns
                                // in the event that we add support for frozen columns, this call will
                                // need to be changed to something like oTable.focusNextFocusableHeader
                                resolve();
                                return;
                            }

                            if (TableUtils.isWideTable(oTable) === true) {
                            // wide-table handling
                                oWideTableHandlers._end(oTable, oEvent).then(function() {
                                    resolve();
                                });
                            } else {
                                if (oTable._getItemNavigation()) {
                                    oTable._getItemNavigation()['onsapend'](oEvent);
                                }
                                resolve();
                            }
                        });
                    },

                    onsapendmodifiers: function(oEvent) {
                        return new Promise(function(resolve) {
                        // return if ctrl not pressed
                            if (!oEvent.metaKey && !oEvent.ctrlKey) {
                                resolve();
                                return;
                            }

                            // put focus on last row
                            // first, jump there and then put focus on correct cell
                            oTable._showLastPage(function() {
                            // wide-table handling
                                if (TableUtils.isWideTable(oTable) === true) {
                                    oWideTableHandlers._endmodifiers(oTable, oEvent).then(function() {
                                        oTable._getKeyboardExtension()._putFocusAtBottomRightOfCurrentPage(oTable, oEvent).then(function() {
                                            resolve();
                                        });
                                    });
                                } else {
                                    oTable._getKeyboardExtension()._putFocusAtBottomRightOfCurrentPage(oTable, oEvent).then(function() {
                                        resolve();
                                    });
                                }
                            });
                        });
                    },

                    onsapup: function(oEvent) {
                        return new Promise(function(resolve) {
                            var oFocusedRow = oTable._rowFromRef(oEvent.target),
                                oVSb = oTable.getDomRef(SharedDomRef.VerticalScrollBar),
                                bColumnGroups = oTable.hasColumnGroups();

                            if (!oFocusedRow) {
                                //check if we're in the column headers
                                if (oTable._getItemNavigation() && $(oEvent.target).closest(".sapUiTableColHdrCnt").length > 0) {
                                    oTable._getItemNavigation()['onsapprevious'](oEvent);
                                    if (bColumnGroups) {
                                        oTable.focusPreviousFocusableHeader(oEvent);
                                    }
                                }

                                resolve();
                                return;
                            }

                            if (oVSb && oFocusedRow && oFocusedRow.getIndex() === 1) {
                            // if on the second-to-last row from the very top
                            // prevent the situation where you could get stuck on the 2nd to last row
                                oVSb.scrollTop = 0;
                                oTable._showFirstPage(function() {
                                    oTable._getKeyboardExtension()._putFocusAtTopOfCurrentPage(oTable, oEvent).then(function() {
                                        resolve();
                                    });
                                });
                                return;
                            } else if (oVSb
                                && oFocusedRow
                                && oFocusedRow.getIndex() === 2
                                && !oTable.getRows()[1].isFullyVisible()) {

                                if (oTable._getItemNavigation()) {
                                    oTable._getItemNavigation()['onsapprevious'](oEvent);
                                }
                                oVSb.scrollTop = 0;
                                resolve();
                                return;

                            } else if (TableUtils.isFirstScrollableRow(oTable, oEvent.target) && oFocusedRow.getIndex() !== 0) {
                            // if on the top row of a page of rows
                            // and not the very first row
                                var iFocusedIndex = oFocusedRow.getIndex();
                                oTable._showPageStartingAt(oTable.getFirstVisibleRow() - 1, function() {
                                // after the refresh, put focus on the cell above the previously-focused cell
                                    var oCell, iColIndex,
                                        oRow = oTable._getRowByAbsoluteIndex(iFocusedIndex - 1);

                                    if (jQuery(oEvent.target).closest(".sapUiTableRowHdr").length === 1) {
                                    // row header
                                        oCell = oTable._rowHeaderFromIndex(oTable.indexOfRow(oRow));
                                    } else {
                                        iColIndex = oTable._colFromRef(oEvent.target).getIndex();
                                        oCell = oRow.getCells()[iColIndex].getDomRef();
                                    }

                                    oTable._getKeyboardExtension()._safeFocus(oTable, oCell, oEvent).then(function() {
                                        resolve();
                                    });
                                });
                                return;
                            } else {
                                if (oTable._getItemNavigation()) {
                                    oTable._getItemNavigation()['onsapprevious'](oEvent);
                                    if (bColumnGroups) {
                                        oTable.focusPreviousFocusableHeader(oEvent);
                                    }
                                }
                                resolve();
                                return;
                            }
                        });
                    },

                    onsapupmodifiers: function(oEvent) {
                        return new Promise(function(resolve) {
                            var oFocusedItem,
                                iFocusableIndex;
                            if (!oEvent.ctrlKey) {
                                resolve();
                                return;
                            }

                            // gather information about current state
                            var oActiveRow = oTable._rowFromRef(oEvent.target),
                                bIsColHeader = jQuery.contains(oTable.$().find(".sapUiTableColHdrCnt")[0], oEvent.target),
                                bIsVeryTopRow = oActiveRow && oActiveRow.getIndex() === 0;

                            // if on col header then no-op
                            if (bIsColHeader) {
                                resolve();
                                return;
                            }

                            // if on very first row, put focus on col header
                            if (bIsVeryTopRow === true) {
                                oFocusedItem = TableUtils.getFocusedItemInfo(oTable);
                                iFocusableIndex = oFocusedItem.cell - oFocusedItem.columnCount;
                                // for headers that span multiple rows, the header cell directly
                                // above the focused cell may not be focusable so keep trying
                                oTable.focusPreviousFocusableHeader(oEvent, iFocusableIndex, true);
                                resolve();
                                return;
                            }

                            // otherwise, put focus on first row
                            // first, jump there and then put focus on correct cell
                            oTable._showFirstPage(function() {
                                oTable._getKeyboardExtension()._putFocusAtTopOfCurrentPage(oTable, oEvent).then(function() {
                                    resolve();
                                });
                            });
                        });
                    },

                    onsapnext: function(oEvent) {
                        return oTable._getKeyboardExtension()._onsapnext(oTable, oEvent);
                    },

                    onsapprevious: function(oEvent) {
                        return oTable._getKeyboardExtension()._onsapprevious(oTable, oEvent);
                    },

                    onsapdown: function(oEvent) {
                        return new Promise(function(resolve) {
                            var oFocusedRow = oTable._rowFromRef(oEvent.target);
                            var bColumnGroups = oTable.hasColumnGroups();

                            if (!oFocusedRow) {
                                if (oTable._getItemNavigation()) {
                                    oTable._getItemNavigation()['onsapnext'](oEvent);
                                    if (bColumnGroups) {
                                        oTable.focusNextFocusableHeader(oEvent);
                                    }
                                }
                                resolve();
                                return;
                            }

                            if (TableUtils.isNoDataVisible(oTable)) {
                                var oInfo = TableUtils.getCellInfo(oEvent.target);
                                if (oInfo && (oInfo.type === TableUtils.CELLTYPES.COLUMNHEADER || oInfo.type === TableUtils.CELLTYPES.COLUMNROWHEADER)) {
                                    oInfo = TableUtils.getFocusedItemInfo(oTable);
                                    if (oInfo.row - TableUtils.getHeaderRowCount(oTable) <= 1) { // we are in the last column header row
                                    // just prevent the navigation to the table content
                                        oEvent.setMarked("sapUiTableSkipItemNavigation");
                                    }
                                }
                                resolve();
                                return;
                            }

                            // if currently on the second-to-last row (requesting to go to the very-last row)
                            if (oFocusedRow.getIndex() === (oTable._getRowCount() - 2)) {
                                oTable._showLastPage(function() {
                                    oTable._getKeyboardExtension()._putFocusAtBottomOfCurrentPage(oTable, oEvent).then(function() {
                                        resolve();
                                    });
                                });
                                return;
                            }

                            // if cursor is on (or past) the last fully visible row in the current page of rows,
                            // then put focus on bottom row and scroll down the current page by 1 row
                            if (TableUtils.isLastScrollableRow(oTable, oEvent.target)) {
                            // scroll down by 1 row unless at very end
                                if (oFocusedRow.getIndex() < (oTable._getRowCount() - 1)) {
                                    var iFocusedIndex = oFocusedRow.getIndex(),
                                        iFirstVisibleRowBeforeScroll = oTable.getFirstVisibleRow(),
                                        fnAfterShowPage = function() {
                                        // after the refresh, put focus on the cell below the previously-focused cell
                                            var iColIndex, oCell,
                                                oRow = oTable._getRowByAbsoluteIndex(iFocusedIndex + 1),
                                                iFirstVisibleRow = oTable.getFirstVisibleRow(),
                                                bRowIsFirstRow = oRow === oTable.getRows()[0],
                                                bRowIsNotFullyVisible = oRow && !oRow.isFullyVisible(),
                                                bScrollProgressMade = iFirstVisibleRowBeforeScroll < iFirstVisibleRow,
                                                bRowIsHeader = false;

                                            if (jQuery(oEvent.target).closest(".sapUiTableRowHdr").length === 1) {
                                            // row header
                                                oCell = oTable._rowHeaderFromIndex(oTable.indexOfRow(oRow));
                                                bRowIsHeader = true;
                                            } else {
                                                iColIndex = oTable._colFromRef(oEvent.target).getIndex();
                                                oCell = oRow.getCells()[iColIndex].getDomRef();
                                            }

                                            if (!bRowIsFirstRow && !bRowIsHeader && bRowIsNotFullyVisible && bScrollProgressMade) {
                                            // Scroll down an extra row
                                                iFirstVisibleRowBeforeScroll = iFirstVisibleRow;
                                                oTable._showPageStartingAt(iFirstVisibleRowBeforeScroll + 1, fnAfterShowPage);
                                                // Call resolve() in the next call to fAfterShowPage.
                                                return;
                                            }

                                            oTable._getKeyboardExtension()._safeFocus(oTable, oCell, oEvent).then(function() {
                                                resolve();
                                            });
                                        };
                                    oTable._showPageStartingAt(iFirstVisibleRowBeforeScroll + 1, fnAfterShowPage);
                                } else {
                                    resolve();
                                }
                            } else {
                                if (oTable._getItemNavigation()) {
                                    oTable._getItemNavigation()['onsapnext'](oEvent);
                                    // we are in the content rows, not the header, so no need to
                                    // find focusable header like when !oFocusedRow
                                }

                                resolve();
                            }
                        });
                    },

                    onsapdownmodifiers: function(oEvent) {
                        return new Promise(function(resolve) {
                            if (!oEvent.ctrlKey) {
                                resolve();
                                return;
                            }

                            var $Target = jQuery(oEvent.target);
                            if ($Target.hasClass('sapUiTableCol')) {
                                var iColIndex = parseInt($Target.attr('data-sap-ui-colindex'), 10);
                                if (isNaN(iColIndex)) {
                                    resolve();
                                    return;
                                }
                            }


                            // if on col header, put focus on first row
                            if (jQuery.contains(oTable.$().find(".sapUiTableColHdrCnt")[0], oEvent.target)) {
                                oTable._getKeyboardExtension()._putFocusAtTopOfCurrentPage(oTable, oEvent).then(function() {
                                    resolve();
                                });
                                return;
                            }

                            // otherwise, put focus on last row
                            // first, jump there and then put focus on correct cell
                            oTable._showLastPage(function() {
                                oTable._getKeyboardExtension()._putFocusAtBottomOfCurrentPage(oTable, oEvent).then(function() {
                                    resolve();
                                });
                            });
                        });
                    }
                }; // end of oHandlers

            // add events that are directly passed onto ItemNav for processing
            aHandledByItemNav.forEach(function(sEventName) {
                oHandlers[sEventName] = function(oEvent) {
                    return new Promise(function(resolve) {
                        var oItemNavigation = oTable._getItemNavigation();
                        if (oItemNavigation) {
                            oItemNavigation['on' + oEvent.type](oEvent);
                        }
                        resolve();
                    });
                };
            });

            return oHandlers;
        };

        // wide table has special handling for some keys
        var oWideTableHandlers = {
            _onsapprevious: function(oTable, oEvent) {
                return new Promise(function(resolve) {
                    var oActiveRow, oActiveColumn,
                        oCellInfo, // holds TableUtils.getCellInfo
                        oRenderRange, // holds _getColumnRangeToRender
                        oTargetCell; // this will be the resulting cell to get focus

                    // don't handle this keypress here
                    function doSuper() {
                        if (oTable._getItemNavigation()) {
                            oTable._getItemNavigation()['onsapprevious'](oEvent);
                        }
                    }

                    // only handle the "left" key here
                    if (oEvent.keyCode !== jQuery.sap.KeyCodes.ARROW_LEFT) {
                        doSuper();
                        resolve();
                        return;
                    }

                    oCellInfo = TableUtils.getCellInfo(oEvent.target) || {};

                    // on row-header, call super
                    if (oCellInfo.type === TableUtils.CELLTYPES.ROWHEADER) { // row header
                        doSuper();
                        resolve();
                        return;
                    }

                    // on select-all, call super
                    if (/-selectAll$/.test(oEvent.target.id)) {
                        doSuper();
                        resolve();
                        return;
                    }

                    // gather current state
                    oActiveRow = oTable._rowFromRef(oEvent.target);
                    oActiveColumn = oTable._colFromRef(oEvent.target);
                    oRenderRange = oTable._getColumnRangeToRender();

                    // we've not reached the start of the current page of cols,
                    // so no special handling needed, call super
                    if (oActiveColumn.getIndex() > (oRenderRange.start + 1)) {
                        doSuper();
                        resolve();
                        return;
                    }

                    // reached very beginning of table, call super
                    if (oActiveColumn.getIndex() === 0 || oRenderRange.start === 0) {
                        doSuper();
                        resolve();
                        return;
                    }

                    if (oActiveColumn.getIndex() < oTable.getTotalFrozenColumnCount()) {
                        doSuper();
                        resolve();
                        return;
                    }

                    // wait for makeFirstVisibleColumn side-effects to complete
                    TableUtils._attachAfterInvalidationOnce(oTable, function() {
                    // gather the new domref to be given focus
                        if (oCellInfo.type === TableUtils.CELLTYPES.COLUMNHEADER) {
                            oTargetCell = oTable.retrieveLeafColumns()[oActiveColumn.getIndex() - 1];
                        } else {
                            oTargetCell = oActiveRow.getCells()[oActiveColumn.getIndex() - 1];
                        }

                        // add additional delay to throttle keypresses
                        oTable._setManagedTimeout(function() {
                        // give focus to new cell
                            if (oTargetCell !== undefined) {
                                oTable._focusItemByDomRef(oTargetCell.getDomRef());
                            }
                            resolve();
                        }, 50);
                    });

                    // load new page of columns
                    oTable.makeFirstVisibleColumn(oRenderRange.start - 1, false);
                });
            },

            _home: function(oTable, oEvent) {
                return new Promise(function(resolve) {
                    var oActiveRow = oTable._rowFromRef(oEvent.target);
                    var oActiveColumn = oTable._colFromRef(oEvent.target);
                    // Assume that if no active column, we're in the row header.
                    var iActiveColumnIndex = oActiveColumn ? oActiveColumn.getIndex() : -1;

                    if (iActiveColumnIndex === -1) {
                        resolve();
                        return;
                    }

                    if (iActiveColumnIndex === 0) {
                    // Move to row header if it exists
                        if (TableUtils.hasRowHeader(oTable)) {
                            var oInfo = TableUtils.getFocusedItemInfo(oTable);
                            var iFocusedIndex = oInfo.cell;
                            // TODO - currently wide table does not support multi headers
                            // in the event that we add support, this call might need to
                            // change to something like oTable.focusPreviousFocusableHeader
                            TableUtils.focusItem(oTable, iFocusedIndex - 1, null);
                        }
                        resolve();
                        return;
                    }

                    var iNextActiveRowIndex = (oActiveRow) ? oActiveRow.$().attr("data-sap-ui-rowindex") : null;
                    var iNextActiveColumnIndex = 0;

                    // If we're in the frozen section or next to it, just move focus.
                    if (iActiveColumnIndex <= oTable.getTotalFrozenColumnCount()) {
                        var nextActiveDomRef = (oActiveRow)
                            ? oTable.getDomRef("rows-row" + iNextActiveRowIndex + "-col" + iNextActiveColumnIndex)
                            : oTable.retrieveLeafColumns()[iNextActiveColumnIndex].$()[0];
                        oTable._focusItemByDomRef(nextActiveDomRef);
                        resolve();
                        return;
                    }

                    iNextActiveColumnIndex = oTable.getTotalFrozenColumnCount();

                    TableUtils._attachAfterInvalidationOnce(oTable, function() {
                        var nextActiveDomRef = (oActiveRow)
                            ? oTable.getDomRef("rows-row" + iNextActiveRowIndex + "-col" + iNextActiveColumnIndex)
                            : oTable.retrieveLeafColumns()[iNextActiveColumnIndex].$()[0];
                        oTable._focusItemByDomRef(nextActiveDomRef);
                        resolve();
                    });

                    oTable.makeFirstVisibleColumn(iNextActiveColumnIndex, false);
                });
            },

            _onsapnext: function(oTable, oEvent) {
                return new Promise(function(resolve) {
                    var oActiveRow, oActiveColumn,
                        oCellInfo, // holds TableUtils.getCellInfo
                        oRenderRange, // holds _getColumnRangeToRender
                        oTargetCell; // this will be the resulting cell to get focus

                    // don't handle this keypress here
                    function doSuper() {
                        if (oTable._getItemNavigation()) {
                            oTable._getItemNavigation()['onsapnext'](oEvent);
                        }
                    }

                    // only handle the "right" key here
                    if (oEvent.keyCode !== jQuery.sap.KeyCodes.ARROW_RIGHT) {
                        doSuper();
                        resolve();
                        return;
                    }

                    oCellInfo = TableUtils.getCellInfo(oEvent.target) || {};

                    // on row-header, call super
                    if (oCellInfo.type === TableUtils.CELLTYPES.ROWHEADER) { // row header
                        doSuper();
                        resolve();
                        return;
                    }

                    // on select-all, call super
                    if (/-selectAll$/.test(oEvent.target.id)) {
                        doSuper();
                        resolve();
                        return;
                    }

                    // gather current state
                    oActiveRow = oTable._rowFromRef(oEvent.target);
                    oActiveColumn = oTable._colFromRef(oEvent.target);
                    oRenderRange = oTable._getColumnRangeToRender();

                    if (oTable.isFrozenColumn(oActiveColumn)) {
                        doSuper();
                        resolve();
                        return;
                    }

                    // we've not reached the end of the current page of cols,
                    // so no special handling needed, call super
                    if (oActiveColumn.getIndex() < (oRenderRange.end - 2)) {
                        doSuper();
                        resolve();
                        return;
                    }

                    // reached very end of table, call super
                    if (oActiveColumn.getIndex() >= (oTable.retrieveLeafColumns().length - 1)) {
                        doSuper();
                        resolve();
                        return;
                    }

                    // wait for makeFirstVisibleColumn side-effects to complete
                    TableUtils._attachAfterInvalidationOnce(oTable, function() {
                    // add additional delay to throttle keypresses
                        oTable._setManagedTimeout(function() {

                            var iNextActiveColumnIndex = oActiveColumn.getIndex() + 1,
                                oUpdatedRenderRange;

                            // gather the new domref to be given focus
                            if (oCellInfo.type === TableUtils.CELLTYPES.COLUMNHEADER) {
                                oTargetCell = oTable.retrieveLeafColumns()[iNextActiveColumnIndex];
                            } else {
                                oTargetCell = oActiveRow.getCells()[iNextActiveColumnIndex];
                            }

                            // give focus to new cell
                            if (oTargetCell !== undefined) {
                                oTable._focusItemByDomRef(oTargetCell.getDomRef());

                                oUpdatedRenderRange = oTable._getColumnRangeToRender();

                                if (iNextActiveColumnIndex === oUpdatedRenderRange.end && oUpdatedRenderRange.lastColIsClipped === true) {
                                    oTable.makeFirstVisibleColumn(oUpdatedRenderRange.start + 1);
                                }
                            }
                            resolve();
                        }, 50);
                    });

                    // load new page of columns
                    oTable.makeFirstVisibleColumn(oRenderRange.start + 1, false);
                });
            },

            _homemodifiers: function(oTable, oEvent) {
                return new Promise(function(resolve) {
                    var oActiveRow = oTable._rowFromRef(oEvent.target);
                    var iNextActiveRowIndex = (oActiveRow) ? oActiveRow.$().attr("data-sap-ui-rowindex") : null;
                    var iNextActiveColumnIndex = 0;

                    TableUtils._attachAfterInvalidationOnce(oTable, function() {
                        var nextActiveDomRef = (oActiveRow)
                            ? oTable.getDomRef("rows-row" + iNextActiveRowIndex + "-col" + iNextActiveColumnIndex)
                            : oTable.retrieveLeafColumns()[iNextActiveColumnIndex].$()[0];
                        oTable._focusItemByDomRef(nextActiveDomRef);
                        // need to delay a little bit more, as I think
                        // the table is invalidating multiple times
                        oTable._setManagedTimeout(function() {
                            resolve();
                        }, 200);
                    });

                    oTable.makeFirstVisibleColumn(iNextActiveColumnIndex, false);
                });
            },

            _end: function(oTable, oEvent) {
                return new Promise(function(resolve) {
                    var aVisibleColumns = oTable._getVisibleColumns();
                    if (aVisibleColumns.length === 0) {
                    // There are no visible columns, we don't need to do anything.
                        resolve();
                        return;
                    }

                    var oActiveRow = oTable._rowFromRef(oEvent.target);
                    var iNextActiveRowIndex = (oActiveRow) ? oActiveRow.$().attr("data-sap-ui-rowindex") : null;
                    var iNextActiveColumnIndex = aVisibleColumns.slice(-1)[0].getIndex();

                    TableUtils._attachAfterInvalidationOnce(oTable, function() {
                        oTable._setManagedTimeout(function() {
                            var nextActiveDomRef = (oActiveRow)
                                ? oTable.getDomRef("rows-row" + iNextActiveRowIndex + "-col" + iNextActiveColumnIndex)
                                : oTable.retrieveLeafColumns()[iNextActiveColumnIndex].$()[0];
                            oTable._focusItemByDomRef(nextActiveDomRef);
                            resolve();
                        }, 50);
                    });

                    oTable.makeFirstVisibleColumn(iNextActiveColumnIndex, false);
                });
            },

            _endmodifiers: function(oTable, oEvent) {
                return new Promise(function(resolve) {
                    var oActiveRow = oTable._rowFromRef(oEvent.target);
                    var iNextActiveRowIndex = (oActiveRow) ? oActiveRow.$().attr("data-sap-ui-rowindex") : null;
                    var iNextActiveColumnIndex = oTable.retrieveLeafColumns().length - 1;

                    TableUtils._attachAfterInvalidationOnce(oTable, function() {
                        var nextActiveDomRef = (oActiveRow)
                            ? oTable.getDomRef("rows-row" + iNextActiveRowIndex + "-col" + iNextActiveColumnIndex)
                            : oTable.retrieveLeafColumns()[iNextActiveColumnIndex].$()[0];
                        oTable._focusItemByDomRef(nextActiveDomRef);
                        resolve();
                    });

                    oTable.makeFirstVisibleColumn(iNextActiveColumnIndex, false);
                });
            }
        }; // end of oWideTableHandlers


        /*
     * Event handling which is independent of the used keyboard delegate.
     * "this" in the function context is the table instance.
     */
        var ExtensionDelegate = {

            onfocusin : function(oEvent) {
                var oExtension = this._getKeyboardExtension();
                if (!oExtension._bIgnoreFocusIn) {
                    oExtension.initItemNavigation();
                    if (ExtensionHelper.isItemNavigationInvalid(this)) {
                        oEvent.setMarked("sapUiTableInitItemNavigation");
                    }
                } else {
                    oEvent.setMarked("sapUiTableIgnoreFocusIn");
                }

                if (oEvent.target && oEvent.target.id === this.getId() + "-rsz") {
                // prevent that the ItemNavigation grabs the focus!
                // only for the column resizing
                    oEvent.preventDefault();
                    oEvent.setMarked("sapUiTableSkipItemNavigation");
                }
            }

        };


        /*
     * Provides utility functions used this extension
     */
        var ExtensionHelper = {

        /*
         * Initialize ItemNavigations (content and header) and transfer relevant dom elements.
         * TabIndexes are set by the ItemNavigation.
         */
            _initItemNavigation : function(oExtension) {
                var oTable = oExtension.getTable();
                var $Table = oTable.$();
                // adjust if wide table
                var iColumnCount =
                (TableUtils.isWideTable(oTable) === true) ?
                    oTable.getDisplayedColumnCount() :
                    TableUtils.getVisibleColumnCount(oTable);
                var iTotalColumnCount = iColumnCount;
                var bHasRowHeader = TableUtils.hasRowHeader(oTable);

                // create the list of item dom refs
                var aItemDomRefs = [];
                if (oTable.getTotalFrozenColumnCount() === 0) {
                    aItemDomRefs = $Table.find(".sapUiTableCtrl td[tabindex]").get();
                } else {
                    var $topLeft = $Table.find('.sapUiTableCtrlFixed.sapUiTableCtrlRowFixed');
                    var $topRight = $Table.find('.sapUiTableCtrlScroll.sapUiTableCtrlRowFixed');
                    var $middleLeft = $Table.find('.sapUiTableCtrlFixed.sapUiTableCtrlRowScroll');
                    var $middleRight = $Table.find('.sapUiTableCtrlScroll.sapUiTableCtrlRowScroll');
                    var $bottomLeft = $Table.find('.sapUiTableCtrlFixed.sapUiTableCtrlRowFixedBottom');
                    var $bottomRight = $Table.find('.sapUiTableCtrlScroll.sapUiTableCtrlRowFixedBottom');
                    for (var i = 0; i < oTable.getVisibleRowCount(); i++) {
                        aItemDomRefs = aItemDomRefs.concat($topLeft.find('tr[data-sap-ui-rowindex="' + i + '"]').find('td[tabindex]').get());
                        aItemDomRefs = aItemDomRefs.concat($topRight.find('tr[data-sap-ui-rowindex="' + i + '"]').find('td[tabindex]').get());
                        aItemDomRefs = aItemDomRefs.concat($middleLeft.find('tr[data-sap-ui-rowindex="' + i + '"]').find('td[tabindex]').get());
                        aItemDomRefs = aItemDomRefs.concat($middleRight.find('tr[data-sap-ui-rowindex="' + i + '"]').find('td[tabindex]').get());
                        aItemDomRefs = aItemDomRefs.concat($bottomLeft.find('tr[data-sap-ui-rowindex="' + i + '"]').find('td[tabindex]').get());
                        aItemDomRefs = aItemDomRefs.concat($bottomRight.find('tr[data-sap-ui-rowindex="' + i + '"]').find('td[tabindex]').get());
                    }
                }

                // to later determine the position of the first TD in the aItemDomRefs we keep the
                // count of TDs => aCount - TDs = first TD (add the row headers to the TD count / except the first one!)
                var iTDCount = aItemDomRefs.length;

                // add the row header items (if visible)
                if (bHasRowHeader) {

                    var aRowHdrDomRefs = oTable._getItemNavRowHdrDomRefs();

                    for (var i = aRowHdrDomRefs.length - 1; i >= 0; i--) {
                        aItemDomRefs.splice(i * iColumnCount, 0, aRowHdrDomRefs[i]);
                        // we ignore the row headers
                        iTDCount++;
                    }
                    // except the first row header
                    iTDCount--;
                    // add the row header to the column count
                    iTotalColumnCount++;
                }

                // add the column headers and select all
                var bShouldShowSummaryHeader = oTable.shouldShowSummaryHeader();
                if (oTable.getColumnHeaderVisible() || bShouldShowSummaryHeader) {
                    var aHeaderDomRefs = [];

                    var $FrozenHeaders = $Table.find(".sapUiTableColHdrFixed tbody").children(); //returns the .sapUiTableColHdr elements
                    var $ScrollHeaders = $Table.find(".sapUiTableColHdrScr tbody").children(); //returns the .sapUiTableColHdr elements

                    var iHRCount = TableUtils.getHeaderRowCount(oTable);
                    // add the summary headers if present
                    if (bShouldShowSummaryHeader) {
                        iHRCount += 2;
                    }

                    var aAllTDs = jQuery($ScrollHeaders).find(".sapUiTableCol");
                    for (var i = 0; i < iHRCount; i++) {
                        if (bHasRowHeader) {
                            aHeaderDomRefs.push(oTable._getItemNavSelectAllDomRef());
                        }

                        if ($FrozenHeaders.length) {
                            aHeaderDomRefs = aHeaderDomRefs.concat(jQuery($FrozenHeaders.get(i)).find(".sapUiTableCol").get());
                        }

                        var aTDs = jQuery($ScrollHeaders.get(i)).find(".sapUiTableCol");
                        for (var j=0; j < aTDs.length; j++) {
                            if (oTable._aItemNavArray && !jQuery(aTDs[j]).find("div").is(":focusable")) {
                                aTDs[j] = aAllTDs[oTable._aItemNavArray[i][j]];
                            }
                        }
                        if ($ScrollHeaders.length) {
                            aHeaderDomRefs = aHeaderDomRefs.concat(aTDs.get());
                        }
                    }

                    aItemDomRefs = aHeaderDomRefs.concat(aItemDomRefs);
                }

                // initialization of item navigation for the Table control
                if (!oExtension._itemNavigation) {

                    oExtension._itemNavigation = oTable._createItemNavigation();

                    oExtension._itemNavigation.setTableMode(true);

                    // create a focus change handler that will apply to both FocusAgain and AfterFocus events
                    var fnFocusChangeHandler = function(oEvent) {
                        var oInfo = TableUtils.getFocusedItemInfo(oTable),
                            oLastInfo = oExtension._oLastTablePosition,
                            oTrigger;
                        oInfo.header = TableUtils.getHeaderRowCount(oTable);
                        oInfo.domRef = null; //Do not keep dom references

                        if (oInfo.row >= oInfo.header) {
                            oExtension._oLastFocusedCellInfo = oInfo;
                        }

                        // last table position helps figure out which cell we came from (if any)
                        oExtension._oLastTablePosition = TableUtils.getTablePositionFromFocusInfo(oTable, oInfo);
                        if (oEvent) {
                            oTrigger = oEvent.getParameter("event");
                        }
                        oTable._getAccExtension().announceCellNavigation(oInfo, oLastInfo, oTrigger);
                    };

                    oExtension._itemNavigation.attachEvent(ItemNavigation.Events.AfterFocus, fnFocusChangeHandler, oTable);
                    oExtension._itemNavigation.attachEvent(ItemNavigation.Events.FocusAgain, fnFocusChangeHandler, oTable);
                }

                // configure the item navigation
                if (TableUtils.isWideTable(oTable) === true) {
                    var iColumns = oTable.getDisplayedColumnCount();
                    if (TableUtils.hasRowHeader(oTable) === true) {
                        iColumns++;
                    }
                    oExtension._itemNavigation.setColumns(iColumns);
                } else {
                    oExtension._itemNavigation.setColumns(iTotalColumnCount);
                }

                oExtension._itemNavigation.setRootDomRef($Table.find(".sapUiTableCnt").get(0));
                oExtension._itemNavigation.setItemDomRefs(aItemDomRefs);
                oExtension._itemNavigation.setFocusedIndex(ExtensionHelper.getInitialItemNavigationIndex(oExtension));

                // revert invalidation flag
                oExtension._itemNavigationInvalidated = false;
            },

            _createKeyboardHandlerDelegate: function(oTable) {
                var oKeyboardHandlers = keyboardHandlers(oTable);
                var oKeyboardHandlerDelegate = {};
                var aTurnOffActionMode = [
                    'onsapescape',
                    'onsaptabprevious',
                    'onsaptabnext'
                ];

                jQuery.each(oKeyboardHandlers, function(sKey, fn) {
                    oKeyboardHandlerDelegate[sKey] = function(oEvent) {
                        var oColumn,
                            $Target = jQuery(oEvent.target),
                            $Cell = TableUtils.getCell(oTable, oEvent.target),
                            oCellInfo = TableUtils.getCellInfo($Cell) || {},
                            $focusable = $Target.closest(":focusable"),
                            bFocusableInCell = false;

                        if (oCellInfo && oCellInfo.cell) {
                            bFocusableInCell = oCellInfo.cell.find($focusable).length > 0;
                        }

                        // only handle events within the navigable bounds of the Table contents
                        if (jQuery.sap.containsOrEquals(oTable.getDomRef("sapUiTableCnt"), oEvent.target) === false) {
                            // we could be in the InPlaceEdit of an editable header
                            var oInPlaceEdit = jQuery(oEvent.target).control()[0].getParent();
                            var oEditColumn;
                            var bInEditableHeader = false;
                            var InPlaceEdit = sap.ui.requireSync("sas/hc/ui/commons/InPlaceEdit");
                            var Column = sap.ui.requireSync("sas/hc/ui/table/Column");
                            if (oInPlaceEdit && oInPlaceEdit instanceof InPlaceEdit) {
                                oEditColumn = oInPlaceEdit.getParent();
                            }
                            if (oEditColumn && oEditColumn instanceof Column) {
                                bInEditableHeader = jQuery.sap.containsOrEquals(oTable.getDomRef("sapUiTableCnt"), oEditColumn.getDomRef());
                                // if we are leaving an editable header input, set focus on the column header cell
                                if (bInEditableHeader && jQuery(oEvent.target).filter("input").length > 0 && sKey !== "onfocusin") {
                                    oTable._getKeyboardExtension().setActionMode(false, {$Dom: oEditColumn.getDomRef(), bInEditableHeader:true});
                                }
                            }
                            // if we're not in an editable header, just return
                            if (!bInEditableHeader) {
                                return;
                            }
                        }

                        // some events call for exiting action mode
                        if (oTable._getKeyboardExtension().isInActionMode() && aTurnOffActionMode.indexOf('on' + oEvent.type) !== -1) {
                            oTable._getKeyboardExtension().setActionMode(false, {event: oEvent});
                        }

                        // stop if in action mode, allow event to propagate
                        if (oTable._getKeyboardExtension()._itemNavigationSuspended || oEvent.isMarked("sapUiTableSkipItemNavigation")) {
                            return;
                        }

                        // stop if user is editing column header label, allow event to propagate
                        oColumn = jQuery(oEvent.target).closest(".sapUiTableCol").control()[0];
                        if (
                            oEvent.originalEvent instanceof KeyboardEvent &&
                        !!oColumn &&
                        jQuery.isFunction(oColumn.getHeaderEditable) &&
                        oColumn.getHeaderEditable() &&
                        !!oColumn._oInPlaceEdit &&
                        oColumn._oInPlaceEdit._bEditMode === true
                        ) {
                        //appear to be editing a column header, so stop attempting to handle the event
                            return;
                        }

                        // stop if keyboard action is currently taking place
                        if (oTable._getKeyboardExtension()._keyboardActionInProcess === true) {
                            oEvent.stopImmediatePropagation(true);
                            oEvent.preventDefault();
                            return;
                        }

                        // handle certain events directly, without the promise wrapper
                        if (/(mouse)/.test(sKey) === true) {
                            oKeyboardHandlers[sKey](oEvent);
                            return;
                        }

                        // There does not exist a valid use-case where the focusin event occurs with a target of .sapUiTableCnt.
                        // This situation happens in Edge when the vertical scrollbar is clicked upon, and the propagated event
                        // ends up at .sapUiTableCnt which is a parent of the div which houses the scrollbar.
                        // This propagation does not seem directly preventable.
                        // When the ItemNav onfocusin handler is then invoked it causes negative side-effects.
                        // If the user has not manually put focus on the Table yet, the onfocusin ItemNavigation handler
                        // will put focus on the first entry in it's aItemDomRefs array, which is the first column header.
                        // Manually moving the horizontal scrollbar does not put focus on the table, so if the user then
                        // touches the vertical scrollbar the ItemNav is left to put focus on the default node (the first column header).
                        // This causes the horizontal scrollbar to snap back all the way to the left, which is a bug.
                        // We prevent this situation by ignoring focusin events that have a target of sapUiTableCnt which is
                        // the closest we can come to a "stopPropagation" when the vertical scrollbar is clicked upon.
                        if (sKey === "onfocusin" && oEvent.target === this.getDomRef("sapUiTableCnt")) {
                            return;
                        }

                        // if coming from column header to a cell we handle the event ourselves
                        if (sKey === 'onsaptabnext' && oEvent.target.getAttribute('role') === 'columnheader') {
                            oEvent.stopImmediatePropagation(true);
                            oEvent.preventDefault();
                        }
                        // if coming from cell to a columnheader we handle the event ourselves
                        if (sKey === 'onsaptabprevious' && oEvent.target.getAttribute('role') === 'gridcell'
                            // exclude the 'no content' element which currently also has gridcell role
                            && oEvent.target !== oTable.getDomRef("noDataCnt")
                        ) {
                            oEvent.stopImmediatePropagation(true);
                            oEvent.preventDefault();
                        }

                        // if coming from selection checkbox to a select all checkbox we handle the event ourselves
                        if (sKey === 'onsaptabprevious' && oEvent.target.getAttribute('role') === 'checkbox' && oEvent.target.parentElement.classList.contains('sapUiTableRowHdr')) {
                            if (oTable.getEnableSelectAll()) {
                                oEvent.stopImmediatePropagation(true);
                                oEvent.preventDefault();
                            }
                        }

                        // Allow propagation to continue when certain keys were pressed
                        // This is because focus will be placed on .sapUiTableCtrlAfter before
                        // the browser has a chance to handle the event.
                        // The native tab-index rules then are used to place focus on the next
                        // tab-able element (the table options button)
                        if (sKey !== 'onsaptabnext' && sKey !== 'onsaptabprevious' && sKey !== 'onsapescape' && oEvent.keyCode !== jQuery.sap.KeyCodes.ENTER) {
                            // make sure we're not inside a control that's embedded in the table cell
                            if (!bFocusableInCell) {
                            // at this point, we are committed to fully handling the event
                                oEvent.stopImmediatePropagation(true);
                                oEvent.preventDefault();
                            }
                        }

                        // run the key handler
                        // manage the flag to indicate that a keyboard action is in process
                        oTable._getKeyboardExtension()._keyboardActionInProcess = true;
                        oKeyboardHandlers[sKey](oEvent).then(function() {
                            oTable._getKeyboardExtension()._keyboardActionInProcess = false;
                        });
                    };
                });
                return oKeyboardHandlerDelegate;
            },

            getInitialItemNavigationIndex : function(oExtension) {
                return TableUtils.hasRowHeader(oExtension.getTable()) ? 1 : 0;
            },

            isItemNavigationInvalid : function(oExtension) {
                return !oExtension._itemNavigation || oExtension._itemNavigationInvalidated;
            }
        };

        /**
         * Extension for sas.hc.ui.table.Table which handles keyboard related things.
         *
         * @class Extension for sas.hc.ui.table.Table which handles keyboard related things.
         *
         * @extends sas.hc.ui.table.TableExtension
         * @author SAP SE
         * @version 904001.11.16.20251118090100_f0htmcm94p
         * @constructor
         * @private
         * @alias sas.hc.ui.table.TableKeyboardExtension
         */
        var TableKeyboardExtension = TableExtension.extend("sas.hc.ui.table.TableKeyboardExtension", /* @lends sas.hc.ui.table.TableKeyboardExtension */ {

            /*
             * @see TableExtension._init
             */
            _init : function(oTable, sTableType, mSettings) {
                this._itemNavigation = null;
                this._itemNavigationInvalidated = false; // determines whether item navigation should be reapplied from scratch
                this._itemNavigationSuspended = false; // switch off event forwarding to item navigation
                this._keyboardActionInProcess = false; // only allow one keyboard action at a time
                this._type = sTableType;

                this._delegate = this._createTableKeyboardDelegate(sTableType);

                this._actionMode = false;

                // Register the delegates in correct order
                oTable.addEventDelegate(ExtensionDelegate, oTable);
                oTable.addEventDelegate(this._delegate, oTable);

                this._keyboardHandlerDelegate = ExtensionHelper._createKeyboardHandlerDelegate(oTable);
                oTable.addEventDelegate(this._keyboardHandlerDelegate, oTable);

                var that = this;
                oTable._getItemNavigation = function() { return that._itemNavigation; };

                return "KeyboardExtension";
            },

            /*
             * @see sap.ui.base.Object#destroy
             */
            destroy : function() {
                // Deregister the delegates
                var oTable = this.getTable();
                if (oTable) {
                    oTable.removeEventDelegate(ExtensionDelegate);
                    oTable.removeEventDelegate(this._delegate);
                }

                if (this._itemNavigation) {
                    this._itemNavigation.destroy();
                    this._itemNavigation = null;
                }

                if (this._delegate) {
                    this._delegate.destroy();
                    this._delegate = null;
                }

                TableExtension.prototype.destroy.apply(this, arguments);
            }

        });


        /*
         * Check whether item navigation should be reapplied from scratch and initializes it if needed.
         * @public (Part of the API for Table control only!)
         */
        TableKeyboardExtension.prototype.initItemNavigation = function() {
            if (ExtensionHelper.isItemNavigationInvalid(this)) {
                ExtensionHelper._initItemNavigation(this);
            }
        };


        /*
         * Invalidates the item navigation (forces a re-initialization with the next initItemNavigation call)
         * @public (Part of the API for Table control only!)
         */
        TableKeyboardExtension.prototype.invalidateItemNavigation = function() {
            this._itemNavigationInvalidated = true;
        };


        /*
         * Set or resets the action mode of the table.
         * In the action mode the user can navigate through the interactive controls of the table.
         * @public (Part of the API for Table control only!)
         */
        TableKeyboardExtension.prototype.setActionMode = function(bActionMode, oArgs) {
            var bInEditableHeader = false;
            if (oArgs && oArgs.bInEditableHeader) {
                // if a header is editable, the user can use the mouse to edit the
                // InPlaceEdit and avoid this._actionMode being set, so always call
                // leaveActionMode if bActionMode is false, even if this._actionMode is false
                bInEditableHeader = oArgs.bInEditableHeader;
            }
            if (bActionMode && !this._actionMode && this.enterActionMode) {
                this._actionMode = !!this.enterActionMode(this.getTable(), oArgs || {});
            } else if (!bActionMode && (this._actionMode || bInEditableHeader) && this.leaveActionMode) {
                this._actionMode = false;
                this.leaveActionMode(this.getTable(), oArgs || {});
            }
        };


        /*
         * Returns true when the table is in action mode, false otherwise.
         * @public (Part of the API for Table control only!)
         */
        TableKeyboardExtension.prototype.isInActionMode = function() {
            return this._actionMode;
        };


        /*
         * Sets the focus depending on the noData or overlay mode.
         * The previous focused element is given (potentially this is not anymore the active focused element,
         * e.g. see Table.setShowOverlay -> tue to CSS changes the focused element might be hidden which forces a focus change)
         * @public (Part of the API for Table control only!)
         */
        TableKeyboardExtension.prototype.updateNoDataAndOverlayFocus = function(oPreviousFocusRef) {
            var oTable = this.getTable();
            if (!oTable || !oTable.getDomRef()) {
                return;
            }

            if (oTable.getShowOverlay()) {
                // The overlay is shown
                if (jQuery.sap.containsOrEquals(oTable.getDomRef(), oPreviousFocusRef)) {
                    oTable.$("overlay").focus(); // Set focus on Overlay Container if it was somewhere in the table before
                }
            } else if (this.showNoDataMessage()) {
                // The noData area is shown
                if (jQuery.sap.containsOrEquals(oTable.getDomRef("sapUiTableCnt"), oPreviousFocusRef)) {
                    oTable.$("noDataCnt").focus(); // Set focus on NoData Container if it was on the content before
                }
            } else if (jQuery.sap.containsOrEquals(oTable.getDomRef("noDataCnt"), oPreviousFocusRef)
                || jQuery.sap.containsOrEquals(oTable.getDomRef("overlay"), oPreviousFocusRef)) {
                // The overlay or noData area is not shown but was shown before
                TableUtils.focusItem(oTable, ExtensionHelper.getInitialItemNavigationIndex(this)); // Set focus on first focusable element
            }
        };


        /*
         * Suspends the event handling of the item navigation.
         * @protected (Only to be used by the keyboard delegate)
         */
        TableKeyboardExtension.prototype._suspendItemNavigation = function() {
            this._itemNavigationSuspended = true;
        };


        /*
         * Resumes the event handling of the item navigation.
         * @protected (Only to be used by the keyboard delegate)
         */
        TableKeyboardExtension.prototype._resumeItemNavigation = function() {
            this._itemNavigationSuspended = false;
        };


        /*
         * Returns the combined info about the last focused data cell (based on the item navigation)
         * @protected (Only to be used by the keyboard delegate)
         */
        TableKeyboardExtension.prototype._getLastFocusedCellInfo = function() {
            var iHeader = TableUtils.getHeaderRowCount(this.getTable());
            if (!this._oLastFocusedCellInfo || this._oLastFocusedCellInfo.header !== iHeader) {
                var oInfo = TableUtils.getFocusedItemInfo(this.getTable());
                var iDfltIdx = ExtensionHelper.getInitialItemNavigationIndex(this);
                return {
                    cellInRow : iDfltIdx,
                    row : iHeader,
                    header : iHeader,
                    cellCount : oInfo.cellCount,
                    columnCount : oInfo.columnCount,
                    cell : oInfo.columnCount * iHeader + iDfltIdx
                };
            }
            return this._oLastFocusedCellInfo;
        };


        /*
         * Sets the focus to the given DOM reference or jQuery Object and
         * marks the resulting focus event to be ignored.
         * @protected (Only to be used by the keyboard delegate)
         */
        TableKeyboardExtension.prototype._setSilentFocus = function(oRef) {
            this._bIgnoreFocusIn = true;
            oRef.focus();
            this._bIgnoreFocusIn = false;
        };


        /*
         * Returns the type of the related table
         * @see TableExtension.TABLETYPES
         * @protected (Only to be used by the keyboard delegate)
         */
        TableKeyboardExtension.prototype._getTableType = function() {
            return this._type;
        };

        /**
         * Factory method. Creates the TableKeyboardDelegate instance.
         * @param {string} sTableType the type of table
         * @protected
         * @return {object} TableKeyboardDelegate instance
         */
        TableKeyboardExtension.prototype._createTableKeyboardDelegate = function(sTableType) {
            return new TableKeyboardDelegate(sTableType);
        };

        //HTMLCOMMONS-6546
        TableKeyboardExtension.prototype.showNoDataMessage = function() {
            var table = this.getTable();
            return !table._hasData() || TableUtils.getVisibleColumnCount(table) === 0;
        };

        /**
         * Put focus on the first visible row (moving to the right-most column) of the current page<br />
         * handle the _skipVScroll flag
         * @private
         * @param {object} oTable table instance
         * @param {object} oEvent wrapped originating event
         */
        TableKeyboardExtension.prototype._putFocusAtBottomRightOfCurrentPage = function(oTable, oEvent) {
            return oTable._getKeyboardExtension()._safeFocus(oTable, oTable._determineCellDownAndRightTillPage(), oEvent);
        };

        /**
         * Put focus on the select-all control
         * @private
         * @param {object} oTable table instance
         * @param {object} oEvent wrapped originating event
         */
        TableKeyboardExtension.prototype._putFocusOnSelectAll = function(oTable, oEvent) {
            return new Promise(function(resolve) {
                TableUtils.focusItem(oTable, oTable._getItemNavigation().getIndexFromDomRef(oTable.getSelectAllControl().getDomRef()), oEvent);
                resolve();
            });
        };

        /**
         * Put focus on the last visible row (staying within the current column) of the current page<br />
         * handle the _skipVScroll flag
         * @private
         * @param {object} oTable table instance
         * @param {object} oEvent wrapped originating event
         */
        TableKeyboardExtension.prototype._putFocusAtBottomOfCurrentPage = function(oTable, oEvent) {
            return oTable._getKeyboardExtension()._safeFocus(oTable, oTable._determineCellDownTillPage(oTable._colFromRef(oEvent.target)), oEvent);
        };

        /**
         * Put focus on the first visible row (staying within the current column) of the current page<br />
         * handle the _skipVScroll flag
         * @private
         * @param {object} oTable table instance
         * @param {object} oEvent wrapped originating event
         */
        TableKeyboardExtension.prototype._putFocusAtTopOfCurrentPage = function(oTable, oEvent) {
            return oTable._getKeyboardExtension()._safeFocus(oTable, oTable._determineCellUpTillPage(oTable._colFromRef(oEvent.target)), oEvent);
        };

        /**
         * Put focus on the first visible row (moving to the left-most column) of the current page<br />
         * handle the _skipVScroll flag
         * @private
         * @param {object} oTable table instance
         * @param {object} oEvent wrapped originating event
         */
        TableKeyboardExtension.prototype._putFocusAtTopLeftOfCurrentPage = function(oTable, oEvent) {
            return oTable._getKeyboardExtension()._safeFocus(oTable, oTable._determineCellUpAndLeftTillPage(), oEvent);
        };

        /**
         * _focusItemByDomRef can cause the vertical scroll bar to adjust<br />
         * when page-down happens very frequently, so make sure onvscroll doesn't<br />
         * execute when that happens for a few milliseconds
         * @private
         * @param {object} oTable table instance
         * @param {object} oDom dom ref to put focus on
         * @param {object} oEvent wrapped originating event
         */
        TableKeyboardExtension.prototype._safeFocus = function(oTable, oDom, oEvent) {
            return new Promise(function(resolve) {
                oTable._skipVScroll = true;
                oTable._focusItemByDomRef(oDom, oEvent);
                oTable._setManagedTimeout(function() {
                    oTable._skipVScroll = false;
                    resolve();
                }, 10);
            });
        };

        /*
         * Restores the focus to the last known cell position.
         */
        TableKeyboardExtension.prototype._restoreFocusOnLastFocusedDataCell = function(oTable, oEvent) {
            var oInfo = TableUtils.getFocusedItemInfo(oTable);
            var oLastInfo = this._getLastFocusedCellInfo();
            TableUtils.focusItem(oTable, oInfo.cellInRow + (oInfo.columnCount * oLastInfo.row), oEvent);
        };

        /*
         * Hook which is called by the keyboard extension when the table should be set to action mode
         * @see TableKeyboardExtension#setActionMode
         */
        TableKeyboardExtension.prototype.enterActionMode = function(oTable, oArgs) {
            var $Focusable = oArgs.$Dom;
            var bEntered = false;

            if ($Focusable.length > 0) {

                var $Tabbables = $Focusable.filter(":sapTabbable");

                if ($Tabbables.length > 0) { //If cell has no tabbable element, we don't do anything
                    bEntered = true;

                    // in the action mode we need no item navigation
                    var oIN = oTable._getItemNavigation();
                    this._suspendItemNavigation();

                    // remove the tab index from the item navigation
                    jQuery(oIN.getFocusedDomRef()).attr("tabindex", "-1");
                    // clear cell's aria-hidden attribute when table is in action mode to allow JAWS to dictate contents
                    jQuery(oIN.getFocusedDomRef()).attr("aria-hidden", null);

                    // set the focus to the active control
                    $Tabbables.eq(0).focus();
                }
            }

            return bEntered;
        };


        /*
         * Hook which is called by the keyboard extension when the table should leave the action mode
         * @see TableKeyboardExtension#setActionMode
         */
        TableKeyboardExtension.prototype.leaveActionMode = function(oTable, oArgs) {
            // TODO: update ItemNavigation position otherwise the position is strange!
            //       EDIT AN SCROLL!

            var oEvent = oArgs.event;
            var $Focusable = oArgs.$Dom;

            // in the navigation mode we use the item navigation
            var oIN = oTable._getItemNavigation(); //TBD: Cleanup
            this._resumeItemNavigation();

            // reset the tabindex of the focused domref of the item navigation
            jQuery(oIN.getFocusedDomRef()).attr("tabindex", "0");
            // set cell's aria-hidden to true when table is not in action mode to keep JAWS from saying more than desired
            jQuery(oIN.getFocusedDomRef()).attr("aria-hidden", true);

            // when we have an event which is responsible to leave the action mode
            // we search for the closest
            if (oEvent) {
                if (jQuery(oEvent.target).closest("td[tabindex='-1']").length > 0) {
                    // triggered when clicking into a cell, then we focus the cell
                    var iIndex = jQuery(oIN.aItemDomRefs).index(jQuery(oEvent.target).closest("td[tabindex='-1']").get(0));
                    TableUtils.focusItem(oTable, iIndex, null);
                } else {
                    // somewhere else means whe check if the click happend inside
                    // the container, then we focus the last focused element
                    // (DON'T KNOW IF THIS IS OK - but we don't know where the focus was!)
                    if (jQuery.sap.containsOrEquals(oTable.$().find(".sapUiTableCCnt").get(0), oEvent.target)) {
                        TableUtils.focusItem(oTable, oIN.getFocusedIndex(), null);
                    }
                }
            } else if ($Focusable) {
                var iIndex = jQuery(oIN.aItemDomRefs).index($Focusable);
                TableUtils.focusItem(oTable, iIndex, null);
            } else {
                // when no event is given we just focus the last focused index
                TableUtils.focusItem(oTable, oIN.getFocusedIndex(), null);
            }

            //Special handling for the tree icon in the TreeTable
            if (this._getTableType() === TableExtension.TABLETYPES.TREE) {
                oTable.$().find(".sapUiTableTreeIcon").attr("tabindex", -1);
            }
        };

        /* onsapnext code is moved here so that large tree can override it*/
        TableKeyboardExtension.prototype._onsapnext = function(oTable, oEvent) {
            if (TableUtils.isWideTable(oTable) === true) {
                return oWideTableHandlers._onsapnext(oTable, oEvent);
            }else {
                return new Promise(function(resolve) {
                    var bColumnGroups = oTable.hasColumnGroups();
                    if (oTable._getItemNavigation()) {
                        oTable._getItemNavigation()['onsapnext'](oEvent);
                        if (bColumnGroups) {
                            oTable.focusNextFocusableHeader(oEvent);
                        }
                    }
                    resolve();
                });
            }
        };

        /* onsapprevious code is moved here so that large tree can override it*/
        TableKeyboardExtension.prototype._onsapprevious = function(oTable, oEvent) {
            if (TableUtils.isWideTable(oTable) === true) {
                return oWideTableHandlers._onsapprevious(oTable, oEvent);
            } else {
                return new Promise(function(resolve) {
                    var bColumnGroups = oTable.hasColumnGroups();
                    if (oTable._getItemNavigation()) {
                        oTable._getItemNavigation()['onsapprevious'](oEvent);
                        if (bColumnGroups) {
                            oTable.focusPreviousFocusableHeader(oEvent);
                        }
                    }
                    resolve();
                });
            }
        };

        return TableKeyboardExtension;

    }, /* bExport= */ true);
