/*!
 * (c) 2014-2018, SAS Institute Inc.
 */

// Provides helper sas.hc.ui.table.TableAccExtension.
sap.ui.define([
    'jquery.sap.global',
    'sap/ui/core/Control',
    './library',
    './TableExtension',
    './TableAccRenderExtension',
    './TableUtils',
    'sas/hc/ui/core/Narrator',
    "sap/ui/core/Core",
],
function(jQuery, Control, library, TableExtension, TableAccRenderExtension, TableUtils, Narrator) {
    "use strict";

    // shortcuts
    var SelectionMode = library.SelectionMode,
        SortOrder = library.SortOrder;

    /*
     * Provides utility functions to handle acc info objects.
     * @see {sap.ui.core.Control#getAccessibilityInfo}
     */
    var ACCInfoHelper = {

        /*
         * Returns a flattened acc info object (infos of children are merged together)
         * Note: The info object does only contain a focusable flag (true if one of the children is focusable)
         *       and a combined description.
         * @see {sap.ui.core.Control#getAccessibilityInfo}
         */
        getAccInfoOfControl: function(oControl, oBundle) {
            if (oControl && typeof oControl.getAccessibilityInfo === "function") {
                if (typeof oControl.getVisible === "function" && !oControl.getVisible()) {
                    return ACCInfoHelper._normalize({});
                }
                var oSource = oControl.getAccessibilityInfo();
                if (oSource) {
                    var oTarget = {};
                    ACCInfoHelper._flatten(oSource, oTarget, oBundle);
                    return oTarget;
                }
            }
            return null;
        },

        /*
         * Normalizes the given acc info object and ensures that all the defaults are set.
         */
        _normalize : function(oInfo) {
            if (!oInfo) {
                return null;
            }

            if (oInfo._normalized) {
                return oInfo;
            }

            oInfo.role = oInfo.role || "";
            oInfo.type = oInfo.type || "";
            oInfo.description = oInfo.description || "";
            oInfo.focusable = !!oInfo.focusable;
            oInfo.enabled = (oInfo.enabled === true || oInfo.enabled === false) ? oInfo.enabled : null;
            oInfo.editable = (oInfo.editable === true || oInfo.editable === false) ? oInfo.editable : null;
            oInfo.children = oInfo.children || [];
            oInfo._normalized = true;

            return oInfo;
        },

        /*
         * Merges the focusable flag and the descriptions of the source and its children into the given target.
         */
        _flatten : function(oSourceInfo, oTargetInfo, oBundle, iLevel) {
            iLevel = iLevel ? iLevel : 0;

            ACCInfoHelper._normalize(oSourceInfo);
            if (iLevel === 0) {
                ACCInfoHelper._normalize(oTargetInfo);
                oTargetInfo._descriptions = [];
            }

            oTargetInfo.focusable = oTargetInfo.focusable || oSourceInfo.focusable;
            oTargetInfo._descriptions.push(ACCInfoHelper._getFullDescription(oSourceInfo, oBundle));

            oSourceInfo.children
                .filter(function(oChild) {
                    return (
                        //to be part of the result, a child must have the getAccessibilityInfo method
                        jQuery.isFunction(oChild.getAccessibilityInfo)
                        //and if it can have a visibility, then it must also be visible
                        && ((!jQuery.isFunction(oChild.getVisible)) || oChild.getVisible())
                    );
                })
                .forEach(function(oChild) {
                    var oChildInfo = oChild.getAccessibilityInfo();
                    if (oChildInfo) {
                        ACCInfoHelper._flatten(oChildInfo, oTargetInfo, oBundle, iLevel + 1);
                    }
                });

            if (iLevel === 0) {
                oTargetInfo.description = oTargetInfo._descriptions.join(" ").trim();
                oTargetInfo._descriptions = undefined;
            }
        },

        /*
         * Returns the full control description incl. control type and enabled/editable state based
         * on the information of the given acc info object
         * Note: The description does not include the description of the children (if available).
         */
        _getFullDescription : function(oInfo, oBundle) {
            var sDesc = oInfo.type + " " + oInfo.description;
            if (oInfo.enabled !== null && !oInfo.enabled) {
                sDesc = sDesc + " " + oBundle.getText("TBL_CTRL_STATE_DISABLED");
            } else if (oInfo.editable !== null && !oInfo.editable) {
                sDesc = sDesc + " " + oBundle.getText("TBL_CTRL_STATE_READONLY");
            }
            return sDesc.trim();
        },

        /**
         * Returns an Array of 1 or more sap.ui.core.Control instances that may be important when describing the content
         * of a cell.  The top-level Control passed in is asked for its accessibility info and if it declares any children,
         * then those children will be used to describe the content (recursively).
         * @param oTop {sap.ui.core.Control} the top-level control to examine
         * @returns {Array} an array of Controls that form the best guess for being important to accessibility
         */
        gatherCellControlsForARIALabeling : function(oTop) {
            var aChildren = [], //the output Controls in order they should be announced
                aList = [oTop], //the Control(s) we will examine
                oControl,       //the current Control
                oInfo;          //a11y info from the control

            // do we have any Controls to examine?
            while (aList.length > 0) {
                // get the top Control from the queue
                oControl = aList.shift();

                // oControl should always be something, but check to be sure
                if (!oControl) {
                    continue;
                }

                // if we can get a11yInfo, do so
                if (jQuery.isFunction(oControl.getAccessibilityInfo)) {
                    oInfo = oControl.getAccessibilityInfo();
                    // if the control defines a11y children,
                    if (oInfo && oInfo.children && oInfo.children.length > 0) {
                        // then add those children to the queue in the correct order
                        aList = aList.concat(oInfo.children);
                        // but do not add the control itself to our list of a11y important children
                    } else {
                        // otherwise, just add the Control
                        aChildren.push(oControl);
                    }
                } else {
                    // otherwise, just add the Control
                    aChildren.push(oControl);
                }
            }

            // return the key Controls that we found
            return aChildren;
        }

    };


    /*
     * Provides utility functions used this extension
     */
    var ExtensionHelper = {

        /*
         * If the current focus is on a cell of the table, this function returns
         * the cell type and the jQuery wrapper object of the corresponding cell:
         * {type: <TYPE>, cell: <$CELL>}
         */
        getInfoOfFocusedCell : function(oExtension) {
            var oTable = oExtension.getTable();
            var oIN = oTable._getItemNavigation();
            var oTableRef = oTable.getDomRef();

            if (!oExtension.getAccMode() || !oTableRef || !oIN) {
                return null;
            }
            var oCellRef = oIN.getFocusedDomRef();
            if (!oCellRef || oCellRef !== document.activeElement) {
                return null;
            }

            return TableUtils.getCellInfo(oCellRef);
        },

        /*
         * Returns whether the given cell is hidden
         */
        isHiddenCell : function($Cell) {
            return $Cell.parent().hasClass("sapUiTableRowHidden") || $Cell.hasClass("sapUiTableCellHidden")
                    || (TableUtils.isInGroupingRow($Cell) && $Cell.hasClass("sapUiTableTdFirst") && !$Cell.hasClass("sapUiTableMeasureCell"));
        },

        /*
         * Returns whether the given cell is in the tree column of a TreeTable
         */
        isTreeColumnCell : function(oExtension, $Cell) {
            return TableUtils.Grouping.isTreeMode(oExtension.getTable()) && $Cell.hasClass("sapUiTableTdFirst");
        },

        /*
         * Returns the tooltip of the column or the contained label, if any.
         */
        getColumnTooltip : function(oColumn) {
            if (!oColumn) {
                return null;
            }

            var sTooltip = oColumn.getTooltip_AsString();
            if (sTooltip) {
                return sTooltip;
            }

            var oLabel = oColumn.getLabel();
            if (oLabel instanceof Control) {
                sTooltip = oLabel.getTooltip_AsString();
            }
            if (sTooltip) {
                return sTooltip;
            }

            return null;
        },

        /*
         * Determines the current row and column and updates the hidden description texts of the table accordingly.
         */
        updateRowColCount : function(oExtension) {
            var oTable = oExtension.getTable(),
                oIN = oTable._getItemNavigation(),
                bIsRowChanged = false,
                bIsColChanged = false,
                bIsInitial = false;

            if (oIN) {
                var iColumnNumber = TableUtils.getColumnIndexOfFocusedCell(oTable) + 1; //+1 -> we want to announce a count and not the index
                var iRowNumber = TableUtils.getRowIndexOfFocusedCell(oTable) + oTable.getFirstVisibleRow() + 1; //same here + take virtualization into account
                var iColCount = TableUtils.getVisibleColumnCount(oTable);
                var iRowCount = TableUtils.isNoDataVisible(oTable) ? 0 : TableUtils.getTotalRowCount(oTable, true);

                bIsRowChanged = oExtension._iLastRowNumber !== iRowNumber || (oExtension._iLastRowNumber === iRowNumber && oExtension._iLastColumnNumber === iColumnNumber);
                bIsColChanged = oExtension._iLastColumnNumber !== iColumnNumber;
                bIsInitial = !oExtension._iLastRowNumber && !oExtension._iLastColumnNumber;

                oTable.$("rownumberofrows").text(bIsRowChanged ? oTable._oResBundle.getText("TBL_ROW_ROWCOUNT.fmt", [iRowNumber, iRowCount]) : " ");
                oTable.$("colnumberofcols").text(bIsColChanged ? oTable._oResBundle.getText("TBL_COL_COLCOUNT.fmt", [iColumnNumber, iColCount]) : " ");
                oTable.$("ariacount").text(bIsInitial ? oTable._oResBundle.getText("TBL_DATA_ROWS_COLS.fmt", [iRowCount, iColCount]) : " ");

                oExtension._iLastRowNumber = iRowNumber;
                oExtension._iLastColumnNumber = iColumnNumber;
            }

            return {
                rowChange: bIsRowChanged,
                colChange: bIsColChanged,
                initial: bIsInitial
            };
        },

        /*
         * Removes the acc modifications of the cell which had the focus before.
         */
        cleanupCellModifications : function(oExtension) {
            if (oExtension._cleanupInfo) {
                oExtension._cleanupInfo.cell.attr(oExtension._cleanupInfo.attr);
                oExtension._cleanupInfo = null;
            }
        },

        /*
         * Stores the defaults before modifications of a cell for later cleanup
         * @see ExtensionHelper.cleanupCellModifications
         */
        storeDefaultsBeforeCellModifications : function(oExtension, $Cell, aDefaultLabels, aDefaultDescriptions) {
            oExtension._cleanupInfo = {
                cell: $Cell,
                attr: {
                    "aria-labelledby" : aDefaultLabels && aDefaultLabels.length ? aDefaultLabels.join(" ") : null,
                    "aria-describedby" : aDefaultDescriptions && aDefaultDescriptions.length ? aDefaultDescriptions.join(" ") : null
                }
            };
        },

        /*
         * Updates the row / column counters, adapts the labels and descriptions of the given cell and stores the the
         * given defaults before the modification.
         * @see ExtensionHelper.updateRowColCount
         * @see ExtensionHelper.storeDefaultsBeforeCellModifications
         */
        performCellModifications : function(oExtension, $Cell, aDefaultLabels, aDefaultDescriptions, aLabels, aDescriptions, sText, fAdapt) {
            ExtensionHelper.storeDefaultsBeforeCellModifications(oExtension, $Cell, aDefaultLabels, aDefaultDescriptions);
            var oCountChangeInfo = ExtensionHelper.updateRowColCount(oExtension);
            oExtension.getTable().$("cellacc").text(sText || " "); //set the custom text to the prepared hidden element

            if (fAdapt) { //Allow to adapt the labels / descriptions based on the changed row / coulmn count
                fAdapt(aLabels, aDescriptions, oCountChangeInfo.rowChange, oCountChangeInfo.colChange, oCountChangeInfo.initial);
            }

            var sLabel = "";
            if (oCountChangeInfo.initial) {
                var oTable = oExtension.getTable();
                sLabel = oTable.getAriaLabelledBy().join(" ") + " " + oTable.getId() + "-ariadesc " + oTable.getId() + "-ariacount";
            }

            if (aLabels && aLabels.length) {
                sLabel = sLabel + " " + aLabels.join(" ");
            }

            $Cell.attr({
                "aria-labelledby" : sLabel ? sLabel : null,
                "aria-describedby" : aDescriptions && aDescriptions.length ? aDescriptions.join(" ") : null
            });
        },

        /*
         * Modifies the labels and descriptions of a data cell.
         * @see ExtensionHelper.performCellModifications
         */
        modifyAccOfDATACELL : function($Cell, bOnCellFocus) {
            var oTable = this.getTable(),
                oIN = oTable._getItemNavigation();

            if (!oIN) {
                return;
            }

            // explicitly remove any labelledby or describedby attributes as we are using Narrator for data cells and do not
            //  want to confuse JAWS
            $Cell.attr({
                "aria-labelledby": null,
                "aria-describedby": null
            });
        },

        /*
         * Modifies the labels and descriptions of a row header cell.
         * @see ExtensionHelper.performCellModifications
         */
        modifyAccOfROWHEADER : function($Cell, bOnCellFocus) {
            var oTable = this.getTable(),
                sTableId = oTable.getId(),
                bGroupHeader = TableUtils.isInGroupingRow($Cell),
                bSum = TableUtils.isInSumRow($Cell),
                oRow = oTable.getRows()[$Cell.attr("data-sap-ui-rowindex")],
                aDefaultLabels = ExtensionHelper.getAriaAttributesFor(this, TableAccExtension.ELEMENTTYPES.ROWHEADER)["aria-labelledby"] || [],
                aLabels = aDefaultLabels.concat([sTableId + "-rownumberofrows"]);

            if (!bSum && !bGroupHeader && oRow !== undefined) {
                if ($Cell.attr("aria-selected") === "true") {
                    aLabels.push(sTableId + "-ariarowselected");
                }
                if (!$Cell.hasClass("sapUiTableRowHidden")) {
                    aLabels.push(oRow.getId() + "-rowselecttext");
                }
            }

            if (bGroupHeader) {
                aLabels.push(sTableId + "-ariarowgrouplabel");
                //aLabels.push(oRow.getId() + "-groupHeader"); //Not needed: Screenreader seems to announce this automatically
            }

            if (bSum) {
                var iLevel = $Cell.data("sap-ui-level");
                if (iLevel === 0) {
                    aLabels.push(sTableId + "-ariagrandtotallabel");
                } else if (iLevel > 0) {
                    aLabels.push(sTableId + "-ariagrouptotallabel");
                    //aLabels.push(oRow.getId() + "-groupHeader"); //Not needed: Screenreader seems to announce this automatically
                }
            }

            ExtensionHelper.performCellModifications(this, $Cell, aDefaultLabels, null, aLabels, null, null);
        },

        /*
         * Modifies the labels and descriptions of a column header cell.
         * @see ExtensionHelper.performCellModifications
         */
        modifyAccOfCOLUMNHEADER : function($Cell, bOnCellFocus) {
            var oTable = this.getTable(),
                oColumn = sap.ui.getCore().byId($Cell.attr("data-sap-ui-colid")),
                mAttributes = ExtensionHelper.getAriaAttributesFor(this, TableAccExtension.ELEMENTTYPES.COLUMNHEADER, {
                    headerId: $Cell.attr("id"),
                    column: oColumn,
                    index: $Cell.attr("data-sap-ui-colindex")
                }),
                sText = ExtensionHelper.getColumnTooltip(oColumn),
                aLabels = [oTable.getId() + "-colnumberofcols"].concat(mAttributes["aria-labelledby"]);

            if (sText) {
                aLabels.push(oTable.getId() + "-cellacc");
            }

            //TBD: Improve handling for multiple headers
            ExtensionHelper.performCellModifications(this, $Cell, mAttributes["aria-labelledby"], mAttributes["aria-describedby"],
                aLabels, mAttributes["aria-describedby"], sText);
        },

        /*
         * Modifies the labels and descriptions of the column row header.
         * @see ExtensionHelper.performCellModifications
         */
        modifyAccOfCOLUMNROWHEADER : function($Cell, bOnCellFocus) {
            var mAttributes = ExtensionHelper.getAriaAttributesFor(this, TableAccExtension.ELEMENTTYPES.COLUMNROWHEADER, {enabled: $Cell.hasClass("sapUiTableSelAllEnabled")});
            ExtensionHelper.performCellModifications(this, $Cell, mAttributes["aria-labelledby"], mAttributes["aria-describedby"],
                mAttributes["aria-labelledby"], mAttributes["aria-describedby"], null);
        },

        /*
         * Returns the default aria attibutes for the given element type with the given settings.
         * @see TableAccExtension.ELEMENTTYPES
         */
        getAriaAttributesFor : function(oExtension, sType, mParams) {
            var mAttributes = {},
                oTable = oExtension.getTable(),
                sTableId = oTable.getId();

            function addAriaForOverlayOrNoData(oTable, mAttr, bOverlay, bNoData) {
                var sMarker = "";
                if (bOverlay && bNoData) {
                    sMarker = "overlay,nodata";
                } else if (bOverlay && !bNoData) {
                    sMarker = "overlay";
                } else if (!bOverlay && bNoData) {
                    sMarker = "nodata";
                }

                var bHidden = false;
                if (bOverlay && oTable.getShowOverlay() || bNoData && TableUtils.isNoDataVisible(oTable)) {
                    bHidden = true;
                }

                if (bHidden) {
                    mAttributes["aria-hidden"] = "true";
                }
                if (sMarker) {
                    mAttributes["data-sap-ui-table-acc-covered"] = sMarker;
                }
            }

            switch (sType) {
                case TableAccExtension.ELEMENTTYPES.COLUMNROWHEADER:
                    mAttributes["aria-labelledby"] = [sTableId + "-ariacolrowheaderlabel"];
                    mAttributes["role"] = ["columnheader"];
                    if (mParams && mParams.enabled) {
                        mAttributes["aria-labelledby"].push(sTableId + "-ariaselectall");
                    }
                    mAttributes["aria-colindex"] = 1;
                    mAttributes["aria-rowindex"] = 1;
                    break;

                case TableAccExtension.ELEMENTTYPES.ROWHEADER: //ROW HEADER SELECTION DIV (rendered after table content)
                    mAttributes["role"] = ["rowheader"];
                    mAttributes["aria-colindex"] = 1;
                    if (mParams && mParams.hasOwnProperty("index")) {
                        mAttributes["aria-rowindex"] = mParams.index+1;
                        mAttributes["aria-label"] = oTable._oResBundle.getText("table.aria.row_label.fmt", (mParams.index+1));
                    }
                    if (oTable.getSelectionMode() !== SelectionMode.None) {
                        var bSelected = mParams && mParams.rowSelected;
                        mAttributes["aria-selected"] = "" + bSelected;
                    }
                    break;

                case TableAccExtension.ELEMENTTYPES.COLUMNHEADER:
                    mAttributes["aria-hidden"] = "true";
                    break;

                case TableAccExtension.ELEMENTTYPES.DATACELL:
                    mAttributes["role"] = "gridcell";
                    if (mParams && typeof mParams.index === "number") {
                        mAttributes["headers"] = sTableId + "_col" + mParams.index
                                + " " + sTableId+"-rowsel"+oTable.indexOfRow(mParams.row);
                    }

                    // clearing these A11Y attributes to avoid JAWS saying context info about them while Narrator is trying to do the same
                    mAttributes["aria-labelledby"] = [];
                    mAttributes["aria-hidden"] = true;
                    mAttributes["aria-colindex"] = null;

                    // Handle expand state for first Column in TreeTable
                    if (TableUtils.Grouping.isTreeMode(oTable) && mParams && mParams.firstCol && mParams.row) {
                        var oBindingInfo = oTable.mBindingInfos["rows"];
                        if (mParams.row.getBindingContext(oBindingInfo && oBindingInfo.model)) {
                            mAttributes["aria-level"] = mParams.row._iLevel + 1;
                            mAttributes["aria-expanded"] = "" + mParams.row._bIsExpanded;
                        }
                    }
                    break;

                case TableAccExtension.ELEMENTTYPES.ROOT: //The tables root dom element
                    break;

                case TableAccExtension.ELEMENTTYPES.TABLE: //The "real" table element(s)
                    mAttributes["role"] = "presentation";
                    addAriaForOverlayOrNoData(oTable, mAttributes, true, true);
                    break;

                case TableAccExtension.ELEMENTTYPES.CONTENT: //The content area of the table which contains all the table elements, rowheaders, columnheaders, etc
                    mAttributes["role"] = TableUtils.Grouping.isGroupMode(oTable) || TableUtils.Grouping.isTreeMode(oTable) ? "treegrid" : "grid";
                    mAttributes["aria-labelledby"] = [].concat(oTable.getAriaLabelledBy());
                    if (oTable.getTitle()) {
                        mAttributes["aria-labelledby"].push(oTable.getTitle().getId());
                    }
                    if (oTable.getSelectionMode() === SelectionMode.MultiToggle) {
                        mAttributes["aria-multiselectable"] = "true";
                    }
                    break;

                case TableAccExtension.ELEMENTTYPES.TABLEHEADER: //The table header area
                    mAttributes["role"] = "heading";
                    addAriaForOverlayOrNoData(oTable, mAttributes, true, false);
                    break;

                case TableAccExtension.ELEMENTTYPES.COLUMNHEADER_ROW: //The area which contains the column headers (TableUtils.CELLTYPES.COLUMNHEADER)
                    mAttributes["role"] = "presentation";
                    addAriaForOverlayOrNoData(oTable, mAttributes, true, false);
                    break;

                case TableAccExtension.ELEMENTTYPES.ROWHEADER_COL: //The area which contains the row headers (TableUtils.CELLTYPES.ROWHEADER)
                    addAriaForOverlayOrNoData(oTable, mAttributes, true, true);
                    break;

                case TableAccExtension.ELEMENTTYPES.TH: //The "technical" column headers
                    var bHasFrozenColumns = oTable.getTotalFrozenColumnCount() > 0;
                    mAttributes["role"] = bHasFrozenColumns ? "columnheader" : "presentation";
                    mAttributes["scope"] = "col";
                    if (bHasFrozenColumns) {
                        if (mParams && mParams.column) {
                            mAttributes["aria-owns"] = mParams.column.getId();
                            mAttributes["aria-labelledby"] = [mParams.column.getId()];
                        }
                    } else {
                        mAttributes["aria-hidden"] = "true";
                    }
                    break;

                case TableAccExtension.ELEMENTTYPES.SELECTALL_COLHEADER: //The initial column header cell for a table section
                    mAttributes["role"] = "columnheader";
                    mAttributes["scope"] = "col";
                    mAttributes["aria-colindex"] = 1;
                    mAttributes["aria-labelledby"] = [oTable.getId()+"-selall"];
                    break;

                case TableAccExtension.ELEMENTTYPES.ROWHEADER_TD: //The "technical" row headers
                    mAttributes["role"] = "rowheader";
                    mAttributes["aria-labelledby"] = [];
                    mAttributes["headers"] = sTableId + "-colsel";
                    if (mParams && typeof mParams.index === "number") {
                        mAttributes["aria-owns"] = sTableId + "-rowsel" + mParams.index;
                        mAttributes["aria-labelledby"].push(sTableId + "-rowsel" + mParams.index); //row header labeling
                    }
                    if (oTable.getSelectionMode() !== SelectionMode.None) {
                        var bSelected = mParams && mParams.rowSelected;
                        mAttributes["aria-selected"] = "" + bSelected;
                    }
                    break;

                case TableAccExtension.ELEMENTTYPES.TR: //The rows
                    //if there is no row header, then JAWS will announce 'row selected' for every row erroneously
                    if (TableUtils.hasRowHeader(oTable)) {
                        mAttributes["role"] = "row";
                    }
                    var bSelected = false;
                    if (mParams && typeof mParams.index === "number" && oTable.getSelectionMode() !== SelectionMode.None) {
                        mAttributes["aria-owns"] = oTable.getId() + "-rowsel" + mParams.index;
                        if (oTable.isIndexSelected(mParams.index)) {
                            mAttributes["aria-selected"] = "true";
                            bSelected = true;
                        }
                    }
                    break;

                case TableAccExtension.ELEMENTTYPES.TREEICON: //The expand/collapse icon in the TreeTable
                    if (TableUtils.Grouping.isTreeMode(oTable)) {
                        mAttributes = {
                            "aria-label" : "",
                            "title" : "",
                            "role" : ""
                        };
                        if (oTable.getBinding("rows")) {
                            mAttributes["role"] = "button";
                            if (mParams && mParams.row) {
                                if (!mParams.row._bHasChildren) {
                                    mAttributes["aria-label"] = oTable._oResBundle.getText("TBL_LEAF");
                                }
                            }
                        }
                    }
                    break;

                case TableAccExtension.ELEMENTTYPES.NODATA: //The no data container
                    mAttributes["role"] = "gridcell";
                    var oNoData = oTable.getNoData();
                    mAttributes["aria-labelledby"] = [oNoData instanceof Control ? oNoData.getId() : (sTableId + "-noDataMsg")];
                    addAriaForOverlayOrNoData(oTable, mAttributes, true, false);
                    break;

                case TableAccExtension.ELEMENTTYPES.OVERLAY: //The overlay container
                    mAttributes["role"] = "region";
                    mAttributes["aria-labelledby"] = [].concat(oTable.getAriaLabelledBy());
                    if (oTable.getTitle()) {
                        mAttributes["aria-labelledby"].push(oTable.getTitle().getId());
                    }
                    mAttributes["aria-labelledby"].push(sTableId + "-ariainvalid");
                    break;

                case TableAccExtension.ELEMENTTYPES.TABLEFOOTER: //The table footer area
                case TableAccExtension.ELEMENTTYPES.TABLESUBHEADER: //The table toolbar and extension areas
                    addAriaForOverlayOrNoData(oTable, mAttributes, true, false);
                    break;
                case TableAccExtension.ELEMENTTYPES.COLUMN_RSZ:
                case TableAccExtension.ELEMENTTYPES.PRESENTATION:
                    mAttributes["role"] = "presentation";
                    break;
            }

            return mAttributes;
        }

    };


    /**
     * Extension for sas.hc.ui.table.Table which handles ACC related things.
     *
     * @class Extension for sas.hc.ui.table.Table which handles ACC related things.
     *
     * @extends sas.hc.ui.table.TableExtension
     * @author SAP SE
     * @version 904001.11.16.20251118090100_f0htmcm94p
     * @constructor
     * @private
     * @alias sas.hc.ui.table.TableAccExtension
     */
    var TableAccExtension = TableExtension.extend("sas.hc.ui.table.TableAccExtension", /* @lends sas.hc.ui.table.TableAccExtension */ {

        /*
         * @see TableExtension._init
         */
        _init : function(oTable, sTableType, mSettings) {
            this._accMode = sap.ui.getCore().getConfiguration().getAccessibility();
            this._readonly = sTableType === TableExtension.TABLETYPES.ANALYTICAL ? true : false;

            oTable.addEventDelegate(this);

            // Initialize Render extension
            TableExtension.enrich(oTable, TableAccRenderExtension);

            return "AccExtension";
        },

        /*
         * @see sap.ui.base.Object#destroy
         */
        destroy : function() {
            this.getTable().removeEventDelegate(this);

            this._readonly = false;

            TableExtension.prototype.destroy.apply(this, arguments);
        },

        /*
         * Provide protected access for TableACCRenderExtension
         * @see ExtensionHelper.getAriaAttributesFor
         */
        _getAriaAttributesFor : function(sType, mParams) {
            return ExtensionHelper.getAriaAttributesFor(this, sType, mParams);
        },

        /*
         * Delegate function for focusin event
         * @public (Part of the API for Table control only!)
         */
        onfocusin : function(oEvent) {
            var oTable = this.getTable();
            if (!oTable || !TableUtils.getCellInfo(oEvent.target)) {
                return;
            }
            if (oTable._mTimeouts._cleanupACCExtension) {
                jQuery.sap.clearDelayedCall(oTable._mTimeouts._cleanupACCExtension);
                oTable._mTimeouts._cleanupACCExtension = null;
            }
            this.updateAccForCurrentCell(true);
        },

        /*
         * Delegate function for focusout event
         * @public (Part of the API for Table control only!)
         */
        onfocusout: function(oEvent) {
            var oTable = this.getTable();
            if (!oTable) {
                return;
            }
            oTable._mTimeouts._cleanupACCExtension = jQuery.sap.delayedCall(100, this, function() {
                var oTable = this.getTable();
                if (!oTable) {
                    return;
                }
                this._iLastRowNumber = null;
                this._iLastColumnNumber = null;
                ExtensionHelper.cleanupCellModifications(this);
                oTable._mTimeouts._cleanupACCExtension = null;
            });
        }
    });

    /*
     * Known element types (DOM areas) in the table
     * @see TableAccRenderExtension.writeAriaAttributesFor
     * @public (Part of the API for Table control only!)
     */
    TableAccExtension.ELEMENTTYPES = {
        DATACELL :          TableUtils.CELLTYPES.DATACELL,          // @see TableUtils.CELLTYPES
        COLUMNHEADER :      TableUtils.CELLTYPES.COLUMNHEADER,      // @see TableUtils.CELLTYPES
        ROWHEADER :         TableUtils.CELLTYPES.ROWHEADER,         // @see TableUtils.CELLTYPES
        COLUMNROWHEADER :   TableUtils.CELLTYPES.COLUMNROWHEADER,   // @see TableUtils.CELLTYPES
        ROOT :              "ROOT",                                 // The tables root dom element
        CONTENT:            "CONTENT",                              // The content area of the table which contains all the table elements, rowheaders, columnheaders, etc
        TABLE :             "TABLE",                                // The "real" table element(s)
        TABLEHEADER :       "TABLEHEADER",                          // The table header area
        TABLEFOOTER :       "TABLEFOOTER",                          // The table footer area
        TABLESUBHEADER :    "TABLESUBHEADER",                       // The table toolbar and extension areas
        COLUMNHEADER_ROW :  "COLUMNHEADER_ROW",                     // The area which contains the column headers (TableUtils.CELLTYPES.COLUMNHEADER)
        ROWHEADER_COL :     "ROWHEADER_COL",                        // The area which contains the row headers (TableUtils.CELLTYPES.ROWHEADER)
        TH :                "TH",                                   // The "technical" column headers
        SELECTALL_COLHEADER : "SELECTALL_COLHEADER",                // The select-all column header
        ROWHEADER_TD :      "ROWHEADER_TD",                         // The "technical" row headers
        TR :                "TR",                                   // The rows
        TREEICON :          "TREEICON",                             // The expand/collapse icon in the TreeTable
        NODATA :            "NODATA",                               // The no data container
        OVERLAY :           "OVERLAY",                              // The overlay container
        COLUMN_RSZ :        "COLUMN_RSZ",                           // The column resizer
        PRESENTATION :      "PRESENTATION"                          // Any structural element that should not impact screen readers
    };

    /*
     * Returns whether acc mode is switched on ore not.
     * @public (Part of the API for Table control only!)
     */
    TableAccExtension.prototype.getAccMode = function() {
        return this._accMode;
    };

    /*
     * Determines the current focused cell and modifies the labels and descriptions if needed.
     * @public (Part of the API for Table control only!)
     */
    TableAccExtension.prototype.updateAccForCurrentCell = function(bOnCellFocus) {
        if (!this._accMode || !this.getTable()._getItemNavigation()) {
            return;
        }

        var oTable = this.getTable();

        if (oTable._mTimeouts._cleanupACCFocusRefresh) {
            jQuery.sap.clearDelayedCall(oTable._mTimeouts._cleanupACCFocusRefresh);
            oTable._mTimeouts._cleanupACCFocusRefresh = null;
        }

        if (bOnCellFocus) {
            ExtensionHelper.cleanupCellModifications(this);
        }

        var oInfo = ExtensionHelper.getInfoOfFocusedCell(this);
        if (!oInfo || !oInfo.cell || !oInfo.type || !ExtensionHelper["modifyAccOf" + oInfo.type]) {
            return;
        }

        if (!bOnCellFocus) {
            // we will use Narrator to announce cell context and content, so keep the cell attributes simple to avoid
            //  confusing JAWS
            if (oInfo.type === TableUtils.CELLTYPES.DATACELL || TableUtils.CELLTYPES.ROWHEADER) {
                oInfo.cell.attr("aria-hidden", "true"); //hide so as not to confuse the screen reader
            }
            return;
        }

        ExtensionHelper["modifyAccOf" + oInfo.type].apply(this, [oInfo.cell, bOnCellFocus]);
    };

    /*
     * Is called by the Column whenever the sort or filter state is changed and updates the corresponding
     * ARIA attributes.
     * @public (Part of the API for Table control only!)
     */
    TableAccExtension.prototype.updateAriaStateOfColumn = function(oColumn, $Ref) {
        if (!this._accMode) {
            return;
        }

        var mAttributes = ExtensionHelper.getAriaAttributesFor(this, TableAccExtension.ELEMENTTYPES.COLUMNHEADER, {
            headerId: oColumn.getId(),
            column: oColumn,
            index: oColumn.getIndex(),
            label: oColumn.getLabel()
        });

        $Ref = $Ref ? $Ref : oColumn.$();

        $Ref.attr({
            "aria-sort" : mAttributes["aria-sort"] || null,
            "aria-labelledby" : mAttributes["aria-labelledby"] ? mAttributes["aria-labelledby"].join(" ") : null
        });
    };

    /*
     * Is called by the Row whenever the selection state is changed and updates the corresponding
     * ARIA attributes.
     * @public (Part of the API for Table control only!)
     */
    TableAccExtension.prototype.updateAriaStateOfRow = function(oRow, $Ref, bIsSelected) {
        if (!this._accMode) {
            return;
        }

        if (!$Ref) {
            $Ref = oRow.getDomRefs(true);
        }

        if ($Ref.row) {
            $Ref.row.children("td").add($Ref.row).attr("aria-selected", bIsSelected ? "true" : null);
        }
    };

    /*
     * Updates the expand state and level for accessibility in case of grouping
     * @public (Part of the API for Table control only!)
     */
    TableAccExtension.prototype.updateAriaExpandAndLevelState = function(oRow, $ScrollRow, $RowHdr, $FixedRow, bGroup, bExpanded, iLevel, $TreeIcon) {
        if (!this._accMode) {
            return;
        }

        var sTitle = null,
            oTable = this.getTable(),
            aRefs = [$ScrollRow, $ScrollRow.children(), $RowHdr, $FixedRow, $FixedRow ? $FixedRow.children() : null],
            bTreeMode = !!$TreeIcon,
            oBinding = oTable.getBinding("rows");

        if (!bGroup && $RowHdr && !bTreeMode) {
            var iIndex = $RowHdr.attr("data-sap-ui-rowindex");
            var mAttributes = ExtensionHelper.getAriaAttributesFor(this, TableAccExtension.ELEMENTTYPES.ROWHEADER, {
                rowSelected: !oRow._bHidden && oTable.isIndexSelected(iIndex),
                index: oRow.getIndex()
            });
            sTitle = mAttributes["title"] || null;
        }

        if ($RowHdr && !bTreeMode) {
            $RowHdr.attr({
                "aria-haspopup" : bGroup ? "true" : null,
                "title" : sTitle
            });
        }

        if (oBinding && oBinding.hasTotaledMeasures && iLevel > 0 && (!oBinding.bProvideGrandTotals || !oBinding.hasTotaledMeasures())) {
            // Summary top-level row is not displayed (always has level 0) -> for aria we can shift all the levels 1 step up;
            iLevel = iLevel - 1;
        }

        for (var i = 0; i < aRefs.length; i++) {
            if (aRefs[i]) {
                aRefs[i].attr({
                    "aria-expanded" : bGroup ? bExpanded + "" : null,
                    "aria-level": iLevel < 0 ? null : (iLevel + 1)
                });
            }
        }

        if (bTreeMode) {
            $TreeIcon.attr(ExtensionHelper.getAriaAttributesFor(this, TableAccExtension.ELEMENTTYPES.TREEICON, {row: oRow}));
        }
    };

    /*
     * Updates the relevant aria-properties in case of overlay or noData is set / reset.
     * @public (Part of the API for Table control only!)
     */
    TableAccExtension.prototype.updateAriaStateForOverlayAndNoData = function() {
        var oTable = this.getTable();

        if (!oTable || !oTable.getDomRef() || !this._accMode) {
            return;
        }

        if (oTable.getShowOverlay()) {
            oTable.$().find("[data-sap-ui-table-acc-covered*='overlay']").attr("aria-hidden", "true");
        } else {
            oTable.$().find("[data-sap-ui-table-acc-covered*='overlay']").removeAttr("aria-hidden");
            if (TableUtils.isNoDataVisible(oTable)) {
                oTable.$().find("[data-sap-ui-table-acc-covered*='nodata']").attr("aria-hidden", "true");
            } else {
                oTable.$().find("[data-sap-ui-table-acc-covered*='nodata']").removeAttr("aria-hidden");
            }
        }
    };

    /*
     * Retrieve Aria descriptions from resource bundle for a certain selection mode
     * @param {Boolean} [bConsiderSelectionState] set to true if the current selection state of the table shall be considered
     * @param {String} [sSelectionMode] optional parameter. If no selection mode is set, the current selection mode of the table is used
     * @returns {{mouse: {rowSelect: string, rowDeselect: string}, keyboard: {rowSelect: string, rowDeselect: string}}}
     * @public (Part of the API for Table control only!)
     */
    TableAccExtension.prototype.getAriaTextsForSelectionMode = function (bConsiderSelectionState, sSelectionMode) {
        var oTable = this.getTable();

        if (!sSelectionMode) {
            sSelectionMode = oTable.getSelectionMode();
        }

        var oResBundle = oTable._oResBundle;
        var mTooltipTexts = {
            mouse: {
                rowSelect: "",
                rowDeselect: ""
            },
            keyboard: {
                rowSelect: "",
                rowDeselect: ""
            }
        };

        var bSomeRowIsSelected = oTable._isSomeRowSelected();

        if (sSelectionMode === SelectionMode.Single) {
            mTooltipTexts.mouse.rowSelect = oResBundle.getText("TBL_ROW_SELECT");
            mTooltipTexts.mouse.rowDeselect = oResBundle.getText("TBL_ROW_DESELECT");
            mTooltipTexts.keyboard.rowSelect = oResBundle.getText("TBL_ROW_SELECT_KEY");
            mTooltipTexts.keyboard.rowDeselect = oResBundle.getText("TBL_ROW_DESELECT_KEY");
        } else if (sSelectionMode === SelectionMode.MultiToggle) {
            mTooltipTexts.mouse.rowSelect = oResBundle.getText("TBL_ROW_SELECT_MULTI_TOGGLE");
            // text for de-select is the same like for single selection
            mTooltipTexts.mouse.rowDeselect = oResBundle.getText("TBL_ROW_DESELECT");
            mTooltipTexts.keyboard.rowSelect = oResBundle.getText("TBL_ROW_SELECT_MULTI_TOGGLE_KEY");
            // text for de-select is the same like for single selection
            mTooltipTexts.keyboard.rowDeselect = oResBundle.getText("TBL_ROW_DESELECT_KEY");

            if (bConsiderSelectionState === true && !bSomeRowIsSelected) {
                // if there is no row selected yet, the selection is like in single selection case
                mTooltipTexts.mouse.rowSelect = oResBundle.getText("TBL_ROW_SELECT");
                mTooltipTexts.keyboard.rowSelect = oResBundle.getText("TBL_ROW_SELECT_KEY");
            }
        }

        return mTooltipTexts;
    };

    /**
     * surface the ACCInfoHelper for targeted QUnit testing
     * @returns {Object} the ACCInfoHelper utility class
     * @private
     */
    TableAccExtension.prototype._getACCInfoHelper = function() {
        return ACCInfoHelper;
    };

    /**
     * Gathers information from the given parameters and has Narrator announce a coherent message describing to the user
     * the current contents and context of their position in the table.
     *
     * @param {Object} oFocusedCellInfo an object from TableUtils.getFocusedItemInfo about the cell to describe
     * @param {Object} oFromPosition an object from TableUtils.getTablePositionFromFocusInfo about the previous location in the table (if defined)
     * @param {Object} [oTriggeringEvent] an optional event that caused the narration to be needed
     * @returns {boolean} true if a message was passed to Narrator, false otherwise
     */
    TableAccExtension.prototype.announceCellNavigation = function(oFocusedCellInfo, oFromPosition, oTriggeringEvent) {
        var oTable = this.getTable(),
            bDataCell = false,
            bColHeaderCell = false,
            //default will be to assume that both row and column should be announced
            bAnnounceRow = true,
            bAnnounceColumn = true,
            bShowingSummaryHeader,
            iHeaderCount, //how many column header rows in table
            iRowHeaderCount, //selection row header or not?
            iVColIndex, //visible col index
            iRowIndex,  //logical row index
            oRow, oColumn, oCell, aVisibleColumns, sMessage = null;

        if (!oTable) {
            // there's really not much to do without a table reference
            return;
        }

        // find the actual values from Table
        iHeaderCount = TableUtils.getHeaderRowCount(oTable);
        iRowHeaderCount = (TableUtils.hasRowHeader(oTable) ? 1 : 0);
        bShowingSummaryHeader = oTable.shouldShowSummaryHeader();

        if (!oFocusedCellInfo) {
            oFocusedCellInfo = TableUtils.getFocusedItemInfo(oTable);
        }

        if (bShowingSummaryHeader) {
            iHeaderCount += 2;
        }

        // check the available cell info to see if we are in the data cell or column header parts of the table
        if (oFocusedCellInfo) {
            //check if we are to the right of the row header
            if (oFocusedCellInfo.cellInRow >= iRowHeaderCount) {
                //check if this is a data cell
                bDataCell = (oFocusedCellInfo.row >= iHeaderCount);
                //we must be in the header if not the data
                bColHeaderCell = !bDataCell;

                //find column index and column
                iVColIndex = oFocusedCellInfo.cellInRow - iRowHeaderCount;
                aVisibleColumns = oTable._getVisibleColumns();
                oColumn = aVisibleColumns[iVColIndex];
            }
        }

        // we will only create narration for data cells for the moment
        if (bDataCell) {
            //  - find row
            oRow = oTable.getRows()[oFocusedCellInfo.row - iHeaderCount];
            // S1478518: The table may have changed the # of rows since cell focus was last assigned, leaving outside valid
            //  range of available rows - simply avoid announcing anything unless focus is on a determinable cell
            if (!oRow) {
                return false;   // no announcement made
            }

            iRowIndex = oRow.getIndex();

            //  - find cell for column in row
            oCell = oRow.getCells()[iVColIndex];

            //keyboard navigation may leave off announcing either row or column
            if (oTriggeringEvent && (oTriggeringEvent.type !== 'focusin' && oTriggeringEvent.type !== 'mousedown')) {
                if (oFromPosition) {
                    bAnnounceRow = (iRowIndex !== oFromPosition.rowIndex);
                    bAnnounceColumn = (oColumn !== oFromPosition.column);
                }
            }

            // === now build the message ===
            sMessage = this.createNarratorMessage(oCell, oRow, oColumn, bAnnounceRow, bAnnounceColumn);
        } else if (bColHeaderCell) {
            //no row instance in the column headers, but we can use row index
            oRow = null;
            iRowIndex = oFocusedCellInfo.row;
            //attempt to locate column header cell control
            if (bShowingSummaryHeader) {
                //focus is on either the summary label or summary control
                if (iRowIndex === 0 && oColumn.getSummaryLabel) {
                    oCell = oColumn.getSummaryLabel();
                } else if (iRowIndex === 1 && oColumn.getSummaryControl) {
                    oCell = oColumn.getSummaryControl();
                } else {
                    // leave oCell unassigned?
                }
            }

            //keyboard navigation may leave off announcing either row or column
            if (oTriggeringEvent) {
                if (oTriggeringEvent.type !== 'focusin' && oTriggeringEvent.type !== 'mousedown' && !!oFromPosition) {
                    bAnnounceRow = (iRowIndex !== oFromPosition.rowIndex);
                }
            }

            // === now build the message ===
            sMessage = this.createNarratorMessageForColumnHeader(oColumn, oCell, iRowIndex, bAnnounceRow);
        }

        //if we have a message, then pass it to the Narrator
        if (sMessage !== null) {
            Narrator.addTextNow(sMessage); //addTextNow circumvents some of JAWS's random chattiness
            return true;
        }

        return false;   //maybe some value to know if announcement was made or not?
    };

    /**
     * Generate and return text describing the selection state of the given cell, row, or column.
     * @param {sap.ui.core.Control} oCell the cell to describe
     * @param {sas.hc.ui.table.Row} oRow the row that contains the cell
     * @param {sas.hc.ui.table.Column} oColumn the column that contains the cell
     * @returns {string|null} the message or null if no message is needed
     * @private
     */
    TableAccExtension.prototype._createSelectionStateNarration = function(oCell, oRow, oColumn) {
        var oTable = this.getTable(),
            sText = null;

        if (oTable.getSelectionMode() !== SelectionMode.None) {
            if (oRow && oTable.isIndexSelected(oRow.getIndex())) {
                sText = oTable._oResBundle.getText("TBL_ROW_DESC_SELECTED");
            }
        }

        return sText;
    };

    /**
     * Generate and return text describing the label (or labels) for the given column.
     * @param {sas.hc.ui.table.Column} oColumn the column to describe
     * @returns {string|null} the message or null if no message is needed
     * @private
     */
    TableAccExtension.prototype._createColumnLabelNarration = function(oColumn) {
        var oTable = this.getTable(),
            aMessageParts = [],
            aMultiLabels,
            oInfo,
            sText = null;

        if (oColumn) {
            //determine column header and add it here
            // gather multi-labels if used
            aMultiLabels = oColumn.getMultiLabels();

            if (aMultiLabels && aMultiLabels.length > 0) {
                //if there are multi-labels, add those into the set
                for (var i = 0; i < aMultiLabels.length; i++) {
                    oInfo = ACCInfoHelper.getAccInfoOfControl(aMultiLabels[i], oTable._oResBundle);
                    if (oInfo && oInfo.description) {
                        aMessageParts.push(oInfo.description);
                    }
                }
            } else {
                //add the column label
                oInfo = ACCInfoHelper.getAccInfoOfControl(oColumn.getLabel(), oTable._oResBundle);
                if (oInfo && oInfo.description) {
                    aMessageParts.push(oInfo.description);
                }
            }

            sText = aMessageParts.join(" ");
        }

        return sText;
    };

    /**
     * Generate and return text describing the given cell.
     * @param {sap.ui.core.Control} oCell the cell to describe
     * @returns {string|null} the message or null if no message is needed
     * @private
     */
    TableAccExtension.prototype._createCellNarration = function(oCell) {
        var oTable = this.getTable(),
            oInfo, sText = null;

        if (oCell) {
            //determine cell data text and add it here
            oInfo = ACCInfoHelper.getAccInfoOfControl(oCell, oTable._oResBundle);
            if (oInfo && oInfo.description) {
                sText = oInfo.description;
            }
        }

        return sText;
    };

    /**
     * Generate and return the narration message for the status of the given cell within the table.  Status elements
     * include whether the row and/or column is frozen and if the column is filtered or sorted.
     * @param {sap.ui.core.Control} oCell the cell being described
     * @param {sas.hc.ui.table.Column} oColumn the column that contains the given cell
     * @param {sas.hc.ui.table.Row} oRow the row that contains the given cell
     * @returns {string|null} the message to pass to the narrator describing the status of the cell
     * @private
     */
    TableAccExtension.prototype._createCellStatusNarration = function(oCell, oColumn, oRow) {
        var oTable = this.getTable(),
            aMessageParts = [],
            sText = null;

        if (oColumn) {
            if (oTable.isFrozenColumn(oColumn)) {
                aMessageParts.push(oTable._oResBundle.getText("TBL_FIXED_COLUMN"));
            }
            if (oColumn.getSorted && oColumn.getSorted()) {
                if (oColumn.getSortOrder && oColumn.getSortOrder() === SortOrder.Ascending) {
                    aMessageParts.push(oTable._oResBundle.getText("TBL_COL_DESC_SORTED_ASC"));
                } else {
                    aMessageParts.push(oTable._oResBundle.getText("TBL_COL_DESC_SORTED_DES"));
                }
            }
            if (oColumn.getFiltered && oColumn.getFiltered()) {
                aMessageParts.push(oTable._oResBundle.getText("TBL_COL_DESC_FILTERED"));
            }
            if (aMessageParts.length > 0) {
                sText = aMessageParts.join(" ");
            }
        }

        return sText;
    };

    /**
     * Generate and return text describing the position of the row within the table.
     * @param {sas.hc.ui.table.Row} oRow the row to describe
     * @returns {string|null} the message or null if no message is needed
     * @private
     */
    TableAccExtension.prototype._createRowPositionNarration = function(oRow) {
        var oTable = this.getTable(),
            iCount = oTable._getRowCount(),
            sText = null;

        if (oRow) {
            sText = oTable._oResBundle.getText("TBL_ROW_ROWCOUNT.fmt", [(oRow.getIndex()+1), iCount]);
        }

        return sText;
    };


    /**
     * Generate and return text describing the position of the column within the table.
     * @param {sas.hc.ui.table.Column} oColumn the column to describe
     * @returns {string|null} the message or null if no message is needed
     * @private
     */
    TableAccExtension.prototype._createColumnPositionNarration = function(oColumn) {
        var oTable = this.getTable(),
            aVisibleColumns = oTable._getVisibleColumns(),
            iColIndex = aVisibleColumns.indexOf(oColumn),
            sText = null;

        if (oColumn) {
            //the column index and total column count are based on the set of visible columns (disregarding any hidden ones)
            sText = oTable._oResBundle.getText("TBL_COL_COLCOUNT.fmt", [(iColIndex+1), aVisibleColumns.length]);
        }

        return sText;
    };

    /**
     * Generate and return text describing the cell, row, and/or column based on the given parameters.
     * 1. 'Row selected', if entering a new row and row has been selected
     * 2. Context changes column/row headers as appropriate
     * 3. Cell data
     * 4. Role (if a header)
     * 5. Other info (frozen, sorted asc/desc, filtered)
     * 6. Column # and/or row number (whichever has changed)
     * @param {sas.hc.ui.table.Column} oCell the cell to describe
     * @param {sas.hc.ui.table.Row} oRow the row to describe
     * @param {sas.hc.ui.table.Column} oColumn the column to describe
     * @param {boolean} bAnnounceRow true/false whether information about the row should be included in the message
     * @param {boolean} bAnnounceColumn true/false whether information about the column should be included in the message
     * @returns {string|null} the message or null if no message is needed
     */
    TableAccExtension.prototype.createNarratorMessage = function(oCell, oRow, oColumn, bAnnounceRow, bAnnounceColumn) {
        var aMessageParts = [],
            sMessage = null,
            sText;

        //1. Selection state
        sText = this._createSelectionStateNarration(oCell, oRow, oColumn);
        if (sText !== null) {   //only add message part if it was non-null/trivial
            aMessageParts.push(sText);
        }

        //2. Header Context changes
        if (bAnnounceColumn) {
            sText = this._createColumnLabelNarration(oColumn);
            if (sText !== null) {
                aMessageParts.push(sText);
            }
        }

        //3. Cell data
        aMessageParts.push(this._createCellNarration(oCell));

        //5. Status elements - frozen, sorted asc/desc, filtered
        //announce column specific state only on column change
        if (bAnnounceColumn) {
            sText = this._createCellStatusNarration(oCell, oColumn, oRow);
            // only add the text if this cell has a non-trivial status narration
            if (sText !== null) {
                aMessageParts.push(sText);
            }
        }

        //6. Column # and/or row number (whichever has changed)
        if (bAnnounceRow) {
            aMessageParts.push(this._createRowPositionNarration(oRow));
        }
        // Column X of Y
        if (bAnnounceColumn) {
            aMessageParts.push(this._createColumnPositionNarration(oColumn));
        }

        //build a complete message if any of the above resulted in narration text
        if (aMessageParts.length > 0) {
            //join everything together with spaces (an empty array returns an empty string)
            sMessage = aMessageParts.join(" ");
        }

        return sMessage;
    };

    /**
     * Generate and return text describing the column header based on the given parameters.
     * @param {sas.hc.ui.table.Column} oColumn the column to describe
     * @param {sas.hc.ui.table.Column} oCell the cell to describe
     * @param {int} iRowIndex the index of the column header row to describe
     * @param {boolean} bAnnounceRow true/false whether information about the row should be included in the message
     * @returns {string|null} the message or null if no message is needed
     */
    TableAccExtension.prototype.createNarratorMessageForColumnHeader = function(oColumn, oCell, iRowIndex, bAnnounceRow) {
        var oTable = this.getTable(),
            aMessageParts = [],
            sMessage = null,
            sText;

        //2. Header Context changes
        sText = this._createColumnLabelNarration(oColumn);
        if (sText !== null) {
            aMessageParts.push(sText);
        }

        //3. Cell content
        if (!!oCell) {
            aMessageParts.push(this._createCellNarration(oCell));
        }

        //4. Role (Header)
        sText = oTable._oResBundle.getText("TBL_COL_HEADER_LABEL");
        aMessageParts.push(sText);

        //5. Status elements - frozen, sorted asc/desc, filtered
        //announce column specific state only on column change
        sText = this._createCellStatusNarration(oCell, oColumn, null);
        // only add the text if this cell has a non-trivial status narration
        if (sText !== null) {
            aMessageParts.push(sText);
        }

        // Column X of Y
        aMessageParts.push(this._createColumnPositionNarration(oColumn));

        //build a complete message if any of the above resulted in narration text
        if (aMessageParts.length > 0) {
            //join everything together with spaces (an empty array returns an empty string)
            sMessage = aMessageParts.join(" ");
        }

        return sMessage;
    };

    return TableAccExtension;

});
