// (c) 2016, SAS Institute Inc.
sap.ui.define([
    "jquery.sap.global",
    "sas/hc/m/CheckBox",
    "sas/hc/ui/table/TableUtils",
    "sap/ui/core/Core"
], function(jQuery, CheckBox, TableUtils, sapCore) {
    "use strict";

    /**
     * Constructor for a new SelectAllControl.
     * This is an extension a sas.hc.m.CheckBox but at the
     * sap layer was just a simple icon.
     *
     * @class
     *
     * ## Features: ##
     *
     * * upon invalidation, queries parent Table to determine state
     * * manages tooltip depending on state
     *
     * @name sas.hc.ui.table.SelectAllControl
     * @extends sas.hc.m.CheckBox
     * @version 904001.11.16.20251118090100_f0htmcm94p
     * @author Jonathan Brink
     *
     * @constructor
     * @public
     */
    var SelectAllControl = CheckBox.extend("sas.hc.ui.table.SelectAllControl", {
        metadata: {
            library: "sas.hc.ui.table",
            associations: {
                // the parent sas.hc.ui.table.Table that aggregates this control
                table: {type: "sas.hc.ui.table.Table"}
            }
        },
        renderer: {
        }
    });

    /**
     * init lifecycle callback
     * @private
     */
    SelectAllControl.prototype.init = function() {
        if (CheckBox.prototype.init) {
            CheckBox.prototype.init.apply(this, arguments);
        }
        this.addStyleClass("sasSelectAllControl");
    };

    SelectAllControl.prototype.exit = function() {
        var oTable = sap.ui.getCore().byId(this.getTable());
        if (oTable) {
            oTable.detachFilter(this._onTableFilter, this);
            oTable.detachEvent("_rowsUpdated", this._onTableRowsUpdate, this);
        }

        if (CheckBox.prototype.exit) {
            CheckBox.prototype.exit.apply(this, arguments);
        }
    };

    /**
     * onAfterRendering lifecycle callback
     * @private
     */
    SelectAllControl.prototype.onAfterRendering = function() {
        if (CheckBox.prototype.onAfterRendering) {
            CheckBox.prototype.onAfterRendering.apply(this, arguments);
        }
        this.updateRelativeToTable();
    };

    /**
     * tap event handler (also accounts for "click")
     * @private
     */
    SelectAllControl.prototype.ontap = function() {
        if (CheckBox.prototype.ontap) {
            CheckBox.prototype.ontap.apply(this, arguments);
        }

        this._updateTableSelection();
    };

    /**
     * "selected" property override.
     * Manages tooltip.
     * @param {boolean} bSelected Whether checkbox is selected
     * @private
     */
    SelectAllControl.prototype.setSelected = function(bSelected) {
        var rb = sap.ui.getCore().getLibraryResourceBundle("sas.hc.ui.table");

        if (CheckBox.prototype.setSelected) {
            CheckBox.prototype.setSelected.apply(this, arguments);
        }

        if (bSelected === true) {
            this.setTooltip(rb.getText("TBL_DESELECT_ALL.txt"));
        } else {
            this.setTooltip(rb.getText("TBL_SELECT_ALL.txt"));
        }
    };

    /**
     * "table" association override.
     * @param {object} oTable The parent Table
     * @private
     */
    SelectAllControl.prototype.setTable = function(oTable) {
        this.setAssociation("table", oTable);

        // update state when Table's filter event is emitted
        oTable.attachFilter(this._onTableFilter, this);

        // update state when Table's _rowsUpdated event is emitted
        oTable.attachEvent("_rowsUpdated", this._onTableRowsUpdate, this);
    };

    /**
     * Handle the Table's filter event
     * @private
     */
    SelectAllControl.prototype._onTableFilter = function() {
        // the Table is not done "filtering" when this event fires, so delay a little
        var self = this;
        setTimeout(function() {
            self.updateRelativeToTable();
        }, 100);
    };

    /**
     * Handle the Table's _rowsUpdated event.
     * @private
     */
    SelectAllControl.prototype._onTableRowsUpdate = function() {
        this.updateRelativeToTable();
    };

    /**
     * toggle "selected" property
     * @private
     */
    SelectAllControl.prototype.toggleSelected = function() {
        this.setSelected(!!!this.getSelected());
        this._updateTableSelection();
    };

    /**
     * Call to Table to alter its selection.
     * @private
     */
    SelectAllControl.prototype._updateTableSelection = function() {
        var oTable = sap.ui.getCore().byId(this.getTable());
        if (this.getSelected() === true) {
            oTable.selectAll();
        } else {
            oTable.clearSelection(true);
        }
    };

    /**
     * Set properties based off of Table state
     * @public
     */
    SelectAllControl.prototype.updateRelativeToTable = function() {
        var oTable = sap.ui.getCore().byId(this.getTable()),
            oRowsBinding, bHasVisibleColumns, bHasRows = false;

        if (oTable === undefined) {
            return;
        }

        oRowsBinding = oTable.getBinding('rows');
        bHasVisibleColumns = TableUtils.getVisibleColumnCount(oTable) > 0;

        if (oRowsBinding !== undefined && jQuery.isFunction(oRowsBinding.getLength) === true) {
            bHasRows = oRowsBinding.getLength() > 0;
        }

        function determineVisibility() {
            return oTable._useSelectionControls() === true &&
                   bHasVisibleColumns === true &&
                   bHasRows === true;
        }

        this.setVisible(determineVisibility());

        if (bHasVisibleColumns === false || bHasRows === false) {
            this.setSelected(false);
        } else {
            this.setSelected(oTable.isEveryIndexSelected());
        }

        //after a *short* delay, make sure the table's item navigation array is in sync as we may have rerendered
        jQuery.sap.delayedCall(50, this, this._updateTableItemNavigation);
    };

    /**
     * @private
     */
    SelectAllControl.prototype._updateTableItemNavigation = function() {
        var oTable = sap.ui.getCore().byId(this.getTable()),
            oNav,
            i, $itemNavDomRef,
            // if any of the ItemNavigation dom nodes needed to be updated
            bChangeMade = false,
            // the existing ItemNavigation dom nodes (in correct order)
            aItemNavDomRefs,
            $self = this.$();

        if (!!oTable) {     //it's possible we don't have a valid table ID
            oNav = oTable._getKeyboardExtension()._itemNavigation;
        }

        if (!!!oNav || $self.length < 1) {  //it's possible the table's nav array isn't ready or our DOM node isn't available
            return;
        }

        //get the navigation array
        aItemNavDomRefs = oNav.aItemDomRefs;

        // iterate through the ItemNavigation dom refs,
        // looking for 'select all' refs, and updating them if any are found
        for (i = 0; i < aItemNavDomRefs.length; i++) {
            $itemNavDomRef = jQuery(aItemNavDomRefs[i]);

            // if an item is a select all control, make sure it is *this* select all control
            if ($itemNavDomRef.hasClass('sasSelectAllControl') === true || $itemNavDomRef.hasClass('sapUiTableSelAll') === true) {
                aItemNavDomRefs[i] = $self.get(0);
                bChangeMade = true;
            }
        }

        if (bChangeMade === true) {
            oNav.setItemDomRefs(aItemNavDomRefs);
        }
    };

    return SelectAllControl;
}, /* bExport= */ true);
