//@sasPreCore - Annotation to signal build to merge this file into preCore.js.
sap.ui.define(
    ["jquery.sap.global", "./BootTask", "./util", "./CommonsDialogDisplayHelper",'sas/hc/ui/core/Core'],
    function(jQuery, BootTask, util, CommonsDialogDisplayHelper) {
        var TimeoutHandlerTask = BootTask.extend("sas.hc.ui.core.TimeoutHandlerTask", {
            name: "sas.hc.ui.core.TimeoutHandlerTask",
            dependsOn: ["sas.hc.ui.core.RegistryTask"],

            constructor: function() {
                this.inactiveInterval = sas.config.getSessionMaxInActiveInterval();
                // if its undefined or negative or not a number then disable session time out.
                if (isNaN(this.inactiveInterval) || this.inactiveInterval < 0) {
                    // timeout feature not possible.
                    this.disabled = true;
                    return;
                }

                this.warningDuration = sas.config.getSessionTimeoutWarningDuration();
                // don't entertain negative values.
                this.warningDuration = this.warningDuration > 0 ? this.warningDuration : 0;
                this.appReady = false;

                this.actualInterval = this.inactiveInterval - this.warningDuration;
                // reset warning duration if inactive interval is less than warning duration.
                if (this.inactiveInterval < this.warningDuration) {
                    this.warningDuration = this.inactiveInterval;
                }

                this.altTimeoutActivityInterval = sas.config.get('altTimeoutActivityInterval');
                if (this.altTimeoutActivityInterval === -1){
                    sas.log.trace('sas.hc.ui.core.TimeoutHandlerTask',"#constructor", "AltTimeout disabled with -1 config setting.");
                    this.altTimeoutActivityInterval = undefined;
                }

            }
        });
        TimeoutHandlerTask.prototype.run = function() {
            if (this.disabled) return;
            var that = this;
            Object.defineProperty(sas, "timeoutManager", {
                enumerable: true,
                value: Object.freeze({
                    events: Object.freeze({
                        TIMED_OUT: "hc.sessionTimedOut",
                        USER_ACTIVE: "hc.userSessionActive",
                        KEEP_ALIVE: "hc.keepSessionAlive",
                        BEFORE_TIME_OUT: "hc.beforeSessionTimeOut"
                    }),
                    start: function(silentStart) {
                        that.start(silentStart);
                    },
                    quiet: function() {
                        that.quiet();
                    },
                    awake: function() {
                        that.awake();
                    },
                    keepAlive: function(bVal) {
                        that.keepAlive(bVal);
                    }
                })
            });
            sas.eventBus.subscribe(sas.timeoutManager.events.KEEP_ALIVE, function() {
                util.pingServer();
                that.recordUserActivity();
            });

            sas.eventBus.subscribe(sas.timeoutManager.events.USER_ACTIVE, function() {
                that.recordUserActivity();
            });

            // make sure if app switcher is enabled that it is not the outer app switcher when starting timeout
            var isMainApp = window === top ? true : false;
            if (
                (sas.config.getApplicationSwitcherEnabled() === true && isMainApp === false) ||
                sas.config.getApplicationSwitcherEnabled() === false
            ) {
                util.startTimeoutManager();
            }

            var callback = function() {
                that.appReady = true;
                sas.timeoutManager.start();
                sas.eventBus.unsubscribe(util.APP_READY, callback);
            };

            sas.eventBus.subscribe(util.APP_READY, callback);
        };

        TimeoutHandlerTask.prototype.pause = function() {
            sas.log.trace("sas.hc.ui.core.TimeoutHandlerTask", "#pause", "Timeout manager paused.");
            clearTimeout(this.sessionTimeoutRef);
            this.sessionTimeoutRef = undefined;
            this._paused = true;
        };

        /**
         * Check for user activity on regular interval and ping server if activity within that window.
         *
         */
        TimeoutHandlerTask.prototype.altTimeoutCheckForPing=function(){
            if(this.isPaused() && !this._keepAlive){
                // session dialog is displayed.
                return;
            }

            var that=this;
            //reset timer
            clearTimeout(this.altTimeoutRef);
            this.altTimeoutRef = undefined;
            sas.log.trace('sas.hc.ui.core.TimeoutHandlerTask',"#altTimeoutCheckForPing","");
            if (this.altTimeoutUserActivity){
                sas.log.trace('sas.hc.ui.core.TimeoutHandlerTask',"#altTimeoutCheckForPing","user active in this interval so call ping");
                this.ping();
                this.altTimeoutUserActivity = false;
                //restart default checks if activity
                this.resume();
            }
            else{
                sas.log.trace('sas.hc.ui.core.TimeoutHandlerTask',"#altTimeoutCheckForPing","user not active in this interval");
            }
            
            this.altTimeoutRef = setTimeout(function() {
                that.altTimeoutCheckForPing();
            }, this.altTimeoutActivityInterval * 1000 /* convert into mili sec*/);
        };


        /**
         * Resume user activity and timeout dialog from where it was paused.
         *
         */
        TimeoutHandlerTask.prototype.resume = function(timeLapsedInSec) {
            this.pause(); // pause to clear timeout reference.
            if (typeof this.sessionTimeoutRef === "undefined") {
                var that = this;
                var interval = this.actualInterval;
                if (timeLapsedInSec !== undefined){
                    // consider time lapsed after last user activity to setup next interval
                    // instead of fixed checker intervals.
                    interval = interval - timeLapsedInSec;
                }
                this.sessionTimeoutRef = setTimeout(function() {
                    that.checkForTimeout();
                }, interval * 1000 /* convert into mili sec*/);
                sas.log.trace(
                    "sas.hc.ui.core.TimeoutHandlerTask",
                    "#resume",
                    "Timeout manager resumed with interval",
                    interval
                );
            }
            this._paused = false;
        };
        TimeoutHandlerTask.prototype.isPaused = function() {
            return this._paused;
        };
        TimeoutHandlerTask.prototype.recordUserActivity = function() {
            if (this.isPaused() && !this._keepAlive) {
                return;
            }

            //alt timeout fix for HTMLCOMMONS-12412
            if(this.altTimeoutActivityInterval) {
                sas.log.trace('sas.hc.ui.core.TimeoutHandlerTask',"#recordUserActivity","set altTimeoutUserActivity to true");
                this.altTimeoutUserActivity = true;
            }

            this.lastActTime = Math.floor(Date.now() / 1000); // in seconds
            sas.log.trace(
                "sas.hc.ui.core.TimeoutHandlerTask",
                "#recordUserActivity",
                "User activity recorded."
            );
        };
        TimeoutHandlerTask.prototype.logoff = function() {
            // log off.
            util.signout(); //TODO should do it over an event?.
        };

        TimeoutHandlerTask.prototype.timeout = function() {
            // before log off.
            sas.eventBus.publish(sas.timeoutManager.events.BEFORE_TIME_OUT);
            // log off.
            // publish an event on the event bus that session is timed out.
            sas.eventBus.publish(sas.timeoutManager.events.TIMED_OUT);
        };

        TimeoutHandlerTask.prototype.ping = function() {
            sas.log.trace("sas.hc.ui.core.TimeoutHandlerTask", "#ping", "Ping server.");
            util.pingServer();
        };

        TimeoutHandlerTask.prototype.start = function(silentStart) {
            if (!!this.started) return;

            if (silentStart === undefined) {
                silentStart = false;
            }

            var that = this;
            // ping server to touch session(s) before starting timeout manager.
            this.ping();
            if (silentStart === false) {
                var onUserActivity = function() {
                    sas.log.trace(
                        "sas.hc.ui.core.TimeoutHandlerTask#onUserActivity",
                        "User activity detected.",
                        arguments
                    );
                    sas.eventBus.publish(sas.timeoutManager.events.USER_ACTIVE);
                };

                // register listener for user activity
                if (!!window.addEventListener) {
                    // list of events borrowed from jQuery.sap.act. Can't use it as sap decided to fire the activity handler
                    // only after 10000 milliseconds of inactivity which is not helping suppress session timeout dialog
                    // when user action continue to happen.
                    var aEvents = [
                        "resize",
                        "orientationchange",
                        "mousedown",
                        "mouseup", //"mouseout", "mouseover",
                        "touchstart",
                        "touchmove",
                        "touchend",
                        "touchcancel",
                        "paste",
                        "cut",
                        "keydown",
                        "keyup",
                        "DOMMouseScroll",
                        "mousewheel"
                    ];
                    for (var i = 0; i < aEvents.length; i++) {
                        window.addEventListener(aEvents[i], onUserActivity, true);
                    }
                    // For now not listen for mutation observers as we want to listen for user activity.
                }
                var supressAjaxCompleteHandler = false;
                sas.eventBus.subscribe(util.BEFORE_SIGN_OUT, function() {
                    supressAjaxCompleteHandler = true;
                });

                // register ajaxComplete event handler to intercept ajax calls to determine
                // if auth session has expired. Timeout if so. This case arise
                // when say cookies are deleted after signing in to app.... or machine is locked or put to sleep
                // for longer time and application is accessed after bringing it up.
                jQuery(document).ajaxComplete(function(evt, jqXHR, options) {
                    if (supressAjaxCompleteHandler === true) return;
                    try {
                        if (
                            jqXHR &&
                            jqXHR.status >= 300 &&
                            jqXHR.getResponseHeader("X-Unauthorized-Reason") === "Authentication"
                        ) {
                            that.logoff();
                        }
                    } catch (secError) {
                        // might be thrown if the response is from cross domain endpoint.
                        // ignore.
                        sas.log.trace(secError);
                    }
                });
                that._hasFocus = true;
                // every time window receives focus, check for timeout
                window.addEventListener("focus", function(){
                    sas.log.trace(
                        "sas.hc.ui.core.TimeoutHandlerTask",
                        "#start",
                        "Received Focus"
                    );
                    that._hasFocus = true;
                    that.checkForTimeout();
                    if(that.altTimeoutActivityInterval) {
                        sas.log.trace('sas.hc.ui.core.TimeoutHandlerTask',"#focus handler","restart altTimeout checks.");
                        that.altTimeoutCheckForPing();
                    }
                }, false);
                window.addEventListener("blur", function(){
                    that._hasFocus = false;
                    sas.log.trace(
                        "sas.hc.ui.core.TimeoutHandlerTask",
                        "#start",
                        "Lost Focus"
                    );
                }, false);
                that.recordUserActivity();
                this.started = true;
            }
            this.resume();

            //alt timeout fix for HTMLCOMMONS-12412
            if(this.altTimeoutActivityInterval) {
                sas.log.trace('sas.hc.ui.core.TimeoutHandlerTask',"#start","start altTimeout checks.");
                this.altTimeoutCheckForPing();
            }

            sas.log.trace(
                "sas.hc.ui.core.TimeoutHandlerTask",
                "#start",
                "Timeout manager started."
            );
        };

        TimeoutHandlerTask.prototype.checkForTimeout = function() {
            if (this.isPaused() && !this._keepAlive) {
                // session dialog is displayed.
                return;
            }
            if (this.lastActTime) {
                // get current time
                var curTime = Math.floor(Date.now() / 1000); // in seconds
                var timeLapsed = curTime - this.lastActTime;
                sas.log.trace(
                    "sas.hc.ui.core.TimeoutHandlerTask",
                    "#checkForTimeout",
                    "time lapsed:" + timeLapsed + ",interval:" + this.actualInterval
                );
                if (timeLapsed >= this.inactiveInterval) {
                    // if time lapsed is greater than max inactive interval then no point in showing timeout dialog.
                    // show timeout page.
                    this.timeout();
                } else if (timeLapsed >= this.actualInterval) {
                    this.sessionTimedOut();
                } else {
                    if (typeof this.altTimeoutActivityInterval === 'undefined'){
                        // user is active. ping the server and continue
                        this.ping();
                    }
                    this.resume(timeLapsed);
                }
            } else {
                this.sessionTimedOut();
            }
        };
        TimeoutHandlerTask.prototype.sessionTimedOut = function() {
            sas.log.trace(
                "sas.hc.ui.core.TimeoutHandlerTask",
                "#sessionTimedOut",
                "session timed out"
            );
            var that = this;
            if (this._keepAlive) {
                this.ping();
                return;
            }
            // Don't show dialog if window isn't focused as the counter could be suppressed 
            // in background tab causing the timeout page never to be displayed.
            // If window doesn't have focus just timeout.
            if (this.warningDuration > 0 && this.appReady === true && this._hasFocus === true) {
                this.pause();
                CommonsDialogDisplayHelper.getSessionTimeoutConfirmation(this.warningDuration).then(
                    function(userAction) {
                        if (userAction === true) {
                            that.resume();
                            that.ping();
                            sas.eventBus.publish(sas.timeoutManager.events.USER_ACTIVE);
                            if(this.altTimeoutActivityInterval) {
                                sas.log.trace('sas.hc.ui.core.TimeoutHandlerTask',"#sessionTimedOut","restart altTimeout checks.");
                                this.altTimeoutCheckForPing();
                            }
                        } else if (false === userAction) {
                            that.logoff();
                        } else {
                            that.timeout();
                        }
                    }
                );
            } else {
                that.timeout();
            }
        };

        /**
         * If timeout manager has started then STOP user activity monitoring and KEEP publishing sas.timeoutManager.events.KEEP_ALIVE event on sas eventbus periodically
         * to help keep user session alive.
         */
        TimeoutHandlerTask.prototype.quiet = function() {
            if (!this.started) return;
            sas.log.trace("sas.hc.ui.core.TimeoutHandlerTask", "#quiet", "Timeout manager quite.");
            this.pause();
            if (typeof this.keeAliveTimerRef === "undefined") {
                this.keeAliveTimerRef = setInterval(function() {
                    sas.eventBus.publish(sas.timeoutManager.events.KEEP_ALIVE);
                }, this.actualInterval * 1000 /* convert into mili sec*/);
            }

            //alt timeout fix for HTMLCOMMONS-12412.pause altTimeout as well
            if(this.altTimeoutActivityInterval) {
                sas.log.trace('sas.hc.ui.core.TimeoutHandlerTask',"#quiet","clear altTimeout.");
                clearTimeout(this.altTimeoutRef);
                this.altTimeoutRef = undefined;
            }
        };

        /**
         * If timeout manager has started then STOP firing sas.timeoutManager.events.KEEP_ALIVE event on sas eventbus and RESUME user activity monitoring.
         */
        TimeoutHandlerTask.prototype.awake = function() {
            if (!this.started) return;
            sas.log.trace("sas.hc.ui.core.TimeoutHandlerTask", "#awake", "Timeout manager wokeup.");
            clearInterval(this.keeAliveTimerRef);
            this.keeAliveTimerRef = undefined;
            this.resume();

            if(this.altTimeoutActivityInterval) {
                sas.log.trace('sas.hc.ui.core.TimeoutHandlerTask',"#awake","restart altTimeout checks.");
                this.altTimeoutCheckForPing();
            }

            this.recordUserActivity();
        };

        /**
         * If timeout manager has started then suppress session timeout and KEEP user session ALIVE.
         * @param {boolean} bVal True to keep user session alive. False otherwise.
         */
        TimeoutHandlerTask.prototype.keepAlive = function(bVal) {
            if (!this.started) return;
            sas.log.trace("sas.hc.ui.core.TimeoutHandlerTask", "#keepAlive", "session keep alive.");
            if (bVal === false) {
                this._keepAlive = false;
            } else {
                this._keepAlive = true;
            }
        };

        return TimeoutHandlerTask;
    }
);
