sap.ui.define([
    "sap/ui/core/Core",
    "sas/hc/ui/core/Core",
    'jquery.sap.global', 
    'sap/ui/base/Object',
    'jquery.sap.act',
    'jquery.sap.script'
], function(sapCore, Core, jQuery, BaseObject/* , jQuerySap1, jQuerySap */) {
    "use strict";

    function lazyInstanceof(o, sModule) {
        var FNClass = sap.ui.require(sModule);
        return typeof FNClass === 'function' && (o instanceof FNClass);
    }


    /**
     * This handler does frequent polling of registered DOM objects checking for size, position or visibility changes
     * and fires events accordingly.
     * 
     * It is expensive by nature and a single instance of it is running.  It is currently created by the first
     * use of MashupOverlay but could be put in a more core place if such functionality is ever needed outside
     * of mashups.  It borrows heavily from sap ResizeHandler code as its foundation.
     *
     * @namespace
     * @alias sas.hc.m.mashup.MashupOverlayPositioningHandler.MashupOverlayPositioningHandler
     * @extends sap.ui.base.Object
     * @author SAP SE
     * @version 904001.11.16.20251118090100_f0htmcm94p
     * @public
     */

    var MashupOverlayPositioningHandler = BaseObject.extend("sas.hc.m.mashup.MashupOverlayPositioningHandler.MashupOverlayPositioningHandler", /** @lends sas.hc.m.mashup.MashupOverlayPositioningHandler.MashupOverlayPositioningHandler.prototype */ {

        constructor : function(oCore) {
            BaseObject.apply(this);

            this.registeredListeners = [];
            this.bRegistered = false;

            this.iIdCounter = 0;

            this.fDestroyHandler = this.destroy.bind(this);

            jQuery(window).bind("unload", this.fDestroyHandler);

            jQuery.sap.act.attachActivate(initListener, this);
        },


    });

    function clearListener(){
        if (this.bRegistered) {
            this.bRegistered = false;
            sap.ui.getCore().detachIntervalTimer(this.performChecks, this);
        }
    }

    function initListener(){
        if (!this.bRegistered && this.registeredListeners.length > 0) {
            this.bRegistered = true;
            sap.ui.getCore().attachIntervalTimer(this.performChecks, this);
        }
    }

    /**
     * Destroy method of the Resize Handler.
     * It unregisters the event handlers.
     *
     * @param {jQuery.Event} oEvent the event that initiated the destruction of the MashupOverlayPositioningHandler
     * @private
     */
    MashupOverlayPositioningHandler.prototype.destroy = function(oEvent) {
        jQuery.sap.act.detachActivate(initListener, this);
        jQuery(window).unbind("unload", this.fDestroyHandler);
        this.registeredListeners = [];
        clearListener.call(this);
    };

    /**
     * Attaches listener to resize event.
     *
     * @param {Element|sap.ui.core.Control} oRef the DOM reference or a control
     * @param {function} fHandler the event handler function
     * @return {string} Registration-ID for later detaching.
     * @private
     */
    MashupOverlayPositioningHandler.prototype.attachListener = function(oRef, fHandler){
        var isOpenUIControl = lazyInstanceof(oRef, 'sap/ui/core/Control'),
            oDom = isOpenUIControl ? oRef.getDomRef() : oRef,
            iWidth = oDom ? oDom.offsetWidth : 0,
            iHeight = oDom ? oDom.offsetHeight : 0,
            sId = "rs-" + Date.now() + "-" + this.iIdCounter++,
            dbg;

        if (isOpenUIControl) {
            dbg = ("Control " + oRef.getId());
        } else if (oRef.id) {
            dbg = oRef.id;
        } else {
            dbg = String(oRef);
        }

        this.registeredListeners.push({sId: sId, oDomRef: isOpenUIControl ? null : oRef, oControl: isOpenUIControl ? oRef : null, fHandler: fHandler, iWidth: iWidth, iHeight: iHeight, dbg: dbg});


        initListener.call(this);

        return sId;
    };

    /**
     * Detaches listener from resize event.
     *
     * @param {string} Registration-ID returned from attachListener
     * @private
     */
    MashupOverlayPositioningHandler.prototype.detachListener = function(sId){
        var registeredListeners = this.registeredListeners;
        for ( var i = 0; i < registeredListeners.length; i++ ) {
            if (registeredListeners[i].sId === sId) {
                registeredListeners.splice(i, 1);
                break;
            }
        }

        // if list is empty now, stop interval
        if (registeredListeners.length === 0) {
            clearListener.call(this);
        }
    };

    MashupOverlayPositioningHandler.prototype.performChecks = function() {

        this.registeredListeners.forEach($.proxy(function(registeredListener){
            if (registeredListener) {
                var bCtrl = !!registeredListener.oControl,
                    oDomRef = bCtrl ? registeredListener.oControl.getDomRef() : registeredListener.oDomRef;

                            
                // Check all things related to having to resize, reposition, hide mashup overlay
                this.checkSizes(registeredListener, oDomRef, bCtrl);
                this.checkPositions(registeredListener, oDomRef, bCtrl);
                this.checkVisibility(registeredListener, oDomRef, bCtrl);
//				this.checkTestEvent(registeredListener,oDomRef,bCtrl);
            }
        },this));

        if (MashupOverlayPositioningHandler._keepActive !== true && MashupOverlayPositioningHandler._keepActive !== false) {
            //initialize default
            MashupOverlayPositioningHandler._keepActive = false;
        }

        if (!jQuery.sap.act.isActive() && !MashupOverlayPositioningHandler._keepActive) {
            clearListener.call(this);
        }
    };

    MashupOverlayPositioningHandler.prototype.checkTestEvent = function(registeredListener, oDomRef, bCtrl) {

        var oEvent = jQuery.Event("intervalTestEvent");
        oEvent.target = oDomRef;
        oEvent.currentTarget = oDomRef;
        oEvent.control = bCtrl ? registeredListener.oControl : null;
    
        registeredListener.fHandler(oEvent);
    };

    /**
     * Check sizes of resize elements.
     * @private
     */
    MashupOverlayPositioningHandler.prototype.checkVisibility = function(registeredListener, oDomRef, bCtrl) {

        var oldVisibleState=Boolean(registeredListener.visibleState);
        var newVisibleState=oldVisibleState;
        // Does it exist in DOM
        newVisibleState=Boolean(oDomRef && jQuery.contains(document.documentElement, oDomRef));

        // Ok so it is in DOM check all the other ways it could potentially be made not visible
        if(newVisibleState===true)
        {
            // OpenUI Control visible property
            if(bCtrl){
                newVisibleState=registeredListener.oControl.getVisible();
            }
            
            // CSS display: none
            // CSS visibility: hidden
            // css opacity: 0           
            
            // Just checking display for none doesn't really cut it since the mashupControl itself could be block but if anything up the chain is none then
            // it isn't actually visible to user and thus mashup should be hidden.
            // window.getComputedStyle doesn't seem to do it also.

            // The best I've found thus far is jquery is(":hidden") which seems to handle the basics anyway.   Not sure how expensive it is but we only take hit if mashup
            // is on the screen.   Could use only the jquery :hidden but it considers it visible if it is taking up space on the screen (even if you can't see it) so make it
            // a combination of :hidden and more direct mechanisms.
            if(oDomRef.style && ( 
                $(oDomRef).is(":hidden") || 
                oDomRef.style.display==="none" ||
                oDomRef.style.visibility==="hidden" || 
                oDomRef.style.opacity==="0"))
            {
                newVisibleState=false;
            }

        }
        if(newVisibleState !== oldVisibleState) {

            registeredListener.visibleState=newVisibleState;
            var oEvent = jQuery.Event("visibilityChange");
            oEvent.target = oDomRef;
            oEvent.currentTarget = oDomRef;
            oEvent.visibleState = newVisibleState;
            oEvent.oldVisibleState = oldVisibleState;
            oEvent.control = bCtrl ? registeredListener.oControl : null;

            registeredListener.fHandler(oEvent);

        }
    };


    /**
     * Check sizes of resize elements.
     * @private
     */
    MashupOverlayPositioningHandler.prototype.checkPositions = function(registeredListener, oDomRef, bCtrl) {
        if ( oDomRef && jQuery.contains(document.documentElement, oDomRef)) { //check that domref is still active
            var offset=$(oDomRef).offset();

            var iOldTop = registeredListener.iTop,
                iOldLeft = registeredListener.iLeft,
                iNewTop = offset.top,
                iNewLeft = offset.left;

            if (iOldTop !== iNewTop || iOldLeft !== iNewLeft) {
                registeredListener.iTop = iNewTop;
                registeredListener.iLeft = iNewLeft;

                var oEvent = jQuery.Event("positionChange");
                oEvent.target = oDomRef;
                oEvent.currentTarget = oDomRef;
                oEvent.offset = {top: iNewTop, left: iNewLeft};
                oEvent.oldOffset = {top: iOldTop, left: iOldLeft};
                oEvent.control = bCtrl ? registeredListener.oControl : null;

                registeredListener.fHandler(oEvent);
            }
        }
    };

    /**
     * Check sizes of resize elements.
     * @private
     */
    MashupOverlayPositioningHandler.prototype.checkSizes = function(registeredListener, oDomRef, bCtrl) {
        if ( oDomRef && jQuery.contains(document.documentElement, oDomRef)) { //check that domref is still active

            var iOldWidth = registeredListener.iWidth,
                iOldHeight = registeredListener.iHeight,
                iNewWidth = oDomRef.offsetWidth,
                iNewHeight = oDomRef.offsetHeight;

            if (iOldWidth !== iNewWidth || iOldHeight !== iNewHeight) {
                registeredListener.iWidth = iNewWidth;
                registeredListener.iHeight = iNewHeight;

                var oEvent = jQuery.Event("sizeChange");
                oEvent.target = oDomRef;
                oEvent.currentTarget = oDomRef;
                oEvent.size = {width: iNewWidth, height: iNewHeight};
                oEvent.oldSize = {width: iOldWidth, height: iOldHeight};
                oEvent.control = bCtrl ? registeredListener.oControl : null;


                registeredListener.fHandler(oEvent);
            }
        }
    };


    /**
     * Registers the given event handler for resize events on the given DOM element or control.
     *
     * <b>Note:</b> This function must not be used before the UI5 framework is initialized.
     * Please use the {@link sap.ui.core.Core#attachInit init event} of UI5 if you are not sure whether this is the case.
     *
     * The resize handler periodically checks the dimensions of the registered reference. Whenever it detects changes, an event is fired.
     * Be careful when changing dimensions within the event handler which might cause another resize event and so on.
     *
     * The available parameters of the resize event are:
     * <ul>
     * <li><code>oEvent.target</code>: The DOM element of which the dimensions were checked</li>
     * <li><code>oEvent.size.width</code>: The current width of the DOM element in pixels</li>
     * <li><code>oEvent.size.height</code>: The current height of the DOM element in pixels</li>
     * <li><code>oEvent.oldSize.width</code>: The previous width of the DOM element in pixels</li>
     * <li><code>oEvent.oldSize.height</code>: The previous height of the DOM element in pixels</li>
     * <li><code>oEvent.control</code>: The control which was given during registration of the event handler (if present)</li>
     * </ul>
     *
     * @param {DOMRef|sap.ui.core.Control} oRef The control or the DOM reference for which the given event handler should be registered (beside the window)
     * @param {function} fHandler
     *             The event handler which should be called whenever the size of the given reference is changed.
     *             The event object is passed as first argument to the event handler. See the description of this function for more details about the available parameters of this event.
     * @return {string}
     *             A registration ID which can be used for deregistering the event handler, see {@link sas.hc.m.mashup.MashupOverlayPositioningHandler.MashupOverlayPositioningHandler.deregister}.
     *             If the UI5 framework is not yet initialized <code>null</code> is returned.
     * @public
     */
    MashupOverlayPositioningHandler.register = function(oRef, fHandler) {
        if (!sas.applicationSwitcher || !sas.applicationSwitcher.mashupOverlayPositioningHandler) {
            return null;
        }
        return sas.applicationSwitcher.mashupOverlayPositioningHandler.attachListener(oRef, fHandler);
    };

    /**
     * Deregisters a previously registered handler for resize events with the given registration ID.
     *
     * @param {string} sId
     *            The registration ID of the handler to deregister. The ID was provided by function {@link sas.hc.m.mashup.MashupOverlayPositioningHandler.MashupOverlayPositioningHandler.register}
     *            when the handler was registered.
     * @public
     */
    MashupOverlayPositioningHandler.deregister = function(sId) {
        if (!sas.applicationSwitcher || !sas.applicationSwitcher.mashupOverlayPositioningHandler) {
            return;
        }
        sas.applicationSwitcher.mashupOverlayPositioningHandler.detachListener(sId);
    };

    /**
     * Deregisters all registered handler for resize events for the given control.
     *
     * @param {string} sControlId The Id of the control.
     * @private
     */
    MashupOverlayPositioningHandler.deregisterAllForControl = function(sControlId) {
        if (!sas.applicationSwitcher || !sas.applicationSwitcher.mashupOverlayPositioningHandler) {
            return;
        }

        sas.applicationSwitcher.mashupOverlayPositioningHandler.registeredListeners.filter(function(registeredListener){
            return registeredListener && registeredListener.oControl && registeredListener.oControl.getId() === sControlId;
        }).forEach(function(registeredListener) {
            MashupOverlayPositioningHandler.deregister(registeredListener.sId);
        });
    };


    return MashupOverlayPositioningHandler;

});
