/*!
 * UI development toolkit for HTML5 (OpenUI5)
 * (c) Copyright 2009-2025 SAP SE or an SAP affiliate company.
 * Licensed under the Apache License, Version 2.0 - see LICENSE.txt.
 */

// Provides control sap.ui.unified.MenuItem.
sap.ui.define(['sap/ui/core/IconPool', './MenuItemBase', './library'],
	function(IconPool, MenuItemBase, library) {
	"use strict";



	/**
	 * Constructor for a new MenuItem element.
	 *
	 * @param {string} [sId] Id for the new control, generated automatically if no id is given
	 * @param {object} [mSettings] Initial settings for the new control
	 *
	 * @class
	 * Standard item to be used inside a menu. A menu item represents an action which can be selected by the user in the menu or
	 * it can provide a submenu to organize the actions hierarchically.
	 * @extends sap.ui.unified.MenuItemBase
	 *
	 * @author SAP SE
	 * @version 904001.7.0.20251118090100_f0htmcm94p
	 * @since 1.21.0
	 *
	 * @constructor
	 * @public
	 * @alias sap.ui.unified.MenuItem
	 * @ui5-metamodel This control/element will also be described in the UI5 (legacy) design time meta model
	 */
	var MenuItem = MenuItemBase.extend("sap.ui.unified.MenuItem", /** @lends sap.ui.unified.MenuItem.prototype */ { metadata : {

		library : "sap.ui.unified",
		properties : {

			/**
			 * Defines the text which should be displayed on the item.
			 */
			text : {type : "string", group : "Appearance", defaultValue : ''},

			/**
			 * Defines the icon of the {@link sap.ui.core.IconPool sap.ui.core.IconPool} or an image which should be displayed on the item.
			 */
			icon : {type : "sap.ui.core.URI", group : "Appearance", defaultValue : ''}
		},

		/** S1174070 This keeps track of when the title (i.e. tooltip) has been set implicitly when the text is truncated. */
		/* @since SAS local modification */
		_titleSetInternally: false

	}});

	IconPool.getIconInfo("", ""); //Ensure Icon Font is loaded

	// HTMLCOMMONS-5606  add bMultipleHeaders param
	MenuItem.prototype.render = function(oRenderManager, oItem, oMenu, oInfo, bMultipleHeaders){
		var rm = oRenderManager;
		var oSubMenu = oItem.getSubmenu();
		rm.write("<li ");

		var sClass = "sapUiMnuItm";
		if (oInfo.iItemNo == 1) {
			sClass += " sapUiMnuItmFirst";
		} else if (oInfo.iItemNo == oInfo.iTotalItems) {
			sClass += " sapUiMnuItmLast";
		}
		if (!oMenu.checkEnabled(oItem)) {
			sClass += " sapUiMnuItmDsbl";
		}
		if (oItem.getStartsSection()) {
			sClass += " sapUiMnuItmSepBefore";
		}

		// HTMLCOMMONS-5606 local sas edit
		if (oMenu.isItemHeader(oItem)) {
			// Add a different style for a single header.
			if (bMultipleHeaders === false) {
				sClass += " sasUiMnuItmSingleHeader";
			} else {
				sClass += " sasUiMnuItmMultipleHeaders";
			}
		}
		// HTMLCOMMONS-6745 local sas edit
		if (oMenu.isItemOverflowFirst(oItem)) {
			sClass += " sasToggleButtonOverflowFirst";
		}

		if (oMenu.isItemOverflowLast(oItem)) {
			sClass += " sasToggleButtonOverflowLast";
		}
		// HTMLCOMMONS-6745 local sas edit

		rm.writeAttribute("class", sClass);
		if (oItem.getTooltip_AsString()) {
			rm.writeAttributeEscaped("title", oItem.getTooltip_AsString());
		}

		if (this.writeAdditionalItemAttributes && typeof this.writeAdditionalItemAttributes === "function") {
			this.writeAdditionalItemAttributes(oRenderManager, oItem, oMenu, oInfo);
		}

		rm.writeElementData(oItem);

		// ARIA
		if (oInfo.bAccessible) {
			rm.writeAccessibilityState(oItem, {
				role: "menuitem",
				disabled: !oMenu.checkEnabled(oItem),
				posinset: oInfo.iItemNo,
				setsize: oInfo.iTotalItems

				// @since SAS local edit. S1316106
				// labelledby: {value: /*oMenu.getId() + "-label " + */this.getId() + "-txt " + this.getId() + "-scuttxt", append: true}
			});
			if (oSubMenu) {
				rm.writeAttribute("aria-haspopup", true);
				rm.writeAttribute("aria-owns", oSubMenu.getId());
			}
		}

		// Left border
		rm.write("><div class=\"sapUiMnuItmL\"></div>");

		// Check column.  HTMLCOMMONS-5608
		// @since SAS modification.
		this.renderCheckColumn(oRenderManager, oItem, oMenu, oInfo);


		// icon/check column

		// If this is a header, ignore the icon column
		// Check column.  HTMLCOMMONS-5606
		// @since SAS modification.
		if (oMenu.isItemHeader(oItem) === false) {
			rm.write("<div class=\"sapUiMnuItmIco\">");
			if (oItem.getIcon()) {
				rm.writeIcon(oItem.getIcon(), null, {title: null});
			}
			rm.write("</div>");
		}

		// Text column
		rm.write("<div id=\"" + this.getId() + "-txt\" class=\"sapUiMnuItmTxt\">");
		rm.writeEscaped(oItem.getText());
		rm.write("</div>");

		// Delete column.  HTMLCOMMONS-6713
		// @since SAS modification.
		this.renderDeleteColumn(oRenderManager, oItem, oMenu, oInfo);


		// Shortcut column
		rm.write("<div id=\"" + this.getId() + "-scuttxt\" class=\"sapUiMnuItmSCut\"></div>");

		// Submenu column
		this.renderSubMenuColumn(oRenderManager, oItem, oMenu, oInfo);

		// Right border
		rm.write("<div class=\"sapUiMnuItmR\"></div>");

		rm.write("</li>");
	};


	/**
	 * @protected
	 */
	MenuItem.prototype.renderSubMenuColumn = function(rm, oItem, oMenu, oInfo) {
		var oSubMenu = oItem.getSubmenu();
		rm.write("<div class=\"sapUiMnuItmSbMnu\">");
		if (oSubMenu) {
			rm.write("<div class=\"sapUiIconMirrorInRTL\"></div>");
		}
		rm.write("</div>");
	};

	/**
	 * HTMLCOMMONS-5608
	 *
	 * @method
	 * @since SAS modification.
	 */
	MenuItem.prototype.renderCheckColumn = function(rm, oItem, oMenu, oInfo) {
		// Do nothing.
	};
	/**
	 * HTMLCOMMONS-5608
	 *
	 * @method
	 * @since SAS modification.
	 */
	MenuItem.prototype.renderDeleteColumn = function(rm, oItem, oMenu, oInfo) {
		// Do nothing.
	};


	MenuItem.prototype.hover = function(bHovered, oMenu){
		this.$().toggleClass("sapUiMnuItmHov", bHovered);
	};

	/**
	 * SAS Edit  S1331995  Remove the handlers added by the Menu
	 *
	 * Called from Element's destroy() method.
	 * @private
	 */
	MenuItem.prototype.exit = function(){

		var oDomRef = this.getDomRef();
		if ( oDomRef ) {
			// This returns the menu items.
			jQuery('.sapUiMnuItmTxt', oDomRef).unbind('mouseenter');
		}
	};

	return MenuItem;

});
